/* ResidueDecoder.c */
/* 2008/11/06       */

#include "StdAfx.h"

#include "ResidueDecoder.h"

BOOL ResidueDecoder_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	ResidueSetup_t*   setup)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	INT32 i;
	UINT8 cascade[64];

	setup->Type = QV_BitDecoder_GetBits(d, 16);
	if (setup->Type > 2) {
		THROW0(ctx, "ResidueDecoder_Setup", "InvalidType")
	}

	setup->Begin    = QV_BitDecoder_GetBits(d, 24);
	setup->End      = QV_BitDecoder_GetBits(d, 24);
	setup->PartSize = QV_BitDecoder_GetBits(d, 24) + 1;

	setup->Classifications = QV_BitDecoder_GetBits(d, 6) + 1;
	setup->ClassBook       = QV_BitDecoder_GetBits(d, 8);

	for (i = 0; i < setup->Classifications; i++) {
		UINT32 high = 0;
		UINT32 low  = QV_BitDecoder_GetBits(d, 3);
		if (QV_BitDecoder_GetBits(d, 1) != 0) {
			high = QV_BitDecoder_GetBits(d, 5);
		}
		cascade[i] = (high << 3) | low;
	}

	setup->ResidueBooks = (INT16(*)[8])QV_MemoryPool_Allocate(pool, sizeof(INT16[8]) * setup->Classifications);
	if (setup->ResidueBooks == NULL) {
		THROW0(ctx, "ResidueDecoder_Setup", "OutOfMemory")
	}

	for (i = 0; i < setup->Classifications; i++) {
		INT32 j;
		for (j = 0; j < 8; j++) {
			if ((cascade[i] & (1 << j)) != 0) {
				setup->ResidueBooks[i][j] = QV_BitDecoder_GetBits(d, 8);
			} else {
				setup->ResidueBooks[i][j] = -1;
			}
		}
	}

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL ResidueDecoder_Decode_1M(
	DecoderContext_t*       ctx,
	BitDecoder_t*           d,
	const ResidueSetup_t*   setup,
	const HuffmanDecoder_t* huff,
	FLOAT**                 z)
{
	BOOL result = FALSE;

	const HuffmanDecoder_t* c_dec = huff + setup->ClassBook;

	INT32 cpc = c_dec->Dimensions;

	INT32 ntr = setup->End - setup->Begin;
	INT32 ptr = ntr / setup->PartSize;

	UINT8 cfs[64*2];

	INT32 i;
	INT32 pass;

	for (pass = 0; pass < 8; pass++) {
		INT32 partition = 0;

		INT32 offset = setup->Begin;

		while (partition < ptr) {
			if (pass == 0) {
				INT32 val = QV_HuffmanDecoder_Lookup(c_dec, d);
				if (val < 0) {
					THROW0(ctx, "ResidueDecoder_Decode_1M", "InvalidClassCode")
				}

				if (d->Count < 0) {
					goto SkipEnd;
				}

				for (i = 0; i < cpc; i++) {
					INT32 temp = val % setup->Classifications;
					cfs[partition + cpc - i - 1] = temp;
					val /= setup->Classifications;
				}
			}

			for (i = 0; i < cpc && partition < ptr; i++) {
				INT32 cf = cfs[partition];
				INT32 cb = setup->ResidueBooks[cf][pass];

				if (cb >= 0) {
					const HuffmanDecoder_t* dec = huff + cb;
					INT32 step = setup->PartSize / dec->Dimensions;
					INT32 vofs = offset;

					INT32 j;
					for (j = 0; j < step; j++) {
						INT32 k;
						const FLOAT* r;

						INT32 val = QV_HuffmanDecoder_Lookup(dec, d);
						if (val < 0) {
							THROW0(ctx, "ResidueDecoder_Decode_1M", "InvalidResidueCode")
						}

						if (d->Count < 0) {
							goto SkipEnd;
						}

						r = dec->Lookup + val * dec->Dimensions;

						for (k = 0; k < dec->Dimensions; k += 1) {
							z[0][vofs] += r[k + 0];
							vofs += 1;
						}
					}
				}

				partition += 1;

				offset += setup->PartSize;
			}
		}
	}

SkipEnd:

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL ResidueDecoder_Decode_2S(
	DecoderContext_t*       ctx,
	BitDecoder_t*           d,
	const ResidueSetup_t*   setup,
	const HuffmanDecoder_t* huff,
	FLOAT**                 z)
{
	BOOL result = FALSE;

	const HuffmanDecoder_t* c_dec = huff + setup->ClassBook;

	INT32 cpc = c_dec->Dimensions;

	INT32 ntr = setup->End - setup->Begin;
	INT32 ptr = ntr / setup->PartSize;

	UINT8 cfs[64*2];

	INT32 i;
	INT32 pass;

	for (pass = 0; pass < 8; pass++) {
		INT32 partition = 0;

		INT32 offset = setup->Begin;

		while (partition < ptr) {
			if (pass == 0) {
				INT32 val = QV_HuffmanDecoder_Lookup(c_dec, d);
				if (val < 0) {
					THROW0(ctx, "ResidueDecoder_Decode_2S", "InvalidClassCode")
				}

				if (d->Count < 0) {
					goto SkipEnd;
				}

				for (i = 0; i < cpc; i++) {
					INT32 temp = val % setup->Classifications;
					cfs[partition + cpc - i - 1] = temp;
					val /= setup->Classifications;
				}
			}

			for (i = 0; i < cpc && partition < ptr; i++) {
				INT32 cf = cfs[partition];
				INT32 cb = setup->ResidueBooks[cf][pass];

				if (cb >= 0) {
					const HuffmanDecoder_t* dec = huff + cb;
					INT32 step = setup->PartSize / dec->Dimensions;
					INT32 vofs = offset / 2;

					INT32 j;
					if ((dec->Dimensions & 1) == 0) {
						for (j = 0; j < step; j++) {
							INT32 k;
							const FLOAT* r;

							INT32 val = QV_HuffmanDecoder_Lookup(dec, d);
							if (val < 0) {
								THROW0(ctx, "ResidueDecoder_Decode_2S", "InvalidResidueCode")
							}

							if (d->Count < 0) {
								goto SkipEnd;
							}

							r = dec->Lookup + val * dec->Dimensions;

							for (k = 0; k < dec->Dimensions; k += 2) {
								z[0][vofs] += r[k + 0];
								z[1][vofs] += r[k + 1];
								vofs += 1;
							}
						}

					} else {
						INT32 ch = 0;

						for (j = 0; j < step; j++) {
							INT32 k;
							const FLOAT* r;

							INT32 val = QV_HuffmanDecoder_Lookup(dec, d);
							if (val < 0) {
								THROW0(ctx, "ResidueDecoder_Decode_2S", "InvalidResidueCode")
							}

							if (d->Count < 0) {
								goto SkipEnd;
							}

							r = dec->Lookup + val * dec->Dimensions;

							for (k = 0; k < dec->Dimensions; k += 1) {
								z[ch++][vofs] += r[k + 0];
								if (ch == 2) {
									ch = 0;
									vofs += 1;
								}
							}
						}
					}
				}

				partition += 1;

				offset += setup->PartSize;
			}
		}
	}

SkipEnd:

	result = TRUE;

ErrorEnd:

	return result;
}

