// Utils.h
// 2008/11/27

#pragma once

namespace QAX {

/* */

extern UINT32 g_OFFSET_ALIGN;
extern UINT32 g_PAGE_SIZE;

/* */

// AutoPtr<T>
template<typename T>
class AutoPtr {

	T* m_p;

public:

	AutoPtr(T* p) : m_p(p)
	{
	}

	~AutoPtr()
	{
		delete m_p;
	}

	operator T*()
	{
		return m_p;
	}

	T* operator->()
	{
		return m_p;
	}

	T* Detach()
	{
		T* p = m_p;
		m_p = 0;
		return p;
	}

}; // AutoPtr<T>

/* */

// CriticalSection
class CriticalSection {

	CRITICAL_SECTION m_csec;

public:

	CriticalSection()
	{
		InitializeCriticalSection(&m_csec);
	}

	~CriticalSection()
	{
		DeleteCriticalSection(&m_csec);
	}

	void Lock()
	{
		EnterCriticalSection(&m_csec);
	}

	void Unlock()
	{
		LeaveCriticalSection(&m_csec);
	}

}; // CriticalSection

// Lock
class Lock {

	CriticalSection* m_p;

public:

	Lock(CriticalSection* p) : m_p(p)
	{
		m_p->Lock();
	}

	~Lock()
	{
		m_p->Unlock();
	}

}; // Lock

/* */

extern const BYTE VN_SIZE[0x100];

INT32 DecodeVNumber(
	const BYTE* pos,
	const BYTE* end,
	UINT64*     value)
{
	const BYTE* p = pos;

	if (p >= end) {
		FormatError::Throw("DecodeVNumber.OverRun");
	}

	BYTE by = p[0];
	if (by == 0xff) {
		FormatError::Throw("DecodeVNumber.InvalidValue");
	}

	INT32 sz = VN_SIZE[by];
	if (p + sz > end) {
		FormatError::Throw("DecodeVNumber.OverRun");
	}

	INT32  s = 0;
	UINT64 v = 0;

	if (sz < 7) {
		s = 8 - sz;
		v = by & ((1 << s) - 1);
	}

	for (INT32 i = 1; i < sz; i++, s += 8) {
		UINT64 b = p[i];
		v |= (b << s);
	}

	*value = v;

	return sz;
}

INT32 DecodeVNumber(
	const BYTE* pos,
	const BYTE* end,
	UINT32*     value)
{
	UINT64 v;
	INT32 sz = DecodeVNumber(pos, end, &v);

	if (v >= 0x100000000UL) {
		FormatError::Throw("DecodeVNumber.OverFlow");
	}

	*value = UINT32(v);

	return sz;
}

/* */

// Entry
struct Entry {

	UINT32 Offset;
	UINT32 Size;

}; // Entry

typedef std::vector<Entry> Index;

void DecodeIndex(
	const BYTE* pos,
	const BYTE* end,
	Index&      index)
{
	const BYTE* p = pos;

	UINT32 count;
	p += DecodeVNumber(p, end, &count);

	index.resize(count);

	Entry* e = &(index[0]);

	UINT32 offset = 0;
	for (UINT32 i = 0; i < count; i++, e++) {
		UINT32 size;
		p += DecodeVNumber(p, end, &size);

		e->Offset = offset;
		e->Size   = size;

		offset += size;
	}
}

/* */

// Table
class Table {

	QFileMapping* m_Mapping;

	const BYTE* m_pBase;
	const BYTE* m_pChunk;

	Index m_Index;

	UINT32 m_Size;

public:

	Table(QFileMapping* m) :
		m_Mapping(m),
		m_pBase(0),
		m_pChunk(0),
		m_Size(0)
	{
	}

	~Table()
	{
		if (m_pBase != 0) {
			m_Mapping->UnmapView(m_pBase);
		}
	}

	void Setup(
		UINT64 pos,
		UINT32 size,
		UINT32 index)
	{
		UINT64 bpos = (pos / g_OFFSET_ALIGN) * g_OFFSET_ALIGN;
		UINT32 diff = UINT32(pos - bpos);

		m_pBase = static_cast<const BYTE*>(
			m_Mapping->MapView(
				bpos,
				diff + size));

		m_pChunk = m_pBase + diff;

		const BYTE* idx = m_pChunk + index;

		DecodeIndex(
			idx,
			m_pChunk + size,
			m_Index);

		m_Size = index;
	}

	INT32 GetCount()
	{
		return INT32(m_Index.size());
	}

	const void* GetEntry(
		INT32   index,
		UINT32* size)
	{
		if (index >= INT32(m_Index.size())) {
			FormatError::Throw("Table.OutOfIndex");
		}

		const Entry& e = m_Index[index];

		if (e.Offset >= m_Size) {
			FormatError::Throw("Table.OutOfIndex");
		}

		if (e.Offset + e.Size > m_Size) {
			FormatError::Throw("Table.OutOfIndex");
		}

		if (size != 0) {
			*size = e.Size;
		}

		return m_pChunk + e.Offset;
	}

}; // Table

/* */

// StringTable
class StringTable : public Table {

public:

	StringTable(QFileMapping* m) : Table(m)
	{
	}

	std::string GetString(INT32 index)
	{
		UINT32 size = 0;
		const char* pch = static_cast<const char*>(
			GetEntry(index, &size));
		return std::string(pch, size);
	}

}; // StringTable

/* */

typedef std::map<std::string, std::string> ItemProperty;

INT32 DecodeProperty(
	StringTable*  strings,
	const BYTE*   pos,
	const BYTE*   end,
	ItemProperty* props)
{
	const BYTE* p = pos;

	UINT32 count;
	p += DecodeVNumber(p, end, &count);

	for (UINT32 i = 0; i < count; i++) {
		UINT32 type;
		p += DecodeVNumber(p, end, &type);

		UINT32 value;
		p += DecodeVNumber(p, end, &value);

		props->insert(
			ItemProperty::value_type(
				strings->GetString(type),
				strings->GetString(value)));
	}

	return INT32(p - pos);
}

/* */

// CRC32
class CRC32 {

	static UINT32 CRC_TABLE[0x100];

	UINT32 m_crc;

public:

	static void MakeTable()
	{
		for (UINT32 i = 0; i < 0x100; i++) {
			UINT32 r = i;
			for (UINT32 j = 0; j < 8; j++) {
				if ((r & 1) != 0) {
					r = (r >> 1) ^ 0xedb88320;
				} else {
					r >>= 1;
				}
			}
			CRC_TABLE[i] = r;
		}
	}

public:

	CRC32() : m_crc(0xffffffff)
	{
	}

	UINT32 Generate(
		const VOID* buffer,
		SIZE_T      size)
	{
		UINT32 crc = m_crc;

		const UINT8* p = static_cast<const UINT8*>(buffer);
		const UINT8* end = p + size;

		while (p < end) {
			crc = CRC_TABLE[(crc & 0xff) ^ *(p++)] ^ (crc >> 8);
		}

		m_crc = crc;

		return m_crc ^ 0xffffffff;
	}

}; // CRC32

/* */

} // namespace QAX

