// QPin.h
// 2009/08/21

#pragma once

// QPinHost
class QPinHost {

public:

	virtual ATL::CComCriticalSection& FilterCSec() = 0;

	virtual bool IsRunning() = 0;

}; /* QPinHost */

// QPinAudio
class ATL_NO_VTABLE QPinAudio :
	public ATL::CComObjectRootEx<ATL::CComMultiThreadModel>,
	public IPin {

	QPinHost* m_pHost;

	IBaseFilter* m_pFilter; /* weak */

	ATL::CStringW m_Id;

	INT32       m_cMediaTypes;
	QMediaType* m_MediaTypes;

	ATL::CComPtr<IPin> m_ConnectedTo;

	ATL::CComPtr<IMemInputPin>  m_MemInputPin;
	ATL::CComPtr<IMemAllocator> m_MemAllocator;

	QMediaType m_ConnectionMediaType;

	WAVEFORMATEX m_WaveHeader;

public:

	DECLARE_NOT_AGGREGATABLE(QPinAudio)

	DECLARE_PROTECT_FINAL_CONSTRUCT()

	BEGIN_COM_MAP(QPinAudio)
		COM_INTERFACE_ENTRY(IPin)
	END_COM_MAP()

public:

	QPinAudio() : m_pHost(0), m_pFilter(0), m_cMediaTypes(0), m_MediaTypes(0)
	{
		memset(&m_WaveHeader, 0, sizeof(m_WaveHeader));
	}

	~QPinAudio()
	{
		delete[] m_MediaTypes;
	}

	void Init(
		QPinHost*    host,
		IBaseFilter* p,
		LPCWSTR      id)
	{
		m_pHost   = host;
		m_pFilter = p;
		m_Id      = id;
	}

	/* */

	HRESULT FinalConstruct()
	{
		ATLTRACE("QPinAudio::FinalConstruct()\n");

		return S_OK;
	}

	void FinalRelease()
	{
		ATLTRACE("QPinAudio::FinalRelease()\n");
	}

	/* */

	QMediaType* MediaTypes()
	{
		return m_MediaTypes;
	}

	void SetupMediaTypes(INT32 c)
	{
		m_cMediaTypes = c;
		m_MediaTypes  = new QMediaType[c];
	}

	/* */

	IMemInputPin* GetMemInputPin()
	{
		return m_MemInputPin;
	}

	IMemAllocator* GetMemAllocator()
	{
		return m_MemAllocator;
	}

	/* */

	void SetupAudioFormat(
		INT32 channels,
		INT32 samplingRate)
	{
		SetupMediaTypes(1);

		QMediaType* mt = MediaTypes();

		mt[0].SetAudio(channels, samplingRate);
	}

	const WAVEFORMATEX* GetWaveHeader()
	{
		return &m_WaveHeader;
	}

	/* */

	HRESULT SetupAllocator(
		IMemAllocator*       alloc,
		const AM_MEDIA_TYPE* target)
	{
		HRESULT hRslt = CheckAccept(target);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		ALLOCATOR_PROPERTIES rprop = { 0 };
		ALLOCATOR_PROPERTIES aprop = { 0 };

		rprop.cBuffers = 20;
		rprop.cbBuffer = 2 * m_WaveHeader.nChannels * m_WaveHeader.nSamplesPerSec / 10;
		rprop.cbAlign  = 1;
		rprop.cbPrefix = 0;

		hRslt = alloc->SetProperties(
			&rprop,
			&aprop);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	HRESULT CheckAccept(
		const AM_MEDIA_TYPE* pmt)
	{
		if (pmt->majortype != MEDIATYPE_Audio) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		if (pmt->formattype == FORMAT_WaveFormatEx  &&
			pmt->cbFormat   >= sizeof(WAVEFORMATEX) &&
			pmt->pbFormat   != 0) {

			const WAVEFORMATEX* w = (const WAVEFORMATEX*)(pmt->pbFormat);

			ATLTRACE("Wave: %d ch %d Hz\n", w->nChannels, w->nSamplesPerSec);

			memcpy(&m_WaveHeader, w, sizeof(m_WaveHeader));

		} else {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		return S_OK;
	}

	/* IPin */

	STDMETHOD(Connect)(
		IPin*                pReceivePin,
		const AM_MEDIA_TYPE* pmt)
	{
		ATLTRACE("QPinAudio::Connect()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (pReceivePin == 0) {
			return E_INVALIDARG;
		}

		if (m_ConnectedTo != 0) {
			return VFW_E_ALREADY_CONNECTED;
		}

		if (m_pHost->IsRunning()) {
			return VFW_E_NOT_STOPPED;
		}

		const AM_MEDIA_TYPE* target = 0;

		for (INT32 i = 0; i < m_cMediaTypes; i++) {
			const AM_MEDIA_TYPE* mt = m_MediaTypes + i;

			if (pmt != 0) {
				if (mt->majortype != pmt->majortype ||
					mt->subtype   != pmt->subtype) {
					continue;
				}

				if (pmt->formattype != GUID_NULL) {
					if (mt->formattype != pmt->formattype) {
						continue;
					}

					if (mt->cbFormat != pmt->cbFormat) {
						continue;
					}

					if (memcmp(mt->pbFormat, pmt->pbFormat, mt->cbFormat) != 0) {
						continue;
					}
				}
			}

			HRESULT hRslt = pReceivePin->QueryAccept(mt);
			if (hRslt == S_OK) {
				target = mt;
				break;
			}
		}

		if (target == 0) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		HRESULT hRslt = pReceivePin->ReceiveConnection(
			this,
			target);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		m_ConnectionMediaType.SafeRelease();
		hRslt = pReceivePin->ConnectionMediaType(&m_ConnectionMediaType);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		ATL::CComQIPtr<IMemInputPin> mem = pReceivePin;
		if (mem == 0) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		ATL::CComPtr<IMemAllocator> alloc;
		hRslt = mem->GetAllocator(&alloc);
		if (FAILED(hRslt)) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		hRslt = SetupAllocator(
			alloc,
			target);
		if (FAILED(hRslt)) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		hRslt = mem->NotifyAllocator(alloc, FALSE);
		if (FAILED(hRslt)) {
			return VFW_E_TYPE_NOT_ACCEPTED;
		}

		m_MemInputPin  = mem;
		m_MemAllocator = alloc;

		m_ConnectedTo = pReceivePin;

		return S_OK;
	}

	STDMETHOD(ReceiveConnection)(
		IPin*                /* pConnector */,
		const AM_MEDIA_TYPE* /* pmt */)
	{
		ATLTRACE("QPinAudio::ReceiveConnection()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(Disconnect)(void)
	{
		ATLTRACE("QPinAudio::Disconnect()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (m_pHost->IsRunning()) {
			return VFW_E_NOT_STOPPED;
		}

		if (m_ConnectedTo == 0) {
			return S_FALSE;
		}

		m_MemAllocator.Release();
		m_MemInputPin. Release();

		m_ConnectedTo.Release();

		return S_OK;
	}

	STDMETHOD(ConnectedTo)(
		IPin** pPin)
	{
		ATLTRACE("QPinAudio::ConnectedTo()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (m_ConnectedTo == 0) {
			if (pPin != 0) {
				*pPin = 0;
			}

			return VFW_E_NOT_CONNECTED;
		}

		if (pPin == 0) {
			return E_INVALIDARG;
		}

		*pPin = m_ConnectedTo;

		if (*pPin != 0) {
			(*pPin)->AddRef();
		}

		return S_OK;
	}

	STDMETHOD(ConnectionMediaType)(
		AM_MEDIA_TYPE* pmt)
	{
		ATLTRACE("QPinAudio::ConnectionMediaType()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (pmt == 0) {
			return E_INVALIDARG;
		}

		if (!QMediaType::Copy(pmt, &m_ConnectionMediaType)) {
			return E_OUTOFMEMORY;
		}

		return S_OK;
	}

	/* */

	STDMETHOD(QueryPinInfo)(
		PIN_INFO* pInfo)
	{
		ATLTRACE("QPinAudio::QueryPinInfo()\n");

		if (pInfo == 0) {
			return E_INVALIDARG;
		}

		pInfo->pFilter = m_pFilter;
		pInfo->dir     = PINDIR_OUTPUT;

		SIZE_T len = m_Id.GetLength();
		if (len > 127) {
			len = 127;
		}
		wcsncpy_s(pInfo->achName, 128, m_Id, len);

		if (m_pFilter != 0) {
			m_pFilter->AddRef();
		}

		return S_OK;
	}

	STDMETHOD(QueryDirection)(
		PIN_DIRECTION* pPinDir)
	{
		ATLTRACE("QPinAudio::QueryDirection()\n");

		if (pPinDir == 0) {
			return E_INVALIDARG;
		}

		*pPinDir = PINDIR_OUTPUT;

		return S_OK;
	}

	STDMETHOD(QueryId)(
		LPWSTR* Id)
	{
		ATLTRACE("QPinAudio::QueryId()\n");

		if (Id == 0) {
			return E_INVALIDARG;
		}

		LPCWSTR id = m_Id;
		SIZE_T  cb = (m_Id.GetLength() + 1) * sizeof(WCHAR);

		*Id = static_cast<LPWSTR>(CoTaskMemAlloc(cb));
		if (*Id == 0) {
			return E_OUTOFMEMORY;
		}

		memcpy(*Id, id, cb);

		return S_OK;
	}

	/* */

	STDMETHOD(QueryAccept)(
		const AM_MEDIA_TYPE* pmt)
	{
		ATLTRACE("QPinAudio::QueryAccept()\n");

		ATL::CComCritSecLock<ATL::CComCriticalSection> lock(m_pHost->FilterCSec());

		if (pmt == 0) {
			return E_INVALIDARG;
		}

		HRESULT hRslt = CheckAccept(pmt);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		m_ConnectionMediaType.Assign(pmt);

		return S_OK;
	}

	STDMETHOD(EnumMediaTypes)(
		IEnumMediaTypes** ppEnum)
	{
		ATLTRACE("QPinAudio::EnumMediaTypes()\n");

		if (ppEnum == 0) {
			return E_INVALIDARG;
		}

		HRESULT hRslt = QEnumMediaTypes::Create(m_MediaTypes, m_MediaTypes + m_cMediaTypes, this, ppEnum);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	/* */

	STDMETHOD(QueryInternalConnections)(
		IPin** apPin,
		ULONG* nPin)
	{
		ATLTRACE("QPinAudio::QueryInternalConnections()\n");

		if (apPin != 0 && nPin != 0) {
			if (*nPin > 0) {
				memset(apPin, 0, *nPin * sizeof(IPin*));
			}
			*nPin = 0;
		}

		return E_NOTIMPL;
	}

	/* */

	STDMETHOD(EndOfStream)(void)
	{
		ATLTRACE("QPinAudio::EndOfStream()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(BeginFlush)(void)
	{
		ATLTRACE("QPinAudio::BeginFlush()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(EndFlush)(void)
	{
		ATLTRACE("QPinAudio::EndFlush()\n");

		return E_NOTIMPL;
	}

	STDMETHOD(NewSegment)( 
		REFERENCE_TIME /* tStart */,
		REFERENCE_TIME /* tStop  */,
		double         /* dRate  */)
	{
		ATLTRACE("QPinAudio::NewSegment()\n");

		return E_NOTIMPL;
	}

	/* */

}; // QPinAudio

