/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;

import net.morilib.db.expr.RelationExpression;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sql.DbSqlLexer;
import net.morilib.db.sql.DbSqlParser;
import net.morilib.db.sqlcs.ddl.SqlAlterTableAdd;
import net.morilib.db.sqlcs.ddl.SqlAlterTableDrop;
import net.morilib.db.sqlcs.ddl.SqlAlterTableModify;
import net.morilib.db.sqlcs.ddl.SqlAlterTableRenameColumn;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlDropTable;
import net.morilib.db.sqlcs.ddl.SqlTruncateTable;
import net.morilib.db.sqlcs.dml.SqlDelete;
import net.morilib.db.sqlcs.dml.SqlExpression;
import net.morilib.db.sqlcs.dml.SqlInsertSelect;
import net.morilib.db.sqlcs.dml.SqlInsertValues;
import net.morilib.db.sqlcs.dml.SqlRelation;
import net.morilib.db.sqlcs.dml.SqlSelect;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import net.morilib.db.sqlcs.dml.SqlUpdate;
import net.morilib.db.sqlcs.dml.SqlWith;

public abstract class SqlEngine {

	public abstract Relation visit(SqlSchema f, SqlRelation t,
			List<Object> h) throws IOException, SQLException;

	public abstract Relation visit(SqlSchema f, SqlSelect t,
			RelationTuple p,
			List<Object> h) throws IOException, SQLException;

	public abstract RelationExpression visit(SqlExpression e,
			List<Object> h) throws SQLException;

	public abstract int visit(SqlSchema f, SqlInsertValues t,
			List<Object> h) throws IOException, SQLException;

	public abstract int visit(SqlSchema f, SqlInsertSelect t,
			List<Object> h) throws IOException, SQLException;

	public abstract int visit(SqlSchema f, SqlUpdate t,
			List<Object> h) throws IOException, SQLException;

	public abstract int visit(SqlSchema f, SqlDelete t,
			List<Object> h) throws IOException, SQLException;

	public abstract Relation visit(SqlSchema f, SqlSetExpression s,
			List<Object> h) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlCreateTable c) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlDropTable c) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlTruncateTable c) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlAlterTableAdd c) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlAlterTableModify c) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlAlterTableDrop c) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlAlterTableRenameColumn c
			) throws IOException, SQLException;

	public abstract Object visit(SqlSchema f,
			SqlWith c) throws IOException, SQLException;

	public Object visit(SqlSchema f, Object o,
			List<Object> h) throws IOException, SQLException {
		if(o instanceof SqlSetExpression) {
			return visit(f, (SqlSetExpression)o, h);
		} else if(o instanceof SqlInsertValues) {
			return visit(f, (SqlInsertValues)o, h);
		} else if(o instanceof SqlInsertSelect) {
			return visit(f, (SqlInsertSelect)o, h);
		} else if(o instanceof SqlUpdate) {
			return visit(f, (SqlUpdate)o, h);
		} else if(o instanceof SqlDelete) {
			return visit(f, (SqlDelete)o, h);
		} else if(o instanceof SqlCreateTable) {
			return visit(f, (SqlCreateTable)o);
		} else if(o instanceof SqlDropTable) {
			return visit(f, (SqlDropTable)o);
		} else if(o instanceof SqlTruncateTable) {
			return visit(f, (SqlTruncateTable)o);
		} else if(o instanceof SqlAlterTableAdd) {
			return visit(f, (SqlAlterTableAdd)o);
		} else if(o instanceof SqlAlterTableModify) {
			return visit(f, (SqlAlterTableModify)o);
		} else if(o instanceof SqlAlterTableDrop) {
			return visit(f, (SqlAlterTableDrop)o);
		} else if(o instanceof SqlAlterTableRenameColumn) {
			return visit(f, (SqlAlterTableRenameColumn)o);
		} else if(o instanceof SqlWith) {
			return visit(f, (SqlWith)o);
		} else {
			throw ErrorBundle.getDefault(10039);
		}
	}

	public Object execute(SqlSchema f, String t,
			List<Object> h) throws IOException, SQLException {
		DbSqlLexer l;

		l = new DbSqlLexer(new StringReader(t));
		return visit(f, new DbSqlParser().parse(l),
				Collections.emptyList());
	}

	public Object execute(SqlSchema f,
			String t) throws IOException, SQLException {
		return execute(f, t, Collections.emptyList());
	}

}
