/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.expr;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;

import net.morilib.db.delay.Delay;
import net.morilib.db.engine.SqlEngine;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationAggregate;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.Relations;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sqlcs.dml.SqlSelect;

public class RelationInSubquery extends RelationExpression {

	private RelationExpression exp;
	private SqlSelect select;
	private String name;

	public RelationInSubquery(RelationExpression e,
			SqlSelect s) throws SQLException {
		if(s.getData().size() != 1) {
			throw ErrorBundle.getDefault(10012);
		}
		select = s;
		name   = s.getData().get(0).getAs();
		exp = e;
	}

	Object execsub(SqlEngine v, SqlSchema f,
			Object o) throws IOException, SQLException {
		RelationCursor i;
		RelationTuple t;
		Relation r;

		r = v.visit(f, select, Relations.NULLTUPLE,
				Collections.emptyList());
		i = r.iterator();
		while(i.hasNext()) {
			t = i.next();
			if(o.equals(t.get(name))) {
				return RelationExpression.TRUE;
			}
		}
		return RelationExpression.FALSE;
	}

	@Override
	public Object eval(final SqlEngine v, final SqlSchema f,
			RelationTuple tuple, RelationAggregate m,
			List<String> group,
			List<Object> h) throws IOException, SQLException {
		final Object o;

		o = exp.eval(v, f, tuple, m, group, h);
		if(o instanceof Delay) {
			return new Delay() {

				@Override
				public Object force() throws SQLException {
					try {
						return execsub(v, f, ((Delay)o).force());
					} catch(IOException e) {
						throw ErrorBundle.getDefault(10037);
					}
				}

				@Override
				public void add(Object... args) throws SQLException {
					((Delay)o).add(args);
				}

			};
		} else {
			return execsub(v, f, o);
		}
	}

	@Override
	public boolean isAggregate() {
		return false;
	}

	@Override
	public Object init(final SqlEngine v,
			final SqlSchema f) throws SQLException {
		final Object o;

		o = exp.init(v, f);
		if(o instanceof Delay) {
			return new Delay() {

				@Override
				public Object force() throws SQLException {
					try {
						return execsub(v, f, ((Delay)o).force());
					} catch(IOException e) {
						throw ErrorBundle.getDefault(10037);
					}
				}

				@Override
				public void add(Object... args) throws SQLException {
					((Delay)o).add(args);
				}

			};
		} else {
			return "";
		}
	}

}
