/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.fichier;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;

public class MemoryFichier implements Fichier {

	private static final int BUFSIZE = 16384;

	private String name;
	private char[] buf;
	private int length = -1;

	public MemoryFichier(String n) {
		name = n;
		buf = new char[BUFSIZE];
	}

	@Override
	public boolean isFile() {
		return length >= 0;
	}

	@Override
	public boolean createNewFile() throws IOException {
		length = 0;
		return true;
	}

	@Override
	public Reader openReader() throws IOException {
		final int[] c = new int[1];

		if(length < 0)  throw new IOException();
		c[0] = 0;
		return new Reader() {

			@Override
			public int read(char[] cbuf, int off,
					int len) throws IOException {
				int l;

				if(c[0] < 0) {
					throw new IOException();
				} else if(c[0] >= length) {
					return -1;
				} else {
					l = c[0] + len >= length ? length - c[0] : len;
					System.arraycopy(buf, c[0], cbuf, off, l);
					c[0] += l;
					return l;
				}
			}

			@Override
			public void close() throws IOException {
				if(c[0] < 0)  throw new IOException();
				c[0] = -2;
			}

		};
	}

	@Override
	public Writer openWriter() throws IOException {
		final boolean[] c = new boolean[1];

		length = 0;
		c[0] = false;
		return new Writer() {

			@Override
			public void write(char[] cbuf, int off,
					int len) throws IOException {
				if(c[0])  throw new IOException();
				if(length + len >= BUFSIZE) {
					throw new IOException();
				}
				System.arraycopy(cbuf, off, buf, length, len);
				length += len;
			}

			@Override
			public void close() throws IOException {
				if(c[0])  throw new IOException();
				c[0] = true;
			}

			@Override
			public void flush() throws IOException {
				// do nothing
			}

		};
	}

	@Override
	public InputStream openInputStream() throws IOException {
		final int[] c = new int[1];

		if(length < 0)  throw new IOException();
		c[0] = 0;
		return new InputStream() {

			@Override
			public int read(byte[] cbuf, int off,
					int len) throws IOException {
				int l;

				if(c[0] < 0) {
					throw new IOException();
				} else if(c[0] >= length) {
					return -1;
				} else {
					l = c[0] + len >= length ? length - c[0] : len;
					System.arraycopy(buf, c[0], cbuf, off, l);
					c[0] += l;
					return l;
				}
			}

			@Override
			public void close() throws IOException {
				if(c[0] < 0)  throw new IOException();
				c[0] = -2;
			}

			@Override
			public int read() throws IOException {
				byte[] b = new byte[1];

				return read(b, 0, 1) > 0 ? b[0] : -1;
			}

		};
	}

	@Override
	public OutputStream openOutputStream() throws IOException {
		final boolean[] c = new boolean[1];

		length = 0;
		c[0] = false;
		return new OutputStream() {

			@Override
			public void write(byte[] cbuf, int off,
					int len) throws IOException {
				if(c[0])  throw new IOException();
				if(length + len >= BUFSIZE) {
					throw new IOException();
				}
				System.arraycopy(cbuf, off, buf, length, len);
				length += len;
			}

			@Override
			public void close() throws IOException {
				if(c[0])  throw new IOException();
				c[0] = true;
			}

			@Override
			public void flush() throws IOException {
				// do nothing
			}

			@Override
			public void write(int c) throws IOException {
				byte[] b = new byte[1];

				b[0] = (byte)c;
				write(b, 0, 1);
			}

		};
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public boolean delete() throws IOException {
		length = -1;
		return true;
	}

	@Override
	public Fichier[] listFichiers() {
		return null;
	}

	@Override
	public boolean renameTo(Fichier f) throws IOException {
		return false;
	}

}
