package net.morilib.db.jdbc;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.NClob;
import java.sql.Ref;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.ParseDate;

public abstract class AbstractBeanResultSet extends UnupdatableResultSet {

	/**
	 * 
	 */
	protected List<String> columns;

	/**
	 * 
	 */
	protected Class<?> classe;

	/**
	 * 
	 * @param c
	 * @param l
	 */
	protected AbstractBeanResultSet(Class<?> c, List<String> l) {
		classe = c;
		columns = new ArrayList<String>(l);
	}

	@Override
	public String getString(int columnIndex) throws SQLException {
		return getObject(columnIndex).toString();
	}

	private BigDecimal tonumber(Object o) throws SQLException {
		if(o instanceof Double || o instanceof Float) {
			return BigDecimal.valueOf(((Double)o).doubleValue());
		} else if(o instanceof Number) {
			return BigDecimal.valueOf(((Number)o).longValue());
		} else {
			try {
				return new BigDecimal(o.toString());
			} catch(NumberFormatException e) {
				try {
					return BigDecimal.valueOf(
							Double.parseDouble(o.toString()));
				} catch(NumberFormatException f) {
					throw ErrorBundle.getDefault(10043);
				}
			}
		}
	}

	@Override
	public ResultSetMetaData getMetaData() throws SQLException {
		return new BeanResultSetMetaData(classe, columns);
	}

	@Override
	public boolean getBoolean(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).signum() != 0;
	}

	@Override
	public byte getByte(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).byteValue();
	}

	@Override
	public short getShort(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).shortValue();
	}

	@Override
	public int getInt(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).intValue();
	}

	@Override
	public long getLong(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).longValue();
	}

	@Override
	public float getFloat(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).floatValue();
	}

	@Override
	public double getDouble(int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex)).doubleValue();
	}

	@Override
	public BigDecimal getBigDecimal(int columnIndex,
			int scale) throws SQLException {
		return tonumber(getObject(columnIndex)).setScale(scale);
	}

	@Override
	public byte[] getBytes(int columnIndex) throws SQLException {
		return getString(columnIndex).getBytes();
	}

	private java.util.Date todate(Object o) throws SQLException {
		java.util.Date d;

		if(o instanceof java.util.Date) {
			d = (java.util.Date)o;
		} else if((d = ParseDate.getDate(o.toString())) == null) {
			throw ErrorBundle.getDefault(10043);
		}
		return d;
	}

	@Override
	public Date getDate(int columnIndex) throws SQLException {
		return new Date(todate(getObject(columnIndex)).getTime());
	}

	@Override
	public Time getTime(int columnIndex) throws SQLException {
		return new Time(todate(getObject(columnIndex)).getTime());
	}

	@Override
	public Timestamp getTimestamp(
			int columnIndex) throws SQLException {
		return new Timestamp(todate(getObject(columnIndex)).getTime());
	}

	@Override
	public InputStream getAsciiStream(
			int columnIndex) throws SQLException {
		return new ByteArrayInputStream(getBytes(columnIndex));
	}

	@Override
	public InputStream getUnicodeStream(
			int columnIndex) throws SQLException {
		return new ByteArrayInputStream(getBytes(columnIndex));
	}

	@Override
	public InputStream getBinaryStream(
			int columnIndex) throws SQLException {
		return new ByteArrayInputStream(getBytes(columnIndex));
	}

	@Override
	public String getString(String columnLabel) throws SQLException {
		return getObject(columnLabel).toString();
	}

	@Override
	public boolean getBoolean(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).signum() != 0;
	}

	@Override
	public byte getByte(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).byteValue();
	}

	@Override
	public short getShort(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).shortValue();
	}

	@Override
	public int getInt(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).intValue();
	}

	@Override
	public long getLong(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).longValue();
	}

	@Override
	public float getFloat(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).floatValue();
	}

	@Override
	public double getDouble(String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel)).doubleValue();
	}

	@Override
	public BigDecimal getBigDecimal(String columnLabel,
			int scale) throws SQLException {
		return tonumber(getObject(columnLabel)).setScale(scale);
	}

	@Override
	public byte[] getBytes(String columnLabel) throws SQLException {
		return getString(columnLabel).getBytes();
	}

	@Override
	public Date getDate(String columnLabel) throws SQLException {
		return new Date(todate(getObject(columnLabel)).getTime());
	}

	@Override
	public Time getTime(String columnLabel) throws SQLException {
		return new Time(todate(getObject(columnLabel)).getTime());
	}

	@Override
	public Timestamp getTimestamp(
			String columnLabel) throws SQLException {
		return new Timestamp(todate(getObject(columnLabel)).getTime());
	}

	@Override
	public InputStream getAsciiStream(
			String columnLabel) throws SQLException {
		return new ByteArrayInputStream(getBytes(columnLabel));
	}

	@Override
	public InputStream getUnicodeStream(
			String columnLabel) throws SQLException {
		return new ByteArrayInputStream(getBytes(columnLabel));
	}

	@Override
	public InputStream getBinaryStream(
			String columnLabel) throws SQLException {
		return new ByteArrayInputStream(getBytes(columnLabel));
	}

	@Override
	public SQLWarning getWarnings() throws SQLException {
		return null;
	}

	@Override
	public void clearWarnings() throws SQLException {
		// ignore it
	}

	@Override
	public String getCursorName() throws SQLException {
		return "";
	}

	@Override
	public Object getObject(int columnIndex) throws SQLException {
		if(columnIndex < 1 || columnIndex > columns.size()) {
			ErrorBundle.getDefault(10045);
		}
		return getObject(columns.get(columnIndex - 1));
	}

	@Override
	public int findColumn(String columnLabel) throws SQLException {
		int c;

		if((c = columns.indexOf(columnLabel)) < 0) {
			ErrorBundle.getDefault(10045);
		}
		return c + 1;
	}

	@Override
	public Reader getCharacterStream(
			int columnIndex) throws SQLException {
		return new StringReader(getObject(columnIndex).toString());
	}

	@Override
	public Reader getCharacterStream(
			String columnLabel) throws SQLException {
		return new StringReader(getObject(columnLabel).toString());
	}

	@Override
	public BigDecimal getBigDecimal(
			int columnIndex) throws SQLException {
		return tonumber(getObject(columnIndex));
	}

	@Override
	public BigDecimal getBigDecimal(
			String columnLabel) throws SQLException {
		return tonumber(getObject(columnLabel));
	}

	@Override
	public Date getDate(int columnIndex,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(getDate(columnIndex));
		return new Date(c.getTimeInMillis());
	}

	@Override
	public Date getDate(String columnLabel,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(getDate(columnLabel));
		return new Date(c.getTimeInMillis());
	}

	@Override
	public Time getTime(int columnIndex,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(getDate(columnIndex));
		return new Time(c.getTimeInMillis());
	}

	@Override
	public Time getTime(String columnLabel,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(getDate(columnLabel));
		return new Time(c.getTimeInMillis());
	}

	@Override
	public Timestamp getTimestamp(int columnIndex, Calendar cal)
			throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(getDate(columnIndex));
		return new Timestamp(c.getTimeInMillis());
	}

	@Override
	public Timestamp getTimestamp(String columnLabel, Calendar cal)
			throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(getDate(columnLabel));
		return new Timestamp(c.getTimeInMillis());
	}

	@Override
	public Object getObject(int columnIndex,
			Map<String, Class<?>> map) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Ref getRef(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Blob getBlob(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Clob getClob(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Array getArray(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Object getObject(String columnLabel,
			Map<String, Class<?>> map) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Ref getRef(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Blob getBlob(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Clob getClob(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Array getArray(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public URL getURL(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public URL getURL(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public RowId getRowId(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public RowId getRowId(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public NClob getNClob(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public NClob getNClob(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public SQLXML getSQLXML(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public SQLXML getSQLXML(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public String getNString(int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public String getNString(String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Reader getNCharacterStream(
			int columnIndex) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public Reader getNCharacterStream(
			String columnLabel) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

}
