/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;
import java.util.Collections;

import net.morilib.db.misc.Rational;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.Relations;
import net.morilib.db.sql.DbSqlLexer;
import net.morilib.db.sql.DbSqlParser;
import net.morilib.db.sqlcs.dml.SqlSelect;
import junit.framework.TestCase;

public class RelationsExpressionTest extends TestCase {

	Object get(String t) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(t));
			return new DbSqlParser().parse(l);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	void eqs(String s, Object p) {
		Object o = get(s);
		DefaultSqlEngine e = new DefaultSqlEngine();
		RelationCursor c;
		RelationTuple t;
		Relation r;

		try {
			r = e.visit(null, (SqlSelect)o, Relations.NULLTUPLE,
					Collections.emptyList());
			c = r.iterator();
			assertTrue(c.hasNext());
			t = c.next();
			assertEquals(p, t.get("A"));
			assertFalse(c.hasNext());
		} catch (IOException e1) {
			throw new RuntimeException(e1);
		} catch (SQLException e1) {
			throw new RuntimeException(e1);
		}
	}

	void nil(String s) {
		Object o = get(s);
		DefaultSqlEngine e = new DefaultSqlEngine();
		RelationCursor c;
		Relation r;

		try {
			r = e.visit(null, (SqlSelect)o, Relations.NULLTUPLE,
					Collections.emptyList());
			c = r.iterator();
			assertFalse(c.hasNext());
		} catch (IOException e1) {
			throw new RuntimeException(e1);
		} catch (SQLException e1) {
			throw new RuntimeException(e1);
		}
	}

	void eqs(String s, long p) {
		eqs(s, Rational.valueOf(p));
	}

	public void testRel0001() {
		eqs("SELECT 1 AS A", 1);
		eqs("SELECT 4 * 2 AS A", 8);
		eqs("SELECT 4 / 2 AS A", 2);
		eqs("SELECT 4 + 2 AS A", 6);
		eqs("SELECT 4 - 2 AS A", 2);
		eqs("SELECT 4 || 3 AS A", "43");
		eqs("SELECT 4 = 2 AS A", 0);
		eqs("SELECT 4 <> 2 AS A", 1);
		eqs("SELECT 4 < 2 AS A", 0);
		eqs("SELECT 4 <= 2 AS A", 0);
		eqs("SELECT 4 > 2 AS A", 1);
		eqs("SELECT 4 >= 2 AS A", 1);
		eqs("SELECT +4 AS A", 4);
		eqs("SELECT -4 AS A", -4);
		eqs("SELECT NOT 4 AS A", 0);
		eqs("SELECT 4 AND 2 AS A", 1);
		eqs("SELECT 4 OR 2 AS A", 1);
	}

	public void testRel0002() {
		eqs("SELECT 'chihaya' LIKE 'chi%' AS A", 1);
		eqs("SELECT 'chihaya' LIKE 'yu%' AS A", 0);
		eqs("SELECT 'chihaya' LIKE '%i%' AS A", 1);
		eqs("SELECT 'chihaya' LIKE '%z%' AS A", 0);
		eqs("SELECT 'chihaya' LIKE 'c______' AS A", 1);
	}

	public void testRel0003() {
		eqs("SELECT 'ami' IN ('haruka', 'ami', 'mami') AS A", 1);
		eqs("SELECT 'chihaya' IN ('haruka', 'ami', 'mami') AS A", 0);
	}

	public void testRel0004() {
		eqs("SELECT 91 BETWEEN 73 AND 93 AS A", 1);
		eqs("SELECT 93 BETWEEN 73 AND 93 AS A", 1);
		eqs("SELECT 94 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 73 BETWEEN 73 AND 93 AS A", 1);
		eqs("SELECT 72 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 'ac' BETWEEN 'ab' AND 'ay' AS A", 1);
		eqs("SELECT 'ab' BETWEEN 'ab' AND 'ay' AS A", 1);
		eqs("SELECT 'aa' BETWEEN 'ab' AND 'ay' AS A", 0);
		eqs("SELECT 'ay' BETWEEN 'ab' AND 'ay' AS A", 1);
		eqs("SELECT 'az' BETWEEN 'ab' AND 'ay' AS A", 0);
	}

	public void testRel0011() {
		eqs("SELECT 3 * +4 AS A", 12);
		eqs("SELECT 3 * -4 AS A", -12);
		eqs("SELECT 3 * 4 * 2 AS A", 24);
		eqs("SELECT 3 * 4 / 2 AS A", 6);
		eqs("SELECT 3 * 4 + 2 AS A", 14);
		eqs("SELECT 3 * 4 - 2 AS A", 10);
		eqs("SELECT 3 * 4 || 3 AS A", "123");
		eqs("SELECT 3 * 4 = 2 AS A", 0);
		eqs("SELECT 3 * 4 <> 2 AS A", 1);
		eqs("SELECT 3 * 4 < 2 AS A", 0);
		eqs("SELECT 3 * 4 <= 2 AS A", 0);
		eqs("SELECT 3 * 4 > 2 AS A", 1);
		eqs("SELECT 3 * 4 >= 2 AS A", 1);
		eqs("SELECT 3 * 4 LIKE 'a%' AS A", 0);
		eqs("SELECT 3 * 91 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 3 * NOT 4 AS A", 0);
		eqs("SELECT 3 * 4 AND 2 AS A", 1);
		eqs("SELECT 3 * 4 OR 2 AS A", 1);

		eqs("SELECT +4 * 3 AS A", 12);
		eqs("SELECT -4 * 3 AS A", -12);
		eqs("SELECT 4 * 2 * 3 AS A", 24);
		eqs("SELECT 4 / 2 * 3 AS A", 6);
		eqs("SELECT 4 + 2 * 3 AS A", 10);
		eqs("SELECT 4 - 2 * 3 AS A", -2);
		eqs("SELECT 4 || 3 * 3 AS A", "49");
		eqs("SELECT 4 = 2 * 3 AS A", 0);
		eqs("SELECT 4 <> 2 * 3 AS A", 1);
		eqs("SELECT 4 < 2 * 3 AS A", 1);
		eqs("SELECT 4 <= 2 * 3 AS A", 1);
		eqs("SELECT 4 > 2 * 3 AS A", 0);
		eqs("SELECT 4 >= 2 * 3 AS A", 0);
//		eqs("SELECT 4 LIKE 'a%' * 3 AS A", 0);
		eqs("SELECT 91 BETWEEN 73 AND 93 * 3 AS A", 1);
		eqs("SELECT NOT 4 * 3 AS A", 0);
		eqs("SELECT 4 AND 2 * 3 AS A", 1);
		eqs("SELECT 4 OR 2 * 3 AS A", 1);
	}

	public void testRel0012() {
		eqs("SELECT 8 / +4 AS A", 2);
		eqs("SELECT 8 / -4 AS A", -2);
		eqs("SELECT 8 / 4 * 2 AS A", 4);
		eqs("SELECT 8 / 4 / 2 AS A", 1);
		eqs("SELECT 8 / 4 + 2 AS A", 4);
		eqs("SELECT 8 / 4 - 2 AS A", 0);
		eqs("SELECT 8 / 4 || 3 AS A", "23");
		eqs("SELECT 8 / 4 = 2 AS A", 1);
		eqs("SELECT 8 / 4 <> 2 AS A", 0);
		eqs("SELECT 8 / 4 < 2 AS A", 0);
		eqs("SELECT 8 / 4 <= 2 AS A", 1);
		eqs("SELECT 8 / 4 > 2 AS A", 0);
		eqs("SELECT 8 / 4 >= 2 AS A", 1);
		eqs("SELECT 8 / 4 LIKE 'a%' AS A", 0);
		eqs("SELECT 8 / 91 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 8 / NOT 0 AS A", 8);
		eqs("SELECT 8 / 4 AND 2 AS A", 1);
		eqs("SELECT 8 / 4 OR 2 AS A", 1);

		eqs("SELECT +4 / 2 AS A", 2);
		eqs("SELECT -4 / 2 AS A", -2);
		eqs("SELECT 4 * 2 / 2 AS A", 4);
		eqs("SELECT 4 / 2 / 2 AS A", 1);
		eqs("SELECT 4 + 2 / 2 AS A", 5);
		eqs("SELECT 4 - 2 / 2 AS A", 3);
		eqs("SELECT 4 || 3 / 2 AS A", "43/2");
		eqs("SELECT 4 = 2 / 2 AS A", 0);
		eqs("SELECT 4 <> 2 / 2 AS A", 1);
		eqs("SELECT 4 < 2 / 2 AS A", 0);
		eqs("SELECT 4 <= 2 / 2 AS A", 0);
		eqs("SELECT 4 > 2 / 2 AS A", 1);
		eqs("SELECT 4 >= 2 / 2 AS A", 1);
//		eqs("SELECT 4 LIKE 'a%' * 2 AS A", 0);
		eqs("SELECT 91 BETWEEN 73 AND 93 / 2 AS A", 0);
		eqs("SELECT NOT 4 / 2 AS A", 0);
		eqs("SELECT 4 AND 2 / 2 AS A", 1);
		eqs("SELECT 4 OR 2 / 2 AS A", 1);
	}

	public void testRel0013() {
		eqs("SELECT 3 + +4 AS A", 7);
		eqs("SELECT 3 + -4 AS A", -1);
		eqs("SELECT 3 + 4 * 2 AS A", 11);
		eqs("SELECT 3 + 4 / 2 AS A", 5);
		eqs("SELECT 3 + 4 + 2 AS A", 9);
		eqs("SELECT 3 + 4 - 2 AS A", 5);
		eqs("SELECT 3 + 4 || 3 AS A", "73");
		eqs("SELECT 3 + 4 = 2 AS A", 0);
		eqs("SELECT 3 + 4 <> 2 AS A", 1);
		eqs("SELECT 3 + 4 < 2 AS A", 0);
		eqs("SELECT 3 + 4 <= 2 AS A", 0);
		eqs("SELECT 3 + 4 > 2 AS A", 1);
		eqs("SELECT 3 + 4 >= 2 AS A", 1);
		eqs("SELECT 3 + 4 LIKE 'a%' AS A", 0);
		eqs("SELECT 3 + 91 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 3 + NOT 4 AS A", 3);
		eqs("SELECT 3 + 4 AND 2 AS A", 1);
		eqs("SELECT 3 + 4 OR 2 AS A", 1);

		eqs("SELECT +4 + 3 AS A", 7);
		eqs("SELECT -4 + 3 AS A", -1);
		eqs("SELECT 4 * 2 + 3 AS A", 11);
		eqs("SELECT 4 / 2 + 3 AS A", 5);
		eqs("SELECT 4 + 2 + 3 AS A", 9);
		eqs("SELECT 4 - 2 + 3 AS A", 5);
		eqs("SELECT 4 || 3 + 3 AS A", "46");
		eqs("SELECT 4 = 2 + 3 AS A", 0);
		eqs("SELECT 4 <> 2 + 3 AS A", 1);
		eqs("SELECT 4 < 2 + 3 AS A", 1);
		eqs("SELECT 4 <= 2 + 3 AS A", 1);
		eqs("SELECT 4 > 2 + 3 AS A", 0);
		eqs("SELECT 4 >= 2 + 3 AS A", 0);
//		eqs("SELECT 4 LIKE 'a%' * 3 AS A", 0);
		eqs("SELECT 91 BETWEEN 73 AND 93 + 3 AS A", 1);
		eqs("SELECT NOT 4 + 3 AS A", 0);
		eqs("SELECT 4 AND 2 + 3 AS A", 1);
		eqs("SELECT 4 OR 2 + 3 AS A", 1);
	}

	public void testRel0014() {
		eqs("SELECT 3 - +4 AS A", -1);
		eqs("SELECT 3 - -4 AS A", 7);
		eqs("SELECT 3 - 4 * 2 AS A", -5);
		eqs("SELECT 3 - 4 / 2 AS A", 1);
		eqs("SELECT 3 - 4 + 2 AS A", 1);
		eqs("SELECT 3 - 4 - 2 AS A", -3);
		eqs("SELECT 3 - 4 || 3 AS A", "-13");
		eqs("SELECT 3 - 4 = 2 AS A", 0);
		eqs("SELECT 3 - 4 <> 2 AS A", 1);
		eqs("SELECT 3 - 4 < 2 AS A", 1);
		eqs("SELECT 3 - 4 <= 2 AS A", 1);
		eqs("SELECT 3 - 4 > 2 AS A", 0);
		eqs("SELECT 3 - 4 >= 2 AS A", 0);
		eqs("SELECT 3 - 4 LIKE 'a%' AS A", 0);
		eqs("SELECT 3 - 91 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 3 - NOT 4 AS A", 3);
		eqs("SELECT 3 - 4 AND 2 AS A", 1);
		eqs("SELECT 3 - 4 OR 2 AS A", 1);

		eqs("SELECT +4 - 3 AS A", 1);
		eqs("SELECT -4 - 3 AS A", -7);
		eqs("SELECT 4 * 2 - 3 AS A", 5);
		eqs("SELECT 4 / 2 - 3 AS A", -1);
		eqs("SELECT 4 + 2 - 3 AS A", 3);
		eqs("SELECT 4 - 2 - 3 AS A", -1);
		eqs("SELECT 4 || 3 - 3 AS A", "40");
		eqs("SELECT 4 = 2 - 3 AS A", 0);
		eqs("SELECT 4 <> 2 - 3 AS A", 1);
		eqs("SELECT 4 < 2 - 3 AS A", 0);
		eqs("SELECT 4 <= 2 - 3 AS A", 0);
		eqs("SELECT 4 > 2 - 3 AS A", 1);
		eqs("SELECT 4 >= 2 - 3 AS A", 1);
//		eqs("SELECT 4 LIKE 'a%' * 3 AS A", 0);
		eqs("SELECT 91 BETWEEN 73 AND 93 - 3 AS A", 0);
		eqs("SELECT NOT 4 - 3 AS A", 0);
		eqs("SELECT 4 AND 2 - 3 AS A", 1);
		eqs("SELECT 4 OR 2 - 3 AS A", 1);
	}

	public void testRel0015() {
		eqs("SELECT 3 || +4 AS A", "34");
		eqs("SELECT 3 || -4 AS A", "3-4");
		eqs("SELECT 3 || 4 * 2 AS A", "38");
		eqs("SELECT 3 || 4 / 2 AS A", "32");
		eqs("SELECT 3 || 4 + 2 AS A", "36");
		eqs("SELECT 3 || 4 - 2 AS A", "32");
		eqs("SELECT 3 || 4 || 3 AS A", "343");
		eqs("SELECT 3 || 4 = 2 AS A", 0);
		eqs("SELECT 3 || 4 <> 2 AS A", 1);
		eqs("SELECT 3 || 4 < 2 AS A", 0);
		eqs("SELECT 3 || 4 <= 2 AS A", 0);
		eqs("SELECT 3 || 4 > 2 AS A", 1);
		eqs("SELECT 3 || 4 >= 2 AS A", 1);
		eqs("SELECT 3 || 4 LIKE 'a%' AS A", 0);
		eqs("SELECT 3 || 91 BETWEEN 73 AND 93 AS A", 0);
		eqs("SELECT 3 || NOT 4 AS A", "30");
		eqs("SELECT 3 || 4 AND 2 AS A", 1);
		eqs("SELECT 3 || 4 OR 2 AS A", 1);

		eqs("SELECT +4 || 3 AS A", "43");
		eqs("SELECT -4 || 3 AS A", "-43");
		eqs("SELECT 4 * 2 || 3 AS A", "83");
		eqs("SELECT 4 / 2 || 3 AS A", "23");
		eqs("SELECT 4 + 2 || 3 AS A", "63");
		eqs("SELECT 4 - 2 || 3 AS A", "23");
		eqs("SELECT 4 || 3 || 3 AS A", "433");
		eqs("SELECT 4 = 2 || 3 AS A", 0);
		eqs("SELECT 4 <> 2 || 3 AS A", 1);
		eqs("SELECT 4 < 2 || 3 AS A", 0);
		eqs("SELECT 4 <= 2 || 3 AS A", 0);
		eqs("SELECT 4 > 2 || 3 AS A", 1);
		eqs("SELECT 4 >= 2 || 3 AS A", 1);
		eqs("SELECT 4 LIKE 'a%' || 3 AS A", 0);
		eqs("SELECT 91 BETWEEN 73 AND 93 || 3 AS A", 1);
		eqs("SELECT NOT 4 || 3 AS A", 0);
		eqs("SELECT 4 AND 2 || 3 AS A", 1);
		eqs("SELECT 4 OR 2 || 3 AS A", 1);
	}

	public void testRel0016() {
		eqs("SELECT 3 AND +4 AS A", 1);
		eqs("SELECT 3 AND -4 AS A", 1);
		eqs("SELECT 3 AND 4 * 2 AS A", 1);
		eqs("SELECT 3 AND 4 / 2 AS A", 1);
		eqs("SELECT 3 AND 4 + 2 AS A", 1);
		eqs("SELECT 3 AND 4 - 2 AS A", 1);
		eqs("SELECT 3 AND 4 || 3 AS A", 1);
		eqs("SELECT 3 AND 4 = 2 AS A", 0);
		eqs("SELECT 3 AND 4 <> 2 AS A", 1);
		eqs("SELECT 3 AND 4 < 2 AS A", 0);
		eqs("SELECT 3 AND 4 <= 2 AS A", 0);
		eqs("SELECT 3 AND 4 > 2 AS A", 1);
		eqs("SELECT 3 AND 4 >= 2 AS A", 1);
		eqs("SELECT 3 AND 4 LIKE 'a%' AS A", 0);
		eqs("SELECT 3 AND 91 BETWEEN 73 AND 93 AS A", 1);
		eqs("SELECT 3 AND NOT 4 AS A", 0);
		eqs("SELECT 3 AND 4 AND 2 AS A", 1);
		eqs("SELECT 3 AND 4 OR 2 AS A", 1);

		eqs("SELECT +4 AND 3 AS A", 1);
		eqs("SELECT -4 AND 3 AS A", 1);
		eqs("SELECT 4 * 2 AND 3 AS A", 1);
		eqs("SELECT 4 / 2 AND 3 AS A", 1);
		eqs("SELECT 4 + 2 AND 3 AS A", 1);
		eqs("SELECT 4 - 2 AND 3 AS A", 1);
		eqs("SELECT 4 || 3 AND 3 AS A", 1);
		eqs("SELECT 4 = 2 AND 3 AS A", 0);
		eqs("SELECT 4 <> 2 AND 3 AS A", 1);
		eqs("SELECT 4 < 2 AND 3 AS A", 0);
		eqs("SELECT 4 <= 2 AND 3 AS A", 0);
		eqs("SELECT 4 > 2 AND 3 AS A", 1);
		eqs("SELECT 4 >= 2 AND 3 AS A", 1);
		eqs("SELECT 4 LIKE 'a%' AND 3 AS A", 0);
		eqs("SELECT 91 BETWEEN 73 AND 93 AND 3 AS A", 1);
		eqs("SELECT NOT 4 AND 3 AS A", 0);
		eqs("SELECT 4 AND 2 AND 3 AS A", 1);
		eqs("SELECT 4 OR 2 AND 3 AS A", 1);
	}

	public void testRel0017() {
		eqs("SELECT 3 OR +4 AS A", 1);
		eqs("SELECT 3 OR -4 AS A", 1);
		eqs("SELECT 3 OR 4 * 2 AS A", 1);
		eqs("SELECT 3 OR 4 / 2 AS A", 1);
		eqs("SELECT 3 OR 4 + 2 AS A", 1);
		eqs("SELECT 3 OR 4 - 2 AS A", 1);
		eqs("SELECT 3 OR 4 || 3 AS A", 1);
		eqs("SELECT 3 OR 4 = 2 AS A", 1);
		eqs("SELECT 3 OR 4 <> 2 AS A", 1);
		eqs("SELECT 3 OR 4 < 2 AS A", 1);
		eqs("SELECT 3 OR 4 <= 2 AS A", 1);
		eqs("SELECT 3 OR 4 > 2 AS A", 1);
		eqs("SELECT 3 OR 4 >= 2 AS A", 1);
		eqs("SELECT 3 OR 4 LIKE 'a%' AS A", 1);
		eqs("SELECT 3 OR 91 BETWEEN 73 AND 93 AS A", 1);
		eqs("SELECT 3 OR NOT 4 AS A", 1);
		eqs("SELECT 3 OR 4 AND 2 AS A", 1);
		eqs("SELECT 3 OR 4 OR 2 AS A", 1);

		eqs("SELECT +4 OR 3 AS A", 1);
		eqs("SELECT -4 OR 3 AS A", 1);
		eqs("SELECT 4 * 2 OR 3 AS A", 1);
		eqs("SELECT 4 / 2 OR 3 AS A", 1);
		eqs("SELECT 4 + 2 OR 3 AS A", 1);
		eqs("SELECT 4 - 2 OR 3 AS A", 1);
		eqs("SELECT 4 || 3 OR 3 AS A", 1);
		eqs("SELECT 4 = 2 OR 3 AS A", 1);
		eqs("SELECT 4 <> 2 OR 3 AS A", 1);
		eqs("SELECT 4 < 2 OR 3 AS A", 1);
		eqs("SELECT 4 <= 2 OR 3 AS A", 1);
		eqs("SELECT 4 > 2 OR 3 AS A", 1);
		eqs("SELECT 4 >= 2 OR 3 AS A", 1);
		eqs("SELECT 4 LIKE 'a%' OR 3 AS A", 1);
		eqs("SELECT 91 BETWEEN 73 AND 93 OR 3 AS A", 1);
		eqs("SELECT NOT 4 OR 3 AS A", 1);
		eqs("SELECT 4 AND 2 OR 3 AS A", 1);
		eqs("SELECT 4 OR 2 OR 3 AS A", 1);
	}

	public void testRel0021() {
		eqs("SELECT 1 AS A WHERE 1", 1);
		nil("SELECT 1 AS A WHERE 0");
	}

	public void testRel0022() {
		eqs("SELECT A FROM (SELECT 2 AS A) B", 2);
		eqs("SELECT A FROM (SELECT 2 AS A) B WHERE A = 2", 2);
		nil("SELECT 1 FROM (SELECT 2 AS A) B WHERE A = 0");
	}

}
