/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.db.engine.SqlEngine;
import net.morilib.db.engine.SqlEngineFactory;
import net.morilib.db.fichier.FabriqueDeFichier;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.Rational;
import net.morilib.db.misc.SqlResponse;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.schema.FileSqlSchema;
import net.morilib.db.schema.HTMLSqlSchema;
import net.morilib.db.schema.MemorySqlSchema;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.schema.XlsxSqlSchema;
import net.morilib.db.sqlcs.ddl.SqlColumnAttribute;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.transact.RelationsTransaction;

public class RePlus {

	/**
	 * 
	 */
	public static final String VERSION = "0.0.2";

	/**
	 * 
	 */
	public static final int MAJOR_VERSION = 0;

	/**
	 * 
	 */
	public static final int MINOR_VERSION = 0;

	private static final String PS1 = "rel>";
	private static final String PS2 = "%3d>";
	private static final String DESC_N = "NAME";
	private static final String DESC_T = "TYPE";
	private static final String DESC_A = "ATTRIBUTE";

	private static final SimpleDateFormat FM1 =
			new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

	private static final Pattern PT1 = Pattern.compile(
			"show\\p{Space}+tables;*",
			Pattern.CASE_INSENSITIVE);
	private static final Pattern PT2 = Pattern.compile(
			"d(esc(ribe)?)?\\p{Space}+([^\\p{Space};]+|\".*\");*",
			Pattern.CASE_INSENSITIVE);
	private static final Pattern PT3 = Pattern.compile(
			"commit;*",
			Pattern.CASE_INSENSITIVE);
	private static final Pattern PT4 = Pattern.compile(
			"rollback;*",
			Pattern.CASE_INSENSITIVE);

	private static void perror(int code, Object... args) {
		System.err.println(
				ErrorBundle.getDefaultMessage(code, args));
	}

	private static void pout(int code, Object... args) {
		System.out.println(
				ErrorBundle.getDefaultMessage(code, args));
	}

	private static boolean isquit(String s) {
		return (s.equalsIgnoreCase("exit") ||
				s.equalsIgnoreCase("quit"));
	}

	private static void strrep(int x, char c) {
		for(int i = 0; i < x; i++)  System.out.print(c);
		System.out.print(' ');
	}

	private static int len(String s) {
		Character.UnicodeBlock b;
		int r = 0;
		char c;

		for(int i = 0; i < s.length(); i++) {
			c = s.charAt(i);
			b = Character.UnicodeBlock.of(c);
			if(b == null) {
				r++;
			} else if(b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY) ||
					b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_FORMS) ||
					b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS) ||
					b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT) ||
					b.equals(Character.UnicodeBlock.CJK_RADICALS_SUPPLEMENT) ||
//					b.equals(Character.UnicodeBlock.CJK_STROKES) ||
					b.equals(Character.UnicodeBlock.CJK_SYMBOLS_AND_PUNCTUATION) ||
					b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS) ||
					b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A) ||
					b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B) ||
					b.equals(Character.UnicodeBlock.KATAKANA) ||
					b.equals(Character.UnicodeBlock.HIRAGANA) ||
					b.equals(Character.UnicodeBlock.HANGUL_SYLLABLES)) {
				r += 2;
			} else {
				r++;
			}
		}
		return r;
	}

	private static void lft(String s, int l) {
		Character.UnicodeBlock b;
		int r = 0;
		char c;

		for(int i = 0; i < s.length() && r < l; i++) {
			c = s.charAt(i);
			b = Character.UnicodeBlock.of(c);
			if(b == null) {
				r++;
			} else if(b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY) ||
					b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_FORMS) ||
					b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS) ||
					b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT) ||
					b.equals(Character.UnicodeBlock.CJK_RADICALS_SUPPLEMENT) ||
//					b.equals(Character.UnicodeBlock.CJK_STROKES) ||
					b.equals(Character.UnicodeBlock.CJK_SYMBOLS_AND_PUNCTUATION) ||
					b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS) ||
					b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A) ||
					b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B) ||
					b.equals(Character.UnicodeBlock.KATAKANA) ||
					b.equals(Character.UnicodeBlock.HIRAGANA) ||
					b.equals(Character.UnicodeBlock.HANGUL_SYLLABLES)) {
				r += 2;
			} else {
				r++;
			}
			System.out.print(c);
		}
		strrep(l - r, ' ');
	}

	private static void _print(
			Object o) throws IOException, SQLException {
		List<RelationTuple> l = new ArrayList<RelationTuple>();
		Map<String, Integer> m, r;
		RelationCursor c;
		RelationTuple t;
		Integer n;
		String v;
		Object q;
		int z;

		if(o instanceof Relation) {
			m = new LinkedHashMap<String, Integer>();
			r = new LinkedHashMap<String, Integer>();

			c = ((Relation)o).iterator();
			while(c.hasNext()) {
				t = c.next();
				for(String s : t.toMap().keySet()) {
					q = t.get(s);
					if(!(q instanceof Rational)) {
						z = 0;
					} else if(r.get(s) == null) {
						z = ((Rational)q).getScale();
					} else {
						z = Math.max(r.get(s),
								((Rational)q).getScale());
					}
					z = z < 4 ? z : 4;
					r.put(s, z);
				}
			}

			c = ((Relation)o).iterator();
			while(c.hasNext()) {
				t = c.next();
				l.add(t);
				for(String s : t.toMap().keySet()) {
					q = t.get(s);
					if(q instanceof Rational) {
						z = r.get(s);
						v = ((Rational)q).toBigDecimal(z).toString();
					} else if(q instanceof java.util.Date) {
						v = FM1.format((java.util.Date)q);
					} else {
						v = q.toString();
					}

					z = len(v);
					if((n = m.get(s)) == null || z > n.intValue()) {
						m.put(s, len(v));
					}
				}
			}

			for(String s : m.keySet()) {
				if(m.get(s) == 0)  m.put(s, 5);
			}

			for(String s : m.keySet()) {
				z = m.get(s);
				lft(s, z);
			}
			System.out.println();

			for(String s : m.keySet()) {
				z = m.get(s);
				strrep(z, '-');
			}
			System.out.println();

			for(RelationTuple p : l) {
				for(String s : m.keySet()) {
					q = p.get(s);
					if(q instanceof Rational) {
						lft(((Rational)q).toBigDecimal(4).toString(),
								m.get(s));
					} else if(q instanceof java.util.Date) {
						lft(FM1.format((java.util.Date)q), m.get(s));
					} else {
						lft(q.toString(), m.get(s));
					}
				}
				System.out.println();
			}
			System.out.println();
		} else if(o instanceof Integer) {
			pout(99903, ((Number)o).longValue());
		} else if(o instanceof SqlResponse) {
			System.out.println(((SqlResponse)o).toString());
		} else {
			System.out.println(o);
		}
	}

	private static void _showtables(Collection<String> c) {
		int m = -1;

		for(String s : c) {
			m = s.length() > m ? s.length() : m;
		}
		System.out.println("NAME");
		strrep(m, '-');
		System.out.println();
		for(String s : c){
			System.out.println(s.toUpperCase());
		}
		System.out.println();
	}

	private static String attrtostr(EnumSet<SqlColumnAttribute> a) {
		StringBuffer b = new StringBuffer();
		String d = "";

		for(SqlColumnAttribute x : a) {
			b.append(d).append(x.toString());
			d = ",";
		}
		return b.toString();
	}

	static void _desc(SqlSchema f,
			String name) throws IOException, SQLException {
		SqlCreateTable c;
		List<SqlColumnDefinition> l;
		int l1, l2, l3, j;

		c  = f.getCreateTable(name);
		l  = c.getColumnDefinitions();
		l1 = DESC_N.length();
		l2 = DESC_T.length();
		l3 = DESC_A.length();
		for(SqlColumnDefinition x : l) {
			j  = len(x.getName().toUpperCase());
			l1 = j > l1 ? j : l1;
			j  = len(x.getType().toString());
			l2 = j > l2 ? j : l2;
			j  = len(attrtostr(x.getAttributes()));
			l3 = j > l3 ? j : l3;
		}

		lft(DESC_N, l1);  System.out.print(' ');
		lft(DESC_T, l2);  System.out.print(' ');
		lft(DESC_A, l3);  System.out.println();
		strrep(l1, '-');  System.out.print(' ');
		strrep(l2, '-');  System.out.print(' ');
		strrep(l3, '-');  System.out.println();

		for(SqlColumnDefinition x : l) {
			lft(x.getName().toUpperCase(), l1);  System.out.print(' ');
			lft(x.getType().toString(), l2);     System.out.print(' ');
			lft(attrtostr(x.getAttributes()), l3);
			System.out.println();
		}
		System.out.println();
	}

	/**
	 * 
	 * @param f
	 * @param args
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static SqlSchema findSchema(File f,
			String... args) throws IOException, SQLException {
		FabriqueDeFichier fb = FabriqueDeFichier.getDefault();
		SqlSchema fs = null;

		if(f.isFile() && f.getName().matches(
				".*\\.[Cc][Ss][Vv]")) {
			fs = MemorySqlSchema.readCSVs(args);
		} else if(f.isFile() && f.getName().matches(
				".*\\.[Hh][Tt][Mm][Ll]?")) {
			fs = new HTMLSqlSchema(fb.newInstance(f));
		} else if(f.isFile() && f.getName().matches(
				".*\\.[Xx][Ll][Ss][Xx]?")) {
			fs = new XlsxSqlSchema(fb.newInstance(f));
		} else if(f.isDirectory()) {
			fs = new FileSqlSchema(fb, fb.newInstance(f));
		}
		return fs;
	}

	private static void putex(SQLException e) {
		e.printStackTrace();
		System.err.format(
				"ERROR REL-%d: %s\n",
				e.getErrorCode(),
				e.getMessage());
	}

	/**
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		StringBuffer b = new StringBuffer();
		final SqlEngine en;
		boolean tf = true;
		BufferedReader rd;
		SqlSchema fs;
		int l, k = 0;
		Matcher m;
		Object o;
		String s;
		Thread t;
		File f;

		for(;; k++) {
			if(args.length <= k) {
				break;
			} else if(args[k].equals("--transaction")) {
				tf = true;
			} else if(args[k].equals("--autocommit")) {
				tf = false;
			} else {
				break;
			}
		}

		if(args.length > k) {
			f = new File(args[k]);
		} else {
			perror(99901, args[k]);
			System.exit(86);
			return;
		}

		try {
			if((fs = findSchema(f, args)) == null) {
				perror(99902, f.toString());
				System.exit(86);
				return;
			}

			if(tf) {
				en = new RelationsTransaction(fs);
			} else {
				en = SqlEngineFactory.getInstance().getEngine(fs);
			}

			Runtime.getRuntime().addShutdownHook(t = new Thread() {

				public void run() {
					try {
						pout(99906);
						en.rollback();
					} catch(IOException e) {
						throw new RuntimeException(e);
					} catch(SQLException e) {
						putex(e);
					}
				}

			});

			rd = new BufferedReader(new InputStreamReader(System.in));
			System.out.print(PS1);  l = 1;
			while((s = rd.readLine()) != null) {
				try {
					if(b.length() == 0 && isquit(s)) {
						Runtime.getRuntime().removeShutdownHook(t);
						en.commit();
						System.exit(0);
					} else if(b.length() == 0 &&
							PT1.matcher(s).matches()) {
						_showtables(fs.getTableNames());
						b = new StringBuffer();  l = 1;
						System.out.print(PS1);
					} else if(b.length() == 0 &&
							(m = PT2.matcher(s)).matches()) {
						s = m.group(3);
						s = s.replaceFirst("^\"", "");
						s = s.replaceFirst("\"$", "");
						_desc(fs, s);
						b = new StringBuffer();  l = 1;
						System.out.print(PS1);
					} else if(b.length() == 0 &&
							PT3.matcher(s).matches()) {
						en.commit();
						pout(99904);
						System.out.print(PS1);
					} else if(b.length() == 0 &&
							PT4.matcher(s).matches()) {
						en.rollback();
						pout(99905);
						System.out.print(PS1);
					} else if(s.length() == 0) {
						b = new StringBuffer();  l = 1;
						System.out.print(PS1);
					} else if(!s.endsWith(";")) {
						b.append(s).append('\n');
						System.out.format(PS2, ++l);
					} else {
						b.append(s).deleteCharAt(b.length() - 1);
						o = en.execute(b.toString());
						_print(o);
						b = new StringBuffer();  l = 1;
						System.out.print(PS1);
					}
				} catch(SQLException e) {
					putex(e);
					b = new StringBuffer();  l = 1;
					System.out.print(PS1);
				}
			}
			System.exit(0);
		} catch(IOException e) {
			throw new RuntimeException(e);
		} catch(SQLException e) {
			throw new RuntimeException(e);
		}
	}

}
