/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.transact;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.morilib.db.fichier.FabriqueDeFichier;
import net.morilib.db.fichier.Fichier;
import net.morilib.db.map.CSVDataMapper;
import net.morilib.db.map.SqlDataMapper;
import net.morilib.db.misc.NullBoolean;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;

public class TransactionSqlSchema implements SqlSchema, RelationsLock {

	private SqlSchema schema;
	private Map<String, Fichier> trans =
			new HashMap<String, Fichier>();

	public TransactionSqlSchema(SqlSchema schema) {
		this.schema = schema;
	}

	void commit() throws IOException, SQLException {
		List<RelationTuple> l = new ArrayList<RelationTuple>();
		SqlDataMapper m = new CSVDataMapper();
		RelationCursor c;
		Relation r;

		for(Map.Entry<String, Fichier> t : trans.entrySet()) {
			r = m.read(t.getKey(), null, getCreateTable(t.getKey()),
					t.getValue().openReader());
			c = r.iterator();
			while(c.hasNext())  l.add(c.next());
			schema.writeRelation(t.getKey(), l);
			t.getValue().delete();
			schema.unlock(t.getKey());
		}
		trans.clear();
	}

	void rollback() throws IOException {
		for(Map.Entry<String, Fichier> t : trans.entrySet()) {
			t.getValue().delete();
			schema.unlock(t.getKey());
		}
		trans.clear();
	}

	@Override
	public NamedRelation readRelation(String t,
			String as) throws IOException, SQLException {
		SqlDataMapper m = new CSVDataMapper();
		Fichier f;

		if((f = trans.get(t)) != null) {
			return m.read(t, as, getCreateTable(t), f.openReader());
		} else {
			return schema.readRelation(t, as);
		}
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> z
			) throws IOException, SQLException {
		SqlCreateTable c = getCreateTable(name);
		SqlDataMapper m = new CSVDataMapper();
		PrintWriter w = null;
		Fichier t = trans.get(name);

		try {
			w = new PrintWriter(new BufferedWriter(t.openWriter()));
			m.write(name, c, w, z);
			w.close();
		} finally {
			if(w != null)  w.close();
		}
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		return schema.getCreateTable(name);
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return schema.isTable(name);
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		schema.putCreateTable(name, table);
	}

	@Override
	public Collection<String> getTableNames(
			) throws IOException, SQLException {
		return schema.getTableNames();
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		schema.truncateTable(name);
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		schema.removeCreateTable(name);
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		schema.alterCreateTable(name, table);
	}

	@Override
	public SqlSchema fork() {
		return schema.fork();
	}

	@Override
	public NullBoolean isReadonly() {
		return schema.isReadonly();
	}

	@Override
	public NullBoolean usesLocalFiles() {
		return schema.usesLocalFiles();
	}

	@Override
	public NullBoolean usesLocalFilePerTable() {
		return schema.usesLocalFilePerTable();
	}

	@Override
	public FabriqueDeFichier fabrique() {
		return schema.fabrique();
	}

	@Override
	public void bindSchema(String name, Relation r) {
		schema.bindSchema(name, r);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isAutoCommit()
	 */
	@Override
	public boolean isAutoCommit() {
		return false;
	}

	@Override
	public boolean isLocked(String name) {
		return schema.isLocked(name) && !trans.containsKey(name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#lock(java.lang.String)
	 */
	@Override
	public void lock(String name) throws IOException {
		Fichier t;

		if(!schema.isLocked(name) && !trans.containsKey(name)) {
			t = fabrique().createTempFile("pre-", ".tmp");
			trans.put(name, t);
			schema.lock(name);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#unlock(java.lang.String)
	 */
	@Override
	public void unlock(String name) throws IOException {
		schema.unlock(name);
	}

}
