/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.relations;

import java.io.IOException;
import java.sql.SQLException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;

public class UnionAllRelation extends AbstractRelation {

	private Relation rela, relb;

	public UnionAllRelation(Relation a,
			Relation b) throws SQLException {
		rela = a;  relb = b;
		if(a.getColumnNames().size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		} else {
			for(int i = 0; i < a.getColumnNames().size(); i++) {
				if(!a.getDefinition(a.getColumnNames().get(i).getName()).isSame(
						b.getDefinition(b.getColumnNames().get(i).getName()))) {
					throw ErrorBundle.getDefault(10002);
				}
			}
		}
	}

	@Override
	public RelationCursor iterator() {
		final RelationCursor a, b;

		a = rela.iterator();  b = relb.iterator();
		return new RelationCursor() {

			@Override
			public boolean hasNext() {
				return b.hasNext();
			}

			@Override
			public RelationTuple next() throws IOException, SQLException {
				final RelationTuple t;

				if(a.hasNext()) {
					return a.next();
				} else if(b.hasNext()) {
					t = b.next();
					return new RelationTuple() {

						@Override
						public Object get(String name) throws SQLException {
							int c;

							c = -1;
							for(int i = 0; i < rela.getColumnNames().size(); i++) {
								if(rela.getColumnNames().get(i).getName().equals(name)) {
									c = i;
								}
							}

							if(c < 0) {
								throw ErrorBundle.getDefault(10009, name);
							}
							return t.get(relb.getColumnNames().get(c).getName());
						}

						@Override
						public RelationTuple copy() throws SQLException {
							return new DefaultRelationTuple(toMap());
						}

						@Override
						public Map<String, Object> toMap() throws SQLException {
							Map<String, Object> m;
							List<SqlColumnDefinition> l;

							l = rela.getColumnNames();
							m = new LinkedHashMap<String, Object>();
							for(int i = 0; i < l.size(); i++) {
								m.put(l.get(i).getName(),
										get(l.get(i).getName()));
							}
							return m;
						}

					};
				} else {
					throw new NoSuchElementException();
				}
			}

		};
	}

	@Override
	public List<SqlColumnDefinition> getColumnNames() {
		return rela.getColumnNames();
	}

	@Override
	public SqlColumnDefinition getDefinition(String name) {
		SqlColumnDefinition d;

		if((d = rela.getDefinition(name)) != null) {
			return d;
		} else {
			return null;
		}
	}

}
