/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.relations;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import net.morilib.db.misc.ErrorBundle;

public class VirtualCrossJoinRelation extends AbstractRelation {

	private Relation rela, relb;
	private transient List<String> cols = null;

	/**
	 * 
	 * @param a
	 * @param b
	 */
	public VirtualCrossJoinRelation(Relation a, Relation b) {
		rela = a;  relb = b;
	}

	public RelationCursor iterator() {
		return new RelationCursor() {

			private RelationCursor xi = rela.iterator();
			private RelationCursor xj = relb.iterator();
			private RelationTuple xv = null, xw;

			public boolean hasNext() {
				return xi.hasNext() || xj.hasNext();
			}

			public RelationTuple next(
					) throws IOException, SQLException {
				if(xv == null)  xv = xi.next();
				if(xj.hasNext()) {
					// do nothing
				} else if(xi.hasNext()) {
					xv = xi.next();
					xj = relb.iterator();
				} else {
					throw new NoSuchElementException();
				}

				xw = xj.next();
				if(!xj.hasNext())  fireScaned(xv);
				return new RelationTuple() {

					public Object get(String name) throws SQLException {
						Object o;

						if(!getColumnNames().contains(name)) {
							throw ErrorBundle.getDefault(10009, name);
						} else if((o = Relations._get(xv, rela, name)) != null) {
							return o;
						} else if((o = Relations._get(xw, relb, name)) != null) {
							return o;
						} else {
							throw ErrorBundle.getDefault(10009, name);
						}
					}

					@Override
					public RelationTuple copy() throws SQLException {
						return new DefaultRelationTuple(toMap());
					}

					@Override
					public Map<String, Object> toMap() throws SQLException {
						List<String> ases = getColumnNames();
						Map<String, Object> m;

						m = new LinkedHashMap<String, Object>();
						for(int i = 0; i < ases.size(); i++) {
							m.put(ases.get(i), get(ases.get(i)));
						}
						return m;
					}

				};
			}

		};
	}

	public List<String> getColumnNames() {
		synchronized(this) {
			if(cols == null) {
				cols = new ArrayList<String>();
				Relations._add(cols, rela);
				Relations._add(cols, relb);
			}
		}
		return cols;
	}

}
