/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Reader;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.morilib.db.map.CSVDataMapper;
import net.morilib.db.map.SqlDataMapper;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.sql.DbSqlLexer;
import net.morilib.db.sql.DbSqlParser;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;

public class FileSqlSchema implements SqlSchema {

	private static final SimpleDateFormat FMT =
			new SimpleDateFormat("'.bk'yyyyMMddHHmmssSSS");

	private File base;

	public FileSqlSchema(File s) {
		base = s;
	}

	protected File getFile(
			String name) throws IOException, SQLException {
		String t = name.toLowerCase();
		File f;

		if(!isTable(name.toLowerCase())) {
			throw ErrorBundle.getDefault(10015, name);
		} else if((f = new File(base, t + ".csv")).isFile()) {
			return f;
		} else {
			f = new File(base, t + ".csv");
			new FileWriter(f).close();;
			return f;
		}
	}

	protected Reader getReader(
			String name) throws IOException, SQLException {
		return new FileReader(new File(
				base, name.toLowerCase()));
	}

	protected Reader getTableReader(
			String name) throws IOException, SQLException {
		return new FileReader(getFile(name));
	}

	protected PrintWriter getWriter(
			String name) throws IOException, SQLException {
		File f = new File(base, name.toLowerCase());

		return new PrintWriter(new BufferedWriter(new FileWriter(f)));
	}

	protected PrintWriter getTableWriter(
			String name) throws IOException, SQLException {
		return new PrintWriter(new BufferedWriter(
				new FileWriter(getFile(name))));
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		Reader rd = getReader(name + ".ddl");
		DbSqlLexer l;
		Object o;

		try {
			l = new DbSqlLexer(rd);
			o = new DbSqlParser().parse(l);
			if(o instanceof SqlCreateTable) {
				return (SqlCreateTable)o;
			} else {
				throw ErrorBundle.getDefault(10041);
			}
		} finally {
			rd.close();
		}
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return new File(base, name + ".ddl").isFile();
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		PrintWriter w = null;

		if(isTable(name)) {
			throw ErrorBundle.getDefault(10016, name);
		} else {
			try {
				w = getWriter(name + ".ddl");
				w.write(table.toString());
			} finally {
				if(w != null)  w.close();
			}
		}
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		PrintWriter w = null;

		if(!isTable(name)) {
			throw ErrorBundle.getDefault(10015, name);
		} else {
			try {
				w = getWriter(name + ".ddl");
				w.write(table.toString());
			} finally {
				if(w != null)  w.close();
			}
		}
	}

	@Override
	public NamedRelation readRelation(String t,
			String as) throws IOException, SQLException {
		File f = getFile(t);
		SqlDataMapper m;

		if(f.getName().endsWith(".csv")) {
			m = new CSVDataMapper();
		} else {
			throw ErrorBundle.getDefault(10042);
		}
		return m.read(t, as, getCreateTable(t), getTableReader(t));
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> z
			) throws IOException, SQLException {
		SqlCreateTable c = getCreateTable(name);
		File f = getFile(name), t = null;
		char[] b = new char[1024];
		PrintWriter w = null;
		SqlDataMapper m;
		Reader r = null;
		int j;

		if(f.getName().endsWith(".csv")) {
			m = new CSVDataMapper();
		} else {
			throw ErrorBundle.getDefault(10042);
		}

		try {
			t = File.createTempFile("pre-", ".tmp");
			w = new PrintWriter(new BufferedWriter(new FileWriter(t)));
			m.write(name, c, w, z);
			w.close();  w = null;

			r = new BufferedReader(new FileReader(t));
			w = getTableWriter(name);
			while((j = r.read(b)) >= 0)  w.write(b, 0, j);
		} finally {
			if(w != null)  w.close();
			if(r != null)  r.close();
			if(t != null)  t.delete();
		}
	}

	@Override
	public Collection<String> getTableNames() {
		List<String> r = new ArrayList<String>();
		File[] l = base.listFiles();
		String s;

		for(File f : l) {
			s = f.getName();
			if(s.endsWith(".ddl")) {
				r.add(s.substring(0, s.length() - 4));
			}
		}
		return r;
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		File f, g;

		if(isTable(name)) {
			f = getFile(name);
			g = new File(base,
					f.getName() + FMT.format(new java.util.Date()));
			f.renameTo(g);
		} else {
			throw ErrorBundle.getDefault(10015, name);
		}
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		File f, g;
		String s;

		if(isTable(name)) {
			s = FMT.format(new java.util.Date());
			f = getFile(name);
			g = new File(base, f.getName() + s);
			f.renameTo(g);

			f = new File(base, name.toLowerCase() + ".ddl");
			g = new File(base, f.getName() + s);
			f.renameTo(g);
		} else {
			throw ErrorBundle.getDefault(10015, name);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#fork()
	 */
	@Override
	public SqlSchema fork() {
		return new FileSqlSchema(base);
	}

}
