/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.SQLException;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.sqlcs.dml.SqlBinaryOperator;

public class DbSqlLexer {

	private static enum S {
		INI, QUO, BAR, SYM, NUM, NM2, DQO, LT, GT, EX
	}

	static final Object END = new Object() {

		public String toString() {
			return "END";
		}

	};

	private Object lookahead;
	private PushbackReader reader;
	private int placeNumber = 1;

	/**
	 * 
	 * @param r
	 * @throws IOException
	 * @throws SQLException
	 */
	public DbSqlLexer(Reader r) throws IOException, SQLException {
		reader = new PushbackReader(r);
		lookahead = lex(reader);
	}

	public boolean eq(Object c) throws IOException, SQLException {
		Object l = get();

		if(l == c) {
			next();
			return true;
		} else {
			return false;
		}
	}

	public void eat(Object c) throws IOException, SQLException {
		Object l = get();

		if(l == c) {
			next();
		} else {
			throw ErrorBundle.getDefault(10020, c.toString(),
					l.toString());
		}
	}

	public boolean eqchar(char c) throws IOException, SQLException {
		Object l = get();

		if(l instanceof Character && ((Character)l).charValue() == c) {
			next();
			return true;
		} else {
			return false;
		}
	}

	public void eatchar(char c) throws IOException, SQLException {
		Object l = get();

		if(l instanceof Character && ((Character)l).charValue() == c) {
			next();
		} else {
			throw ErrorBundle.getDefault(10020, Character.toString(c),
					l.toString());
		}
	}

	public boolean eqsym(String s) throws IOException, SQLException {
		Object l = get();

		if(l instanceof DbSqlSymbol &&
				((DbSqlSymbol)l).toString().equalsIgnoreCase(s)) {
			next();
			return true;
		} else {
			return false;
		}
	}

	public void eatsym(String s) throws IOException, SQLException {
		Object l = get();

		if(l instanceof DbSqlSymbol &&
				((DbSqlSymbol)l).toString().equalsIgnoreCase(s)) {
			next();
		} else {
			throw ErrorBundle.getDefault(10020, s, l.toString());
		}
	}

	public String getsym() throws IOException, SQLException {
		Object l = get();

		if(l instanceof DbSqlSymbol) {
			next();
			return l.toString();
		} else {
			return null;
		}
	}

	public String eatsym() throws IOException, SQLException {
		Object l = get();

		if(l instanceof DbSqlSymbol) {
			next();
			return l.toString();
		} else {
			throw ErrorBundle.getDefault(10020, "symbol",
					l.toString());
		}
	}

	public String getstr() throws IOException, SQLException {
		Object l = get();

		if(l instanceof String) {
			next();
			return l.toString();
		} else {
			return null;
		}
	}

	public Number getnum() throws IOException, SQLException {
		Object l = get();

		if(l instanceof Number) {
			next();
			return (Number)l;
		} else {
			return null;
		}
	}

	public SqlBinaryOperator getrelop(
			) throws IOException, SQLException {
		Object l = get();

		if(l instanceof SqlBinaryOperator) {
			next();
			return (SqlBinaryOperator)l;
		} else {
			return null;
		}
	}

	public boolean isEnd() {
		return lookahead == END;
	}

	private static Object lex(
			PushbackReader r) throws IOException, SQLException {
		StringBuffer b = null;
		S stat = S.INI;
		String s;
		int c;

		while(true) {
			c = r.read();
			switch(stat) {
			case INI:
				if(c < 0) {
					return END;
				} else if(c == '\'') {
					b = new StringBuffer();
					stat = S.QUO;
				} else if(c == '\"') {
					b = new StringBuffer();
					stat = S.DQO;
				} else if(c == '|') {
					stat = S.BAR;
				} else if(Character.isJavaIdentifierStart(c)) {
					b = new StringBuffer().append((char)c);
					stat = S.SYM;
				} else if(c == '<') {
					stat = S.LT;
				} else if(c == '>') {
					stat = S.GT;
				} else if(c == '!') {
					stat = S.EX;
				} else if(c == '=') {
					return SqlBinaryOperator.EQ;
				} else if(c >= '0' && c <= '9') {
					b = new StringBuffer().append((char)c);
					stat = S.NUM;
				} else if(c == '.') {
					b = new StringBuffer().append('0').append('.');
					stat = S.NM2;
				} else if(!Character.isWhitespace(c)) {
					return Character.valueOf((char)c);
				}
				break;
			case QUO:
				if(c < 0) {
					throw ErrorBundle.getDefault(10021);
				} else if(c == '\'') {
					return b.toString();
				} else {
					b.append((char)c);
				}
				break;
			case BAR:
				if(c == '|') {
					return DbSqlReserved.CONCAT;
				} else {
					if(c >= 0)  r.unread((char)c);
					return Character.valueOf('|');
				}
			case SYM:
				if(Character.isJavaIdentifierPart(c) || c == '.') {
					b.append((char)c);
				} else {
					if(c >= 0)  r.unread((char)c);
					s = b.toString().toUpperCase();
					if(s.equals("LIKE")) {
						return SqlBinaryOperator.LIKE;
					} else {
						try {
							return DbSqlReserved.valueOf(s);
						} catch(IllegalArgumentException e) {
							return new DbSqlSymbol(s);
						}
					}
				}
				break;
			case NUM:
				if(c == '.') {
					b.append((char)c);
					stat = S.NM2;
				} else if(c >= '0' && c <= '9') {
					b.append((char)c);
				} else {
					if(c >= 0)  r.unread((char)c);
					return new BigDecimal(b.toString());
				}
				break;
			case NM2:
				if(c >= '0' && c <= '9') {
					b.append((char)c);
				} else {
					if(c >= 0)  r.unread((char)c);
					return new BigDecimal(b.toString());
				}
				break;
			case DQO:
				if(c < 0) {
					throw ErrorBundle.getDefault(10021);
				} else if(c == '\"') {
					return new DbSqlSymbol(b.toString().toUpperCase());
				} else {
					b.append((char)c);
				}
				break;
			case LT:
				if(c == '=') {
					return SqlBinaryOperator.LE;
				} else if(c == '>') {
					return SqlBinaryOperator.NE;
				} else {
					if(c >= 0)  r.unread((char)c);
					return SqlBinaryOperator.LT;
				}
			case GT:
				if(c == '=') {
					return SqlBinaryOperator.GE;
				} else {
					if(c >= 0)  r.unread((char)c);
					return SqlBinaryOperator.GT;
				}
			case EX:
				if(c == '=') {
					return SqlBinaryOperator.NE;
				} else {
					if(c >= 0)  r.unread((char)c);
					return Character.valueOf('!');
				}
			}
		}
	}

	/**
	 * 
	 * @return
	 */
	public Object next() throws IOException, SQLException {
		return (lookahead == END) ? END : (lookahead = lex(reader));
	}

	/**
	 * 
	 * @return
	 */
	public Object get() {
		return lookahead;
	}

	public int nextPlaceNumber() {
		return placeNumber++;
	}

	public int getPlaceNumber() {
		return placeNumber;
	}

}
