/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.io.StringReader;
import java.math.BigDecimal;
import java.sql.SQLException;
import java.util.Collections;

import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.sql.DbSqlLexer;
import net.morilib.db.sql.DbSqlParser;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import junit.framework.TestCase;

public class RelationsTest extends TestCase {

	Object get(String t) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(t));
			return new DbSqlParser().parse(l);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	RelationCursor itr(String s) {
		Object o = get(s);
		DefaultSqlEngine e = new DefaultSqlEngine();
		Relation r;

		try {
			r = (Relation)e.visit(new SqlFileSystemStub(),
					(SqlSetExpression)o,
					Collections.emptyList());
			return r.iterator();
		} catch (IOException e1) {
			throw new RuntimeException(e1);
		} catch (SQLException e1) {
			throw new RuntimeException(e1);
		}
	}

	RelationTuple eqs(RelationCursor r) {
		try {
			assertTrue(r.hasNext());
			return r.next();
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	void eqs(RelationTuple t, String s, Object o) {
		try {
			assertEquals(o, t.get(s));
		} catch (SQLException e) {
			throw new RuntimeException(s, e);
		}
	}

	void eqs(RelationTuple t, String s, long o) {
		eqs(t, s, BigDecimal.valueOf(o));
	}

	void eqd(RelationTuple t, String s, String o) {
		eqs(t, s, new BigDecimal(o));
	}

	public void testRel0041() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT COUNT(B) AS C FROM IDOLS");
		t = eqs(r);  eqs(t, "C", 12);
		assertFalse(r.hasNext());
	}

	public void testRel0001() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqs(t, "B", 76);
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0002() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0003() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS WHERE NAME LIKE '%か%'");
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0004() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B - 72 AS C FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "C", 11);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "C", 11);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "C", 11);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "C", 18);
		assertFalse(r.hasNext());
	}

	public void testRel0005() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT A.NAME, A.B FROM IDOLS A WHERE B >= 83");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");   eqs(t, "A.B", 83);
		t = eqs(r);  eqs(t, "A.NAME", "我那覇響");   eqs(t, "A.B", 83);
		t = eqs(r);  eqs(t, "A.NAME", "島村卯月");   eqs(t, "A.B", 83);
		t = eqs(r);  eqs(t, "A.NAME", "三村かな子");   eqs(t, "A.B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0006() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "天海春香");
		t = eqs(r);  eqs(t, "NAME", "我那覇響");
		t = eqs(r);  eqs(t, "NAME", "島村卯月");
		t = eqs(r);  eqs(t, "NAME", "三村かな子");
		assertFalse(r.hasNext());
	}

	public void testRel0007() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT * FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0011() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT A.NAME FROM IDOLS A JOIN IDOL_HISTORIES B ON A.NAME = B.NAME");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");
		t = eqs(r);  eqs(t, "A.NAME", "高槻やよい");
		t = eqs(r);  eqs(t, "A.NAME", "菊地真");
		t = eqs(r);  eqs(t, "A.NAME", "菊地真");
		assertFalse(r.hasNext());
	}

	public void testRel0021() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT A.NAME, A.B FROM (SELECT NAME, HEIGHT, B FROM IDOLS WHERE B >= 83) A");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");   eqs(t, "A.B", 83);
		t = eqs(r);  eqs(t, "A.NAME", "我那覇響");   eqs(t, "A.B", 83);
		t = eqs(r);  eqs(t, "A.NAME", "島村卯月");   eqs(t, "A.B", 83);
		t = eqs(r);  eqs(t, "A.NAME", "三村かな子");   eqs(t, "A.B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0031() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS ORDER BY B");
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqs(t, "B", 76);
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0032() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B, AGE FROM IDOLS ORDER BY B, AGE");
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqs(t, "B", 76);
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0033() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B, AGE FROM IDOLS ORDER BY B DESC, AGE");
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqs(t, "B", 76);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		assertFalse(r.hasNext());
	}

	public void testRel0042() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT SUM(B) AS C FROM IDOLS");
		t = eqs(r);  eqs(t, "C", 944);
		assertFalse(r.hasNext());
	}

	public void testRel0043() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT AGE, COUNT(AGE) C FROM IDOLS GROUP BY AGE ORDER BY AGE");
		t = eqs(r);  eqs(t, "AGE", 10);   eqs(t, "C", 1);
		t = eqs(r);  eqs(t, "AGE", 13);   eqs(t, "C", 1);
		t = eqs(r);  eqs(t, "AGE", 14);   eqs(t, "C", 1);
		t = eqs(r);  eqs(t, "AGE", 15);   eqs(t, "C", 1);
		t = eqs(r);  eqs(t, "AGE", 16);   eqs(t, "C", 2);
		t = eqs(r);  eqs(t, "AGE", 17);   eqs(t, "C", 4);
		t = eqs(r);  eqs(t, "AGE", 18);   eqs(t, "C", 1);
		t = eqs(r);  eqs(t, "AGE", 21);   eqs(t, "C", 1);
		assertFalse(r.hasNext());
	}

	public void testRel0044() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT COUNT(B) AS C FROM IDOLS WHERE NAME = 'aaa'");
		t = eqs(r);  eqs(t, "C", 0);
		assertFalse(r.hasNext());
	}

	public void testRel0045() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT AGE, COUNT(AGE) C FROM IDOLS GROUP BY AGE" +
				" HAVING C > 1 ORDER BY AGE");
		t = eqs(r);  eqs(t, "AGE", 16);   eqs(t, "C", 2);
		t = eqs(r);  eqs(t, "AGE", 17);   eqs(t, "C", 4);
		assertFalse(r.hasNext());
	}

	public void testRel0046() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT AVG(B) AS C FROM IDOLS");
		t = eqs(r);  eqd(t, "C", "78.6667");
		assertFalse(r.hasNext());
	}

	public void testRel0051() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS WHERE B > " +
				"  (SELECT AVG(B) FROM IDOLS) " +
				" ORDER BY B");
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0052() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B - (SELECT AVG(B) FROM IDOLS) B" +
				" FROM IDOLS ORDER BY B");
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqd(t, "B", "-14.6667");
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqd(t, "B", "-4.6667");
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqd(t, "B", "-3.6667");
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqd(t, "B", "-2.6667");
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqd(t, "B", "-1.6667");
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqd(t, "B", "-1.6667");
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqd(t, "B", "2.3333");
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqd(t, "B", "2.3333");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqd(t, "B", "4.3333");
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqd(t, "B", "4.3333");
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqd(t, "B", "4.3333");
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqd(t, "B", "11.3333");
		assertFalse(r.hasNext());
	}

	public void testRel0061() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT A.NAME, B.NOTE FROM IDOLS A" +
				" LEFT OUTER JOIN IDOL_HISTORIES B ON A.NAME = B.NAME");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");  eqs(t, "B.NOTE", "Dランク昇格");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");  eqs(t, "B.NOTE", "Cランク昇格");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");  eqs(t, "B.NOTE", "Bランク昇格");
		t = eqs(r);  eqs(t, "A.NAME", "高槻やよい");  eqs(t, "B.NOTE", "Dランク昇格");
		t = eqs(r);  eqs(t, "A.NAME", "菊地真");  eqs(t, "B.NOTE", "Dランク昇格");
		t = eqs(r);  eqs(t, "A.NAME", "菊地真");  eqs(t, "B.NOTE", "Cランク昇格");
		t = eqs(r);  eqs(t, "A.NAME", "我那覇響");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "島村卯月");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "中野有香");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "水本ゆかり");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "福山舞");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "椎名法子");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "今井加奈");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "持田亜里沙");  eqs(t, "B.NOTE", "");
		t = eqs(r);  eqs(t, "A.NAME", "三村かな子");  eqs(t, "B.NOTE", "");
		assertFalse(r.hasNext());
	}

	public void testRel0071() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT A.NAME FROM IDOLS A" +
				" WHERE EXISTS" +
				"  (SELECT NOTE FROM IDOL_HISTORIES WHERE NAME = A.NAME)");
		t = eqs(r);  eqs(t, "A.NAME", "天海春香");
		t = eqs(r);  eqs(t, "A.NAME", "高槻やよい");
		t = eqs(r);  eqs(t, "A.NAME", "菊地真");
		assertFalse(r.hasNext());
	}

	public void testRel0072() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS A" +
				" WHERE AGE = " +
				"  (SELECT AGE - 3 FROM IDOLS B WHERE B.B = A.B + 16)");
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		assertFalse(r.hasNext());
	}

	public void testRel0073() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS A" +
				" WHERE NAME IN " +
				"  (SELECT NAME FROM IDOLS WHERE B = 83)");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		assertFalse(r.hasNext());
	}

	public void testRel0081() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT DISTINCT B FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0082() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT B FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0091() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS WHERE B >= 83" +
				" UNION " +
				" SELECT NAME, B FROM IDOLS WHERE B <= 75");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		assertFalse(r.hasNext());
	}

	public void testRel0092() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS WHERE B >= 83" +
				" UNION ALL " +
				" SELECT NAME, B FROM IDOLS WHERE B <= 75");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		assertFalse(r.hasNext());
	}

	public void testRel0093() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT B FROM IDOLS WHERE B >= 83" +
				" UNION ALL " +
				" SELECT B FROM IDOLS WHERE B <= 75");
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "B", 90);
		t = eqs(r);  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "B", 64);
		assertFalse(r.hasNext());
	}

	public void testRel0094() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS" +
				" INTERSECT " +
				" SELECT NAME, B FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "天海春香");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "我那覇響");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "島村卯月");   eqs(t, "B", 83);
		t = eqs(r);  eqs(t, "NAME", "三村かな子");   eqs(t, "B", 90);
		assertFalse(r.hasNext());
	}

	public void testRel0095() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS" +
				" EXCEPT " +
				" SELECT NAME, B FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqs(t, "B", 76);
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqs(t, "B", 77);
		assertFalse(r.hasNext());
	}

	public void testRel0096() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, B FROM IDOLS" +
				" MINUS " +
				" SELECT NAME, B FROM IDOLS WHERE B >= 83");
		t = eqs(r);  eqs(t, "NAME", "高槻やよい");  eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "NAME", "菊地真");     eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "NAME", "中野有香");   eqs(t, "B", 77);
		t = eqs(r);  eqs(t, "NAME", "水本ゆかり");   eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "福山舞");     eqs(t, "B", 64);
		t = eqs(r);  eqs(t, "NAME", "椎名法子");    eqs(t, "B", 76);
		t = eqs(r);  eqs(t, "NAME", "今井加奈");    eqs(t, "B", 81);
		t = eqs(r);  eqs(t, "NAME", "持田亜里沙");  eqs(t, "B", 77);
		assertFalse(r.hasNext());
	}

	public void testRel0097() {
		RelationCursor r;
		RelationTuple t;

		r = itr("(SELECT 'Dランク昇格' AS NOTE" +
				" UNION " +
				" SELECT 'Cランク昇格')");
		t = eqs(r);  eqs(t, "NOTE", "Dランク昇格");
		t = eqs(r);  eqs(t, "NOTE", "Cランク昇格");
		assertFalse(r.hasNext());
	}

	public void testRel0098() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT NAME, NOTE FROM IDOL_HISTORIES" +
				" DIVIDE BY " +
				" (SELECT 'Dランク昇格' AS NOTE" +
				"  UNION " +
				"  SELECT 'Cランク昇格')");
		t = eqs(r);  eqs(t, "NAME", "天海春香");
		t = eqs(r);  eqs(t, "NAME", "菊地真");
		t = eqs(r);  eqs(t, "NAME", "如月千早");
		assertFalse(r.hasNext());
	}

}
