/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sqlcs.ddl;

import java.math.BigDecimal;
import java.sql.SQLException;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.Rational;

public final class SqlTypeNumeric extends SqlColumnType {

	/**
	 * 
	 */
	public static final int UNLIMITED = 0;

	private int length, scale;

	public SqlTypeNumeric(int j, int k) {
		length = j;
		scale  = k;
	}

	public int getLength() {
		return length;
	}

	public int getScale() {
		return scale;
	}

	@Override
	public Object cast(String s) throws SQLException {
		try {
			return s.equals("") ?
					"" : Rational.valueOf(new BigDecimal(s));
		} catch(NumberFormatException e) {
			throw ErrorBundle.getDefault(10035, s);
		}
	}

	@Override
	public String string(Object o) throws SQLException {
		BigDecimal d;

		if(o.equals("")) {
			return "";
		} else if(o instanceof Rational) {
			d = ((Rational)o).toBigDecimal(scale);
			if(length > 0 && d.scale() > length) {
				throw ErrorBundle.getDefault(10044);
			}
			return d.toString();
		} else {
			throw ErrorBundle.getDefault(10043);
		}
	}

	@Override
	public int getSqlType() {
		return java.sql.Types.NUMERIC;
	}

	@Override
	public Class<?> getTypeClass() {
		return net.morilib.db.misc.Rational.class;
	}

	public int hashCode() {
		return length * scale;
	}

	public boolean equals(Object o) {
		return (o instanceof SqlTypeNumeric &&
				((SqlTypeNumeric)o).length == length &&
				((SqlTypeNumeric)o).scale == scale);
	}

	public String toString() {
		if(scale > 0) {
			return "NUMERIC(" + length + "," + scale + ")";
		} else {
			return "NUMERIC(" + length + ")";
		}
	}

}
