/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.morilib.db.expr.RelationBinaryOperate;
import net.morilib.db.expr.RelationBinaryOperator;
import net.morilib.db.expr.RelationConst;
import net.morilib.db.expr.RelationExists;
import net.morilib.db.expr.RelationExpression;
import net.morilib.db.expr.RelationFunction;
import net.morilib.db.expr.RelationFunctionApply;
import net.morilib.db.expr.RelationIn;
import net.morilib.db.expr.RelationInSubquery;
import net.morilib.db.expr.RelationRefer;
import net.morilib.db.expr.RelationReferSubquery;
import net.morilib.db.expr.RelationTernaryOperate;
import net.morilib.db.expr.RelationTernaryOperator;
import net.morilib.db.expr.RelationUnaryOperate;
import net.morilib.db.expr.RelationUnaryOperator;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.Rational;
import net.morilib.db.misc.SqlResponse;
import net.morilib.db.relations.DefaultRelationTuple;
import net.morilib.db.relations.IsolatedRelation;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.OperatedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationAggregate;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.SingleTableRelation;
import net.morilib.db.relations.TableRelation;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.relations.Relations;
import net.morilib.db.relations.VirtualCrossJoinRelation;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.schema.WithSqlSchema;
import net.morilib.db.sqlcs.ddl.SqlAlterTableAdd;
import net.morilib.db.sqlcs.ddl.SqlAlterTableDrop;
import net.morilib.db.sqlcs.ddl.SqlAlterTableModify;
import net.morilib.db.sqlcs.ddl.SqlAlterTableRenameColumn;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlDropTable;
import net.morilib.db.sqlcs.ddl.SqlTruncateTable;
import net.morilib.db.sqlcs.dml.SqlBinaryOperation;
import net.morilib.db.sqlcs.dml.SqlDelete;
import net.morilib.db.sqlcs.dml.SqlExists;
import net.morilib.db.sqlcs.dml.SqlExpression;
import net.morilib.db.sqlcs.dml.SqlExpressions;
import net.morilib.db.sqlcs.dml.SqlFunction;
import net.morilib.db.sqlcs.dml.SqlIn;
import net.morilib.db.sqlcs.dml.SqlInSubquery;
import net.morilib.db.sqlcs.dml.SqlInsertSelect;
import net.morilib.db.sqlcs.dml.SqlInsertValues;
import net.morilib.db.sqlcs.dml.SqlNumeric;
import net.morilib.db.sqlcs.dml.SqlPlaceHolder;
import net.morilib.db.sqlcs.dml.SqlRelation;
import net.morilib.db.sqlcs.dml.SqlSelect;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import net.morilib.db.sqlcs.dml.SqlString;
import net.morilib.db.sqlcs.dml.SqlSubqueryLiteral;
import net.morilib.db.sqlcs.dml.SqlSymbol;
import net.morilib.db.sqlcs.dml.SqlTernaryOperation;
import net.morilib.db.sqlcs.dml.SqlUnaryOperation;
import net.morilib.db.sqlcs.dml.SqlUpdate;

public class DefaultSqlEngine extends SqlEngine {

	/**
	 * @param schema
	 */
	public DefaultSqlEngine(SqlSchema schema) {
		super(schema);
	}

	public Relation visit(SqlSelect t,
			RelationTuple e,
			List<Object> hl) throws IOException, SQLException {
		Map<String, SqlSetExpression> b;
		boolean[] ag = new boolean[1];
		Collection<RelationTuple> l;
		List<SqlColumnDefinition> h;
		List<RelationTuple> s;
		List<RelationAggregate> m;
		Map<String, Object> n;
		Relation r = null, a;
		RelationAggregate p;
		OperatedRelation o;
		RelationTuple[] d;
		RelationCursor c;
		RelationTuple w;
		SqlSchema f;
		String v, k;

		// with
		if((b = t.getWith()) != null) {
			f = new WithSqlSchema(schema);
			for(String x : b.keySet()) {
				schema.bindSchema(x,
						visit(b.get(x), Collections.emptyList()));
			}
		} else {
			f = schema;
		}

		// select
		if(t.getTables() != null && t.getTables().size() > 0) {
			for(SqlRelation x : t.getTables()) {
				a = visit(x, hl);
				r = (r != null) ?
						new VirtualCrossJoinRelation(r, a) : a;
			}

			if(r instanceof SingleTableRelation) {
				r = new IsolatedRelation(
						(SingleTableRelation)r,
						((SingleTableRelation)r).getName());
			} else if(r instanceof TableRenameRelation) {
				r = new IsolatedRelation(
						((TableRenameRelation)r).getRelation(),
						((TableRenameRelation)r).getName());
			}
		} else {
			r = Relations.DUAL;
		}

		// where
		r = r.select(this, f, visit(t.getWhere(), hl), null,
				t.getDistinct(), e, null, hl);
		o = Relations.operate(this, f, r, t.getData(),
				t.getGroupby(), hl, ag);

		// group by
		if((t.getGroupby() != null && t.getGroupby().size() > 0) ||
				ag[0]) {
			for(c = o.iterator(); c.hasNext(); c.next());
			n = new LinkedHashMap<String, Object>();
			l = t.getDistinct().create();
			h = new ArrayList<SqlColumnDefinition>();
			for(int i = 0; i < t.getData().size(); i++) {
				v = t.getData().get(i).getAs();
				v = v != null ? v : i + "";
				k = o.getColumnNames().get(i).getName();
				h.add(new SqlColumnDefinition(v,
						o.getDefinition(k).getType(),
						o.getDefinition(k).getAttributes()));
			}

			if(!(m = o.getMap()).get(0).isEmpty()) {
				for(Object z : m.get(0).keySet()) {
					for(int i = 0; i < t.getData().size(); i++) {
						p = m.get(i);
						v = t.getData().get(i).getAs();
						v = v != null ? v : i + "";
						if(p.containsKey(z)) {
							n.put(v, p.get(z).force());
						} else {
							n.put(v, "");
						}
					}
					l.add(new DefaultRelationTuple(n));
				}
			} else if(t.getGroupby() == null ||
					t.getGroupby().size() == 0) {
				for(int i = 0; i < t.getData().size(); i++) {
					v = t.getData().get(i).getAs();
					v = v != null ? v : i + "";
					if(m.get(i).getInit() != null) {
						n.put(v, m.get(i).getInit().force());
					} else {
						n.put(v, Rational.ZERO);
					}
				}
				l.add(new DefaultRelationTuple(n));
			}
			r = new TableRelation(h, l);

			// having
			r = r.select(this, f, visit(t.getHaving(), hl), null,
					t.getDistinct(), e, null, hl);
		} else {
			r = o;
		}

		// order by
		if(t.getOrderby() != null && t.getOrderby().size() > 0) {
			s = new ArrayList<RelationTuple>();
			c = r.iterator();
			while(c.hasNext())  s.add(c.next());
			d = s.toArray(new RelationTuple[0]);
			Arrays.sort(d, Relations.getComparator(r, t.getOrderby()));
			s = Arrays.asList(d);
			r = new TableRelation(r.getColumnNames(), s);
		}

		// projection
		l = t.getDistinct().create();
		n = new LinkedHashMap<String, Object>();
		c = r.iterator();
		while(c.hasNext()) {
			w = c.next();
			for(SqlColumnDefinition q : r.getColumnNames()) {
				n.put(q.getName(), w.get(q.getName()));
			}
			l.add(new DefaultRelationTuple(n));
		}
		r = new TableRelation(r.getColumnNames(), l);
		return r;
	}

	public RelationExpression visit(SqlExpression e,
			List<Object> h) throws SQLException {
		List<RelationExpression> l = new ArrayList<RelationExpression>();
		SqlTernaryOperation t;
		SqlBinaryOperation b;
		SqlUnaryOperation u;
		SqlInSubquery q;
		SqlIn n;
		int j;

		if(e == null) {
			return new RelationConst(Rational.ONE);
		} else if(e instanceof SqlNumeric) {
			return new RelationConst(((SqlNumeric)e).getValue());
		} else if(e instanceof SqlString) {
			return new RelationConst(((SqlString)e).getValue());
		} else if(e instanceof SqlSymbol) {
			return new RelationRefer(((SqlSymbol)e).getValue());
		} else if(e instanceof SqlSubqueryLiteral) {
			return new RelationReferSubquery(
					((SqlSubqueryLiteral)e).getQuery());
		} else if(e instanceof SqlPlaceHolder) {
			j = ((SqlPlaceHolder)e).getNumber() - 1;
			if(j < 0 || j >= h.size()) {
				throw ErrorBundle.getDefault(10036);
			}
			return new RelationConst(h.get(j));
		} else if(e instanceof SqlBinaryOperation) {
			b = (SqlBinaryOperation)e;
			return new RelationBinaryOperate(
					RelationBinaryOperator.get(b.getOperator()),
					visit(b.getOperand1(), h),
					visit(b.getOperand2(), h));
		} else if(e instanceof SqlUnaryOperation) {
			u = (SqlUnaryOperation)e;
			return new RelationUnaryOperate(
					RelationUnaryOperator.get(u.getOperator()),
					visit(u.getOperand1(), h));
		} else if(e instanceof SqlTernaryOperation) {
			t = (SqlTernaryOperation)e;
			return new RelationTernaryOperate(
					RelationTernaryOperator.get(t.getOperator()),
					visit(t.getOperand1(), h),
					visit(t.getOperand2(), h),
					visit(t.getOperand3(), h));
		} else if(e instanceof SqlIn) {
			n = (SqlIn)e;
			for(SqlExpression p : n.getValues())  l.add(visit(p, h));
			return new RelationIn(visit(n.getExpression(), h), l);
		} else if(e instanceof SqlInSubquery) {
			q = (SqlInSubquery)e;
			return new RelationInSubquery(
					visit(q.getExpression(), h),
					q.getSelect());
		} else if(e instanceof SqlExists) {
			return new RelationExists(((SqlExists)e).getQuery());
		} else if(e instanceof SqlFunction) {
			for(SqlExpression p : ((SqlFunction)e).getArguments()) {
				l.add(visit(p, h));
			}
			return new RelationFunctionApply(
					RelationFunction.get(((SqlFunction)e).getName()),
					l);
		} else if(e == SqlExpressions.ANY) {
			return Relations.ANY;
		} else {
			throw new RuntimeException();
		}
	}

	private void locktable(String name
			) throws IOException, SQLException {
		if(schema.isAutoCommit()) {
			// do nothing
		} else if(schema.isLocked(name)) {
			throw ErrorBundle.getDefault(10049, name);
		} else {
			schema.lock(name);
		}
	}

	private void checkused(String name
			) throws IOException, SQLException {
		if(schema.isAutoCommit()) {
			// do nothing
		} else if(schema.isUsed(name)) {
			throw ErrorBundle.getDefault(10049, name);
		}
	}

	@Override
	public synchronized int visit(SqlInsertValues t,
			List<Object> h) throws IOException, SQLException {
		Map<String, Object> m = new LinkedHashMap<String, Object>();
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = schema.readRelation(t.getName(), null);
		SqlCreateTable c = schema.getCreateTable(t.getName());
		RelationCursor j = r.iterator();
		RelationExpression x;
		RelationTuple p;
		List<String> l;
		String s;

		locktable(t.getName());
		if((l = t.getColumns()) == null) {
			l = new ArrayList<String>();
			for(SqlColumnDefinition v : r.getColumnNames()) {
				l.add(v.getName());
			}
		}

		for(int i = 0; i < t.getExprs().size(); i++) {
			x = visit((SqlExpression)t.getExprs().get(i), h);
			m.put(l.get(i),
					x.eval(this, schema, Relations.NULLTUPLE, null, null, h));
		}

		for(int i = 0; i < c.getColumnDefinitions().size(); i++) {
			s = c.getColumnDefinitions().get(i).getName();
			if(!m.containsKey(s))  m.put(s, "");
		}

		while(j.hasNext()) {
			p = j.next();
			if(Relations.isKey(p, m, c.getKeys())) {
				throw ErrorBundle.getDefault(10001);
			}
			z.add(p);
		}
		z.add(new DefaultRelationTuple(m));
		schema.writeRelation(t.getName(), z);
		return 1;
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.engine.SqlEngine#visit(net.morilib.db.engine.SqlFileSystem, net.morilib.db.sqlcs.dml.SqlUpdate)
	 */
	@Override
	public int visit(SqlUpdate t,
			List<Object> h) throws IOException, SQLException {
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = schema.readRelation(t.getTable(), null);
		RelationCursor j = r.iterator();
		RelationExpression x = null;
		List<RelationExpression> y;
		Map<String, Object> m;
		RelationTuple p;
		int n = 0;

		locktable(t.getTable());
		if(t.getWhere() != null)  x = visit(t.getWhere(), h);
		y = new ArrayList<RelationExpression>();
		for(SqlExpression b : t.getExpressions())  y.add(visit(b, h));
		while(j.hasNext()) {
			p = j.next();
			if(x != null &&
					!x.test(this, schema, p, null, null, h).isTrue()) {
				z.add(p);
			} else {
				m = new LinkedHashMap<String, Object>(p.toMap());
				for(int i = 0; i < y.size(); i++) {
					m.put(t.getNames().get(i),
							y.get(i).eval(this, schema, p, null, null, h));
				}
				z.add(new DefaultRelationTuple(m));
				n++;
			}
		}
		schema.writeRelation(t.getTable(), z);
		return n;
	}

	@Override
	public int visit(SqlDelete t,
			List<Object> h) throws IOException, SQLException {
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = schema.readRelation(t.getName(), null);
		RelationCursor j = r.iterator();
		RelationExpression x = null;
		RelationTuple p;
		int n = 0;

		locktable(t.getName());
		if(t.getExpression() != null)  x = visit(t.getExpression(), h);
		while(j.hasNext()) {
			p = j.next();
			if(x != null &&
					!x.test(this, schema, p, null, null, h).isTrue()) {
				z.add(p);
			} else {
				n++;
			}
		}
		schema.writeRelation(t.getName(), z);
		return n;
	}

	@Override
	public Object visit(SqlCreateTable c
			) throws IOException, SQLException {
		schema.putCreateTable(c.getName(), c);
		return new SqlResponse(true, "Table created");
	}

	@Override
	public Object visit(SqlDropTable c
			) throws IOException, SQLException {
		checkused(c.getName());
		schema.removeCreateTable(c.getName());
		return new SqlResponse(true, "Table dropped");
	}

	@Override
	public Object visit(SqlTruncateTable c
			) throws IOException, SQLException {
		checkused(c.getName());
		schema.truncateTable(c.getName());
		return new SqlResponse(true, "Table truncated");
	}

	@Override
	public Object visit(SqlAlterTableAdd c
			) throws IOException, SQLException {
		SqlCreateTable t = schema.getCreateTable(c.getName());
		List<SqlColumnDefinition> l;
		Map<String, Object> m;
		List<RelationTuple> z;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		checkused(c.getName());
		l = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		for(SqlColumnDefinition x : c.getColumnDefinitions()) {
			if(t.findColumn(x.getName()) != null) {
				throw ErrorBundle.getDefault(10005);
			}
			l.add(x);
		}

		z = new ArrayList<RelationTuple>();
		r = schema.readRelation(c.getName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(SqlColumnDefinition x : t.getColumnDefinitions()) {
				m.put(x.getName(), v.get(x.getName()));
			}

			for(SqlColumnDefinition x : c.getColumnDefinitions()) {
				m.put(x.getName(), "");
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getName(), l);
		schema.alterCreateTable(c.getName(), t);
		schema.writeRelation(c.getName(), z);
		return new SqlResponse(true, "Table columns added");
	}

	@Override
	public Object visit(SqlAlterTableModify c
			) throws IOException, SQLException {
		SqlCreateTable t = schema.getCreateTable(c.getName());
		List<SqlColumnDefinition> l, p;
		SqlColumnDefinition y;
		List<RelationTuple> z;
		Map<String, Object> m;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		checkused(c.getName());
		l = p = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		for(SqlColumnDefinition x : c.getColumnDefinitions()) {
			if((y = t.findColumn(x.getName())) == null) {
				throw ErrorBundle.getDefault(10006);
			}
			l.set(l.indexOf(y), x);
		}

		z = new ArrayList<RelationTuple>();
		r = schema.readRelation(c.getName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(int i = 0; i < l.size(); i++) {
				m.put(l.get(i).getName(), v.get(p.get(i).getName()));
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getName(), l);
		schema.alterCreateTable(c.getName(), t);
		schema.writeRelation(c.getName(), z);
		return new SqlResponse(true, "Table columns modified");
	}

	@Override
	public Object visit(SqlAlterTableDrop c
			) throws IOException, SQLException {
		SqlCreateTable t = schema.getCreateTable(c.getName());
		List<SqlColumnDefinition> l;
		SqlColumnDefinition y;
		List<RelationTuple> z;
		Map<String, Object> m;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		checkused(c.getName());
		l = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		for(String x : c.getColumnNames()) {
			if((y = t.findColumn(x)) == null) {
				throw ErrorBundle.getDefault(10006);
			}
			l.remove(y);
		}

		z = new ArrayList<RelationTuple>();
		r = schema.readRelation(c.getName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(int i = 0; i < l.size(); i++) {
				m.put(l.get(i).getName(), v.get(l.get(i).getName()));
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getName(), l);
		schema.alterCreateTable(c.getName(), t);
		schema.writeRelation(c.getName(), z);
		return new SqlResponse(true, "Table columns dropped");
	}

	@Override
	public Object visit(SqlAlterTableRenameColumn c
			) throws IOException, SQLException {
		SqlCreateTable t = schema.getCreateTable(c.getTableName());
		List<SqlColumnDefinition> l, p;
		SqlColumnDefinition y, d;
		List<RelationTuple> z;
		Map<String, Object> m;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		checkused(c.getTableName());
		l = p = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		if((y = t.findColumn(c.getColumnNameFrom())) == null) {
			throw ErrorBundle.getDefault(10006);
		}
		d = new SqlColumnDefinition(c.getColumnNameTo(), y.getType(),
				y.getAttributes());
		l.set(l.indexOf(y), d);

		z = new ArrayList<RelationTuple>();
		r = schema.readRelation(c.getTableName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(int i = 0; i < l.size(); i++) {
				m.put(l.get(i).getName(), v.get(p.get(i).getName()));
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getTableName(), l);
		schema.alterCreateTable(c.getTableName(), t);
		schema.writeRelation(c.getTableName(), z);
		return new SqlResponse(true, "Table column renamed");
	}

	@Override
	public int visit(SqlInsertSelect t,
			List<Object> h) throws IOException, SQLException {
		Map<String, Object> m = new LinkedHashMap<String, Object>();
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = schema.readRelation(t.getName(), null);
		SqlCreateTable c = schema.getCreateTable(t.getName());
		RelationCursor j, k;
		RelationTuple p, q;
		Relation v;
		int w;

		locktable(t.getName());
		v = visit(t.getSelect(), Relations.NULLTUPLE, h);
		if(t.getColumns().size() != r.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		j = r.iterator();
		while(j.hasNext())  z.add(j.next());

		k = v.iterator();
		for(w = 0; k.hasNext(); w++) {
			q = k.next();
			for(int i = 0; i < t.getColumns().size(); i++) {
				m.put(t.getColumns().get(i),
						q.get(v.getColumnNames().get(i).getName()));
			}

			j = r.iterator();
			while(j.hasNext()) {
				p = j.next();
				if(Relations.isKey(p, m, c.getKeys())) {
					throw ErrorBundle.getDefault(10001);
				}
			}
			z.add(new DefaultRelationTuple(m));
		}
		schema.writeRelation(t.getName(), z);
		return w;
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.engine.SqlEngine#commit()
	 */
	@Override
	public void commit() throws IOException, SQLException {
		// do nothing
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.engine.SqlEngine#rollback()
	 */
	@Override
	public void rollback() throws IOException, SQLException {
		// do nothing
	}

}
