/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.jdbc;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Statement;
import java.sql.Types;

import junit.framework.TestCase;

public class RelationsJDBCTest extends TestCase {

	public void testConn001() {
		Connection c = null;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:error");
			assertTrue(c instanceof RelationsConnection);
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn002() {
		Connection c = null;
		Statement s = null;
		ResultSet r = null;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:error");
			s = c.createStatement();
			r = s.executeQuery("SELECT 1 AS A");
			assertTrue(r.next());
			assertEquals(r.getString("A"), "1");
			assertFalse(r.next());
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn003() {
		Connection c = null;
		Statement s = null;
		ResultSet r = null;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:error");
			s = c.createStatement();
			r = s.executeQuery("SELECT MESSAGE FROM ERROR_MSG WHERE CODE = '99999'");
			assertTrue(r.next());
			assertEquals(r.getString("MESSAGE"), "unknown error");
			assertFalse(r.next());
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn004() {
		Connection c = null;
		PreparedStatement s = null;
		ResultSet r = null;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:error");
			s = c.prepareStatement("SELECT MESSAGE FROM ERROR_MSG WHERE CODE = ?");
			s.setString(1, "99999");
			r = s.executeQuery();
			assertTrue(r.next());
			assertEquals(r.getString("MESSAGE"), "unknown error");
			assertFalse(r.next());
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn005() {
		Connection c = null;
		PreparedStatement s = null;
		ResultSet r = null;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:error");
			s = c.prepareStatement("SELECT MESSAGE FROM ERROR_MSG WHERE CODE = ?");
			s.setString(1, "999'99");
			r = s.executeQuery();
			assertFalse(r.next());
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn006() {
		Connection c = null;
		Statement s = null;
		int j;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:sandbox");
			s = c.createStatement();
			j = s.executeUpdate("CREATE TABLE A (NAME VARCHAR(20))");
			assertEquals(0, j);
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn007() {
		Connection c = null;
		PreparedStatement s = null;
		ResultSet r = null;
		int j;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:sandbox");
			s = c.prepareStatement("CREATE TABLE IDOLS (NAME VARCHAR(20) PRIMARY KEY, B NUMERIC(3))");
			j = s.executeUpdate();
			assertEquals(0, j);
			s = c.prepareStatement("INSERT INTO IDOLS (NAME, B) VALUES (?, ?)");
			s.setString(1, "双海'十勝'亜美");  s.setInt(2, 78);
			j = s.executeUpdate();
			assertEquals(1, j);
			s = c.prepareStatement("INSERT INTO IDOLS (NAME, B) VALUES (?, ?)");
			s.setString(1, "双海'十勝'真美");  s.setInt(2, 78);
			j = s.executeUpdate();
			assertEquals(1, j);

			s = c.prepareStatement("select * from idols where b = ?");
			s.setInt(1, 78);
			r = s.executeQuery();
			assertTrue(r.next());
			assertEquals(r.getString("NAME"), "双海'十勝'亜美");
			assertTrue(r.next());
			assertEquals(r.getString("NAME"), "双海'十勝'真美");
			assertFalse(r.next());
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testConn008() {
		Connection c = null;
		PreparedStatement s = null;
		ResultSet r = null;
		int j;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:sandbox");

			s = c.prepareStatement("CREATE TABLE IDOLS"
					+ " (NAME VARCHAR(20) PRIMARY KEY, B NUMERIC(3))");
			j = s.executeUpdate();
			assertEquals(0, j);

			s = c.prepareStatement("INSERT INTO IDOLS (NAME, B) VALUES (?, ?)");
			s.setString(1, "双海'十勝'亜美");  s.setInt(2, 78);
			j = s.executeUpdate();
			assertEquals(1, j);

			s = c.prepareStatement("CREATE TABLE IDOLS2 "
					+ "(NAME2 VARCHAR(20) PRIMARY KEY, B2 NUMERIC(3))");
			j = s.executeUpdate();
			assertEquals(0, j);

			s = c.prepareStatement("INSERT INTO IDOLS2 (NAME2, B2) VALUES (?, ?)");
			s.setString(1, "如月千早");  s.setInt(2, 72);
			j = s.executeUpdate();
			assertEquals(1, j);
			s = c.prepareStatement("INSERT INTO IDOLS2 (NAME2, B2) VALUES (?, ?)");
			s.setString(1, "秋月律子");  s.setInt(2, 85);
			j = s.executeUpdate();
			assertEquals(1, j);

			s = c.prepareStatement(
					"INSERT INTO IDOLS (NAME, B) SELECT NAME2, B2 FROM IDOLS2");
			j = s.executeUpdate();
			assertEquals(2, j);

			s = c.prepareStatement("select * from idols order by b");
			r = s.executeQuery();
			assertTrue(r.next());
			assertEquals(r.getString("NAME"), "如月千早");
			assertTrue(r.next());
			assertEquals(r.getString("NAME"), "双海'十勝'亜美");
			assertTrue(r.next());
			assertEquals(r.getString("NAME"), "秋月律子");
			assertFalse(r.next());
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	public void testMeta001() {
		Connection c = null;
		PreparedStatement s = null;
		ResultSet r = null;
		ResultSetMetaData m;
		int j;

		try {
			Class.forName("net.morilib.db.jdbc.RelationsDriver");
			c = DriverManager.getConnection("jdbc:relations:sandbox");
			s = c.prepareStatement("CREATE TABLE IDOLS"
					+ " (NAME VARCHAR(20) PRIMARY KEY, B NUMERIC(4, 1), BIRTH DATE)");
			j = s.executeUpdate();
			assertEquals(0, j);

			s = c.prepareStatement("select * from idols");
			r = s.executeQuery();
			m = r.getMetaData();
			assertEquals(3, m.getColumnCount());
			assertEquals(20, m.getColumnDisplaySize(1));
			assertEquals(20, m.getPrecision(1));
			assertEquals(0, m.getScale(1));
			assertEquals(Types.VARCHAR, m.getColumnType(1));
			assertEquals("NAME", m.getColumnName(1));
			assertEquals("VARCHAR(20)", m.getColumnTypeName(1));
			assertEquals(ResultSetMetaData.columnNoNulls, m.isNullable(1));
			assertFalse(m.isReadOnly(1));
			assertTrue(m.isWritable(1));
			assertTrue(m.isDefinitelyWritable(1));
			assertTrue(m.isSearchable(1));

			assertEquals(4, m.getColumnDisplaySize(2));
			assertEquals(4, m.getPrecision(2));
			assertEquals(1, m.getScale(2));
			assertEquals(Types.NUMERIC, m.getColumnType(2));
			assertEquals("B", m.getColumnName(2));
			assertEquals("NUMERIC(4,1)", m.getColumnTypeName(2));
			assertEquals(ResultSetMetaData.columnNullable, m.isNullable(2));
			assertFalse(m.isReadOnly(2));
			assertTrue(m.isWritable(2));
			assertTrue(m.isDefinitelyWritable(2));
			assertTrue(m.isSearchable(2));

			assertEquals(0, m.getColumnDisplaySize(3));
			assertEquals(0, m.getPrecision(3));
			assertEquals(0, m.getScale(3));
			assertEquals(Types.TIMESTAMP, m.getColumnType(3));
			assertEquals("BIRTH", m.getColumnName(3));
			assertEquals("DATE", m.getColumnTypeName(3));
			assertEquals(ResultSetMetaData.columnNullable, m.isNullable(3));
			assertFalse(m.isReadOnly(3));
			assertTrue(m.isWritable(3));
			assertTrue(m.isDefinitelyWritable(3));
			assertTrue(m.isSearchable(3));
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if(r != null)  r.close();
				if(s != null)  s.close();
				if(c != null)  c.close();
			} catch(Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

}
