/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.misc;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import net.morilib.parser.html.HTMLHandler;
import net.morilib.parser.html.HTMLParser;

public class XlsxSharedString {

	private static enum S {
		INIT, SST, SI, T, PHONETIC
	}

	private class R implements HTMLHandler {

		private int rowno = 0;
		private S stat = S.INIT;

		@Override
		public void string(String s) throws SQLException {
			if(stat == S.T && values.size() < rowno) {
				values.add(s);
			}
		}

		@Override
		public void startTag(String s) throws SQLException {
			switch(stat) {
			case INIT:
				if(s.equals("sst"))  stat = S.SST;
				break;
			case SST:
				if(s.equals("si")) {
					stat = S.SI;
					phonetic.add("");
					rowno++;
				}
				break;
			case SI:
				if(s.equals("t")) {
					stat = S.T;
				} else if(s.equals("phoneticPr")) {
					stat = S.PHONETIC;
				}
				break;
			case T:
			case PHONETIC:
				break;
			}
		}

		@Override
		public void endTag(String s) throws SQLException {
			switch(stat) {
			case INIT:
				break;
			case SST:
				if(s.equals("sst"))  stat = S.INIT;
				break;
			case SI:
				if(s.equals("si"))  stat = S.SST;
				break;
			case T:
				if(s.equals("t"))  stat = S.SI;
				break;
			case PHONETIC:
				if(s.equals("phoneticPr"))  stat = S.SI;
				break;
			}
		}

		@Override
		public void tagAttribute(String k,
				String v) throws SQLException {
			if(stat == S.PHONETIC && k.equals("fontId")) {
				phonetic.set(values.size() - 1, v);
			}
		}

		@Override
		public void meta(String s) throws SQLException {
			// do nothing
		}

	}

	private static final String META =
			"<?xml version=\"1.0\" encoding=\"UTF-8\"" +
			" standalone=\"true\"?>";
	private static final String SSTO =
			"sst uniqueCount=\"%d\" count=\"%d\"" +
			" xmlns=\"http://schemas.openxmlformats.org/spreadsheetml/2006/main\">";

	private List<String> values;
	private List<String> phonetic;

	/**
	 * 
	 * @param ins
	 * @throws IOException
	 * @throws SQLException
	 */
	public XlsxSharedString(
			InputStream ins) throws SQLException, IOException {
		BufferedReader r;
		R h = new R();

		values = new ArrayList<String>();
		phonetic = new ArrayList<String>();
		r = new BufferedReader(new InputStreamReader(ins, "UTF-8"));
		HTMLParser.parse(h, r);
	}

	/**
	 * 
	 */
	public XlsxSharedString() {
		values = new ArrayList<String>();
		phonetic = new ArrayList<String>();
	}

	/**
	 * 
	 * @param ins
	 * @return
	 * @throws SQLException
	 * @throws IOException
	 */
	@SuppressWarnings("resource")
	public static XlsxSharedString fromXlsx(
			InputStream ins) throws SQLException, IOException {
		ZipInputStream zip = null;
		ZipEntry ent;

		zip = new ZipInputStream(ins);
		while((ent = zip.getNextEntry()) != null) {
			if(ent.getName().equals("xl/sharedStrings.xml")) {
				return new XlsxSharedString(zip);
			}
		}
		return null;
	}

	/**
	 * 
	 * @param ind
	 * @return
	 */
	public String getString(int ind) {
		return values.get(ind);
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public int addString(String s) {
		int c;

		if((c = values.indexOf(s)) < 0) {
			values.add(s);
			phonetic.add("1");
			return values.size() - 1;
		} else {
			return c;
		}
	}

	/**
	 * 
	 * @param ous
	 * @throws IOException
	 */
	public void write(OutputStream ous) throws IOException {
		PrintWriter w;

		w = new PrintWriter(new BufferedWriter(
				new OutputStreamWriter(ous, "UTF-8")));
		w.println(META);
		w.format(SSTO, values.size(), values.size());
		for(int i = 0; i < values.size(); i++) {
			w.println("<si>");
			w.format("<t>%s</t>", values.get(i));
			w.format("<phoneticPr fontId=\"%s\"/>", phonetic.get(i));
			w.println("</si>");
		}
		w.println("</sst>");
	}

}
