/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.ParseDate;
import net.morilib.db.relations.DefaultRelationTuple;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.SingleTableRelation;
import net.morilib.db.sqlcs.ddl.SqlColumnAttribute;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlColumnType;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlTypeDate;
import net.morilib.db.sqlcs.ddl.SqlTypeNumeric;
import net.morilib.db.sqlcs.ddl.SqlTypeVarchar;
import net.morilib.parser.csv.CSVConfig;
import net.morilib.parser.csv.CSVException;
import net.morilib.parser.csv.StringCSVPullParser;

public final class SqlSchemata {

	//
	private static final CSVConfig DEFT =
			new CSVConfig(",", '\"', false);
	private static final Pattern PT1 =
			Pattern.compile("([0-9]+)|([0-9]*)\\.([0-9]+)");
	private static final Pattern PT2 =
			Pattern.compile("(.*/)?([^.]+)");

	private static final SqlTypeNumeric UNLIMITED_INTEGER =
			new SqlTypeNumeric(SqlTypeNumeric.UNLIMITED, 0);

	private static int seq = 1;

	private SqlSchemata() {}

	private static SqlTypeNumeric getlen(SqlTypeNumeric n, Matcher m) {
		int j;

		if(m.group(1) != null) {
			return n == null || n.getScale() <= 0 ?
					UNLIMITED_INTEGER : n;
		} else {
			j = m.group(3).length();
			return n == null || n.getScale() <= j ?
					new SqlTypeNumeric(SqlTypeNumeric.UNLIMITED, j) : n;
		}
	}

	public static SqlCreateTable guessTable(String s,
			InputStream ins) throws IOException, SQLException {
		StringCSVPullParser p = null;
		List<SqlColumnDefinition> l;
		BufferedReader r = null;
		SqlColumnType[] t;
		List<String> n;
		Set<String> z;
		String[] y;
		Matcher w;
		String v;

		try {
			r = new BufferedReader(new InputStreamReader(ins));
			p = new StringCSVPullParser(r, DEFT);

			if(!p.next()) {
				throw ErrorBundle.getDefault(10018);
			} else {
				n = new ArrayList<String>(Arrays.asList(p.get()));
				z = new HashSet<String>();
				for(int i = 0; i < n.size(); i++) {
					if(n.get(i).equals("") || z.contains(n.get(i))) {
						n.set(i, n.get(i).toUpperCase() + i);
					} else {
						n.set(i, n.get(i).toUpperCase());
					}
				}
			}

			t = new SqlColumnType[n.size()];
			for(int i = 0; i < n.size(); i++)  t[i] = null;
			while(p.next()) {
				y = p.get();
				for(int i = 0; i < n.size(); i++) {
					if(i >= y.length) {
						// do nothing
					} else if((t[i] == null || t[i] instanceof SqlTypeNumeric) &&
							(w = PT1.matcher(y[i])).matches()) {
						t[i] = getlen((SqlTypeNumeric)t[i], w);
					} else if((t[i] == null || t[i] instanceof SqlTypeDate) &&
							ParseDate.getDate(y[i]) != null) {
						t[i] = t[i] == null ? new SqlTypeDate() : t[i];
					} else if(!(t[i] instanceof SqlTypeVarchar)) {
						t[i] = new SqlTypeVarchar(
								SqlTypeVarchar.UNLIMITED);
					}
				}
			}

			l = new ArrayList<SqlColumnDefinition>();
			for(int i = 0; i < n.size(); i++) {
				l.add(new SqlColumnDefinition(n.get(i),
						t[i],
						EnumSet.noneOf(SqlColumnAttribute.class)));
			}
			w = PT2.matcher(s);
			v = w.lookingAt() ? w.group(2) : "table" + (seq++);
			return new SqlCreateTable(v.toUpperCase(), l);
		} catch(CSVException e) {
			throw ErrorBundle.getDefault(10042);
		} finally {
			if(p != null)  p.close();
		}
	}

	public static SqlCreateTable guessTable(String s,
			List<List<String>> tbl) throws IOException, SQLException {
		List<SqlColumnDefinition> l;
		SqlColumnType[] t;
		List<String> n;
		Set<String> z;
		String[] y;
		Matcher w;
		String v;

		if(tbl.size() == 0) {
			throw ErrorBundle.getDefault(10018);
		} else {
			n = new ArrayList<String>(tbl.get(0));
			z = new HashSet<String>();
			for(int i = 0; i < n.size(); i++) {
				if(n.get(i).equals("") || z.contains(n.get(i))) {
					n.set(i, n.get(i).toUpperCase() + i);
				} else {
					n.set(i, n.get(i).toUpperCase());
				}
			}
		}

		t = new SqlColumnType[n.size()];
		for(int i = 0; i < n.size(); i++)  t[i] = null;
		for(int j = 1; j < tbl.size(); j++) {
			y = tbl.get(j).toArray(new String[0]);
			for(int i = 0; i < n.size(); i++) {
				if(i >= y.length) {
					// do nothing
				} else if((t[i] == null || t[i] instanceof SqlTypeNumeric) &&
						(w = PT1.matcher(y[i])).matches()) {
					t[i] = getlen((SqlTypeNumeric)t[i], w);
				} else if((t[i] == null || t[i] instanceof SqlTypeDate) &&
						ParseDate.getDate(y[i]) != null) {
					t[i] = t[i] == null ? new SqlTypeDate() : t[i];
				} else if(!(t[i] instanceof SqlTypeVarchar)) {
					t[i] = new SqlTypeVarchar(
							SqlTypeVarchar.UNLIMITED);
				}
			}
		}

		l = new ArrayList<SqlColumnDefinition>();
		for(int i = 0; i < n.size(); i++) {
			if(i == 0) {
				l.add(new SqlColumnDefinition(n.get(i),
						t[i],
						EnumSet.of(SqlColumnAttribute.PRIMARY_KEY,
								SqlColumnAttribute.NOT_NULL)));
			} else {
				l.add(new SqlColumnDefinition(n.get(i),
						t[i],
						EnumSet.noneOf(SqlColumnAttribute.class)));
			}
		}
		w = PT2.matcher(s);
		v = w.lookingAt() ? w.group(2) : "table" + (seq++);
		return new SqlCreateTable(v.toUpperCase(), l);
	}

	/**
	 * 
	 * @param name
	 * @param as
	 * @param l
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static NamedRelation readRelation(String name, String as,
			List<List<String>> l) throws IOException, SQLException {
		List<SqlColumnDefinition> d;
		BufferedReader b = null;
		List<RelationTuple> r;
		Map<String, Object> m;
		SqlCreateTable t;

		// from HTML
		try {
			t = SqlSchemata.guessTable(name, l);
			r = new ArrayList<RelationTuple>();
			m = new LinkedHashMap<String, Object>();
			d = t.getColumnDefinitions();
			for(int i = 1; i < l.size(); i++) {
				for(int j = 0; j < d.size(); j++) {
					m.put(d.get(j).getName(),
							d.get(j).getType().cast(l.get(i).get(j)));
				}
				r.add(new DefaultRelationTuple(m));
			}
			return new SingleTableRelation(t,
					as != null ? as : name,
					r);
		} finally {
			if(b != null)  b.close();
		}
	}

	public static SqlCreateTable guessTable(
			String s) throws IOException, SQLException {
		return guessTable(s, new FileInputStream(s));
	}

}
