/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.command;

import jp.sourceforge.expression_computer.Command;
import jp.sourceforge.expression_computer.ComputeContext;
import jp.sourceforge.expression_computer.Function;
import jp.sourceforge.expression_computer.type.FloatingPointLiteral;
import jp.sourceforge.expression_computer.util.Validator;

/**
 * <p>
 * 関数を呼び出し、その結果をスタックにプッシュします。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class FunctionCallCommand implements Command {

    private String name;

    private int    argumentNumber;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param name
     *            関数名。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。<br>
     *            識別子の形式ではない場合、{@link IllegalArgumentException}例外をスローします。
     * @param argumentNumber
     *            引数の個数。<br>
     *            0未満の場合、{@link IllegalArgumentException}例外をスローします。
     */
    public FunctionCallCommand(String name, int argumentNumber) {
        if (name == null) {
            throw new NullPointerException("nameがnullです。");
        }
        if (!Validator.isIdentifier(name)) {
            throw new IllegalArgumentException("nameが識別子の形式ではありません。");
        }
        if (argumentNumber < 0) {
            throw new IllegalArgumentException("{argumentNumber < 0}です。");
        }
        this.name = name;
        this.argumentNumber = argumentNumber;
    }

    /**
     * {@inheritDoc}
     */
    public void execute(ComputeContext context) {
        Function func = context.getFunction(this.name);

        double[] arguments = new double[this.argumentNumber];
        for (int i = (this.argumentNumber - 1); i >= 0; i--) {
            arguments[i] = context.popStack().getValue(context);
        }

        double resultValue = func.call(arguments);

        context.pushStack(new FloatingPointLiteral(resultValue));
    }

    /**
     * <p>
     * 関数名を返します。
     * </p>
     * 
     * @return 関数名。
     */
    public String getName() {
        return this.name;
    }

    /**
     * <p>
     * 引数の個数を返します。
     * </p>
     * 
     * @return 引数の個数。
     */
    public int getArgumentNumber() {
        return this.argumentNumber;
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return this.getClass().getName() + "[name=" + this.name + ", argumentNumber=" + this.argumentNumber + "]";
    }

}
