/**
 * @file memory_check.c
 * @brief Check memory leak
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "memory_check.h"
#include "internal_lock.h"

#ifdef MEMORY_CHECK
typedef struct {
	QUEUE que;
	void* pointer;
	char file[128];
	int line;
	char express[256];
	int size;
}mc_node_t;

mc_node_t __g_mc_queue;
__internal_lock_t __g_mc_lock;

/**********************************************************************/
/* Function name: mc_init                                             */
/* Description: memory check init function                            */
/* Return type - int :                                                */
/* Argument - void:                                                   */
/**********************************************************************/
int mc_init(void)
{
	QueInit((QUEUE*)&__g_mc_queue);
	__internal_lock_init(&__g_mc_lock);
	return 0;
}

/**********************************************************************/
/* Function name: mc_showinfo                                         */
/* Description: Print out all detected memory leak. Free them at last */
/* Return type - void :                                               */
/* Argument - void:                                                   */
/**********************************************************************/
void mc_showinfo(void)
{
	int count = 0;
	mc_node_t* node = NULL;
	__internal_lock(&__g_mc_lock);
	while( (node = (mc_node_t*)QueRemoveNext((QUEUE*)&__g_mc_queue)) != NULL ) {
		printf("-----------------------------------\n%d) %s:%d ---> %d(%d) [%s]\n",
			++count, node->file, node->line, node->pointer, node->size, node->express);
		free(node->pointer);
		free(node);
		node = NULL;
	} 
	if( count > 0 ) printf("-----------------------------------\n");
	__internal_unlock(&__g_mc_lock);
	__internal_lock_destroy(&__g_mc_lock);
}

/**********************************************************************/
/* Function name: mc_malloc                                           */
/* Description: malloc and collect related info                       */
/* Return type - void* :                                              */
/* Argument - const char* file: malloc at which file                  */
/* Argument - const int line:  malloc at which line                   */
/* Argument - const int size:  how many size                          */
/* Argument - const char* exp:  malloc statement                      */
/**********************************************************************/
void* mc_malloc(const char* file, const int line, const int size, const char* exp)
{
	void* p = malloc(size);
	if( p != NULL ) {
		mc_node_t* node = malloc(sizeof(mc_node_t));
		strcpy(node->file, file);
		strcpy(node->express, exp);
		node->pointer = p;
		node->line = line;
		node->size = size;
		__internal_lock(&__g_mc_lock);
		QueInsert((QUEUE*)node, (QUEUE*)&__g_mc_queue);
		__internal_unlock(&__g_mc_lock);
	}
	return p;
}

mc_node_t* find_and_remove_mc_node(const void* p);
/**********************************************************************/
/* Function name: mc_free                                             */
/* Description:  Free and release additionall info                    */
/* Return type - void :                                               */
/* Argument - void* p: dest pointer                                   */
/**********************************************************************/
void mc_free(void* p)
{
	if( p != NULL ) {
		mc_node_t* node = find_and_remove_mc_node(p);
		if( node ) { free(node); }
		free(p);
	}
}

/**********************************************************************/
/* Function name: find_and_remove_mc_node                             */
/* Description: internal use. get related info by pointer             */
/* Return type - mc_node_t* :  related info pointer                   */
/* Argument - const void* p:  dest pointer                            */
/**********************************************************************/
mc_node_t* find_and_remove_mc_node(const void* p)
{
	QUEUE* entry = NULL;
	int offset = sizeof(QUEUE);
	entry = QueSearch((QUEUE*)&__g_mc_queue, (QUEUE*)&__g_mc_queue, (int)p, offset);

	/* Reach end, not find target */
	if( entry == (QUEUE*)&__g_mc_queue ) { 
		return NULL;	
	}

	QueRemove(entry);
	return (mc_node_t*)entry;
}

#endif
