/**
 * @file nes_posix_setpriority.c
 * @brief This file is part of the posix library
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <btron/errcode.h>
#include <btron/proctask.h>

#include "common.h"
#include "nes_posix_setpriority.h"

/**
 * setpriority -  sets the priorities  of  all of the specified processes 
 *                to the specified value.
 * @param 	which :Specifies the type of target. Can be one of PRIO_PROCESS,
 *                 PRIO_PGRP, or PRIO_USER.
 *          who : Is the target of the nes_posix_setpriority() request; a process ID,
 *                process group ID, or user ID, respectively, depending on 
 *                the value of which. A value of 0 indicates that the target
 *                is the current process, process group, or user.
 *          prio:Is the new nice value for the process. Values in the range
 *                [-NZERO,(NZERO-1)] are valid; values outside that range are
 *                silently clipped to this range.
 * @retval int    0: success
 *               -1: fail 
 *                   EINVAL : which was not one of PRIO_PROCESS, PRIO_PGRP,
 *                            or PRIO_USER.
 *                   EPERM :  A process was located, but neither the real nor
 *                            effective user ID of the executing process match
 *                            the effective user ID of the process whose nice 
 *                            value is to be changed. 
 *                   ESRCH :  No process was located using the which and who
 *                            values specified
 * 
 */
int nes_posix_setpriority(int which, int who, int prio)
{
	WERR ret = 0;
	
	if(((which != PRIO_PROCESS) && (which != PRIO_PGRP) && (which != PRIO_USER)) || (who <= 0))
	{
		ret = -1;
		errno = EINVAL;		
		return ret;		
	}

	if(prio < 0  || prio > 255)
	{
		ret = -1;
		errno = EINVAL;		
		return ret;	
	}
	
	ret = b_chg_pri(who, prio, P_ABS);
	if(ret < 0)
	{		
		DP(("b_chg_pri Failed:[%d]\n", ret));
		switch(ret)
		{
			case ER_NOPRC : 
			case ER_ID : 			
											ret = -1;
											errno = ESRCH ;	
											return ret;		
	
			case ER_PPRI : 
											ret = -1;
											errno = EDOM ;	
											return ret;	
											
			case ER_PAR : 
											ret = -1;
											errno = EINVAL ;	
											return ret;	
			default : 
											ret = -1;
											errno = EINVAL ;	
											return ret;									
									 
		}	
	}
	else
	{	
		ret = 0;
		return ret;
	}

}

