/**
 * @file posix_system.c
 * @brief This file is part of the posix library
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <tstring.h>

#include <btron/file.h>
#include <sys/consio.h>
#include <btron/message.h>
#include <btron/proctask.h>
#include <btron/errcode.h>

#include "nes_posix_stdlib.h"

static int run_new_shell(int* port, int* pid);
/**
 * posix_system - Use cli to excute shell command
 * @param command	Command string
 * @retval int		0: Success 1: shell is valid -1: failed
 * 
 * @note
 * Not support '&' mode. Like "ls -l &"
 */
int nes_posix_system(const char* command)
{
	int ret = 0;
	int pid = 0;
	int port = 0;
	int bgmode = 0;

	if( !command ) {
		return 1; /* 'shell' cli is always valid */
	}

	/* Run shell */	
	ret = run_new_shell(&port, &pid);
	if( ret != 0 ) {
		return -1;
	}

	/* Build command that run under shell */	
	int cmd_size = strlen(command) + strlen("\r\nexit\r\n") + 1;
	char* cmd = malloc(cmd_size);
	if( !cmd ) {
		cons_conf(CS_DELETE, &port);
		return -1;
	}

	memset(cmd, 0, sizeof(cmd_size));
	strcpy(cmd, command);
	strcat(cmd, "\r\nexit\r\n");  /* when command end, shell need to exit at once */

	/* Let shell run command */
	ret = cons_put(port, cmd, strlen(cmd), -1);

	if( bgmode ) { /* Sync mode or Aync mode ? */
		//wait_shell_end(pid);
		return -1; /* Not support background run mode */
	}

	int end = 0;
	while( 1 ) {

		ret = b_prc_sts(pid, NULL, NULL);
		if( ret < 0 ) { /* Shell exit now */
			end = 1;
		}

		/* Output command result to current shell */
		char buf[2048 + 1] = {0};
		ret = cons_get(port, buf, 2048, 1);
		if( ret < 0 ) {
			break;
		}
		buf[ret] = 0;
		printf("%s", buf);

		if( end ) {
			break;
		}
	}

	/* Release all resource */	
	cons_conf(CS_DELETE, &port);
	free(cmd);

	return 0;
}

/*
   static int wait_shell_end(const int pid)
   {
   MSG_SYS2 msg;
   memset(&msg, 0, sizeof(msg));

   int ret = 0;
   while( 1 ) {
   ret = b_rcv_msg(MM_SYS2, (MESSAGE*)&msg, sizeof(msg), WAIT);
   if( ret == pid ) {
   break;
   }
   }
   return 0;	
   }
 */

static void conv_pathname( const char *src, TC *dest )
{
	char c; const char *p; int n, k; TC buf;

	p = src; n = 0;
	for ( ; (c = *p) != '\0' && n < FILENAME_MAX - 1; p++) {
		if (c == '/') {
			dest[n++] = TC_FDLM;
		} else if (c == ':') {
			dest[n++] = TC_FSEP;
		} else if ((k = euctotc(&buf, p)) > 0) {
			dest[n++] = buf;
			p += (k - 1); 
		}
	}
	dest[n] = 0x0000; 

}

/**
 * run_new_shell - 
 * @param port
 * @param pid
 * @retval static int 
 * 
 * @note
 */
static int run_new_shell(int* port, int* pid)
{
	UW  arg[5] = {0}; /* Args used to create console */
	int consRet;
	int wRet;
	W pri = -1;
	LINK lnk;
	MESSAGE msg;  
	TC tcPath[256+ 1] = {0};

	/* Set console type */
	arg[1] = CONF_BUFIO;

	/* Create a buffer IO console */
	consRet = cons_conf(CS_CREATE, arg);
	if (consRet != 0) { /* Create console error */
		return -1;
	}

	/* Switche the invoking process console to the created buffer IO console */
	consRet = cons_conf(CS_SETPORT, arg);
	if (consRet != 0) { /* Switche console error */
		cons_conf(CS_DELETE, arg);
		return -1;
	} 

	/* Convert LF to CR.LF mode */
	cons_ioctl(NEWLINE, 1);

	/* Convert path name to TC format */
	conv_pathname("cli", tcPath);

	/* Get link to the file */
	wRet = b_get_lnk(tcPath, &lnk, F_NORM);
	switch(wRet)
	{
		case   0:
		case   1:
		case   2:
			break;
		default :
			return -1;
	}

	/* create process */
	int len;
	TC tcBody[50] = {0};
	char* eucBody = "STARTUP.CLI";
	len = eucstotcs(tcBody, eucBody);
	msg.msg_type = 0;
	msg.msg_size = tc_strlen(tcBody)*2;
	memset(&msg.msg_body, 0, sizeof(msg.msg_body));
	memcpy(&msg.msg_body, tcBody, tc_strlen(tcBody)*2);

	wRet = b_cre_prc(&lnk, pri, &msg);
	if(wRet <= 0) {
		cons_conf(CS_DELETE, arg);
		return -1;
	}

	/* Config child, sent message to parent when child process end */
	b_req_emg((W)wRet, MM_ABORT | MM_EXIT | MM_TERM);

	/* Return the created console port number */
	*port = arg[0];
	*pid = wRet;
	return 0;
}

