// -*- C++ -*-
/*!
 * @file Properties.h
 * @brief Property list class (derived from Java Properties)
 * @date $Date: 2007-12-31 03:08:06 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: Properties.h 1703 2010-01-20 01:12:01Z fsi-katami $
 *
 */

#ifndef COIL_PROPERTIES_H
#define COIL_PROPERTIES_H


#include <string>
#include <vector>
#include <map>
#include <climits>


/*!
 * @if jp
 * @namespace coil 
 *
 * @brief Common Object Interface Layer
 *
 *
 * @else
 * @namespace coil
 *
 * @brief Common Object Interface Layer
 *
 *
 * @endif
 */
namespace coil
{
  /*!
   * @if jp
   *
   * @class Properties
   * @brief vpeBZbg\NX
   *
   * Properties NX́Asς̃vpeBZbg\B Properties Xg[
   * ɕۊǂAXg[烍[h肷邱ƂłB
   * vpeBXg̊eL[AтɑΉl͕ƂȂĂB
   *
   * vpeBXgɂ́ÁuftHglvƂĕʂ̃vpeBXg
   * ƂłB̃vpeBXgŃvpeBL[ȂƁA
   * 2Ԗڂ̃vpeBXgB 
   *
   * vpeB̎擾ɂ getProperty() AvpeB̃Zbgɂ setProperty() 
   * \bhgp邱ƂB
   *
   * vpeBXg[ɕۑƂA܂̓Xg[烍[hƂ
   * ɁAISO 8859-1 GR[fBOgpB̃GR[fBO
   * ڕ\łȂ́AƂłȂB
   *
   * ̃NX́AJava  Properties NX (java.util.Properties) Ƃقړl
   * \bhB܂Ao͂t@C Java  Properties NX
   * o͂̂ƌ݊邪AUnicode ܂ނ͈̂ƂłȂB
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class Properties
   * @brief Class represents a set of properties
   *
   * The Properties class represents a persistent set of properties. The
   * Properties can be saved to a stream or loaded from a stream. Each key and
   * its corresponding value in the property list is a string. 
   *
   * A property list can contain another property list as its "defaults". This
   * second property list is searched if the property key is not found in the
   * original property list. 
   *
   * It is recommended to use these method; setProperty() to get properties,
   * setProperty() to set properties. 
   *
   * When propertis are stored in a stream or when they are loaded from
   * the stream, the ISO 8859-1 character encoding is used.
   * Characters that cannot be directly represented in this encoding can be used.
   *
   * This class has almost same methods of Java's Properties class
   * (java.util.Properties). Also, input and output files are compatible with 
   * outputs of Java's Properties class, but Unicode encoded property file
   * is not be supported.
   *
   * @endif
   */
  class Properties
  {
  public:
    /*!
     * @if jp
     *
     * @brief RXgN^(rootm[ĥݍ쐬)
     *
     * key  value ݂̂^ Property ̃[gm[h쐬B
     * l͑SăftHglƂĐݒ肳B
     *
     * @param key vpeB̃L[(ftHgl:"")
     * @param value vpeB̒l(ftHgl:"")
     * 
     * @else
     *
     * @brief Constructor(Create only root node)
     *
     * Create a root node of Property with root's key and value.
     * All values are set as default value.
     *
     * @param key Properties's keys(The default values:"")
     * @param value Properties's values(The default values:"")
     * 
     * @endif
     */
    Properties(const char* key = "", const char* value = "");
    
    /*!
     * @if jp
     *
     * @brief RXgN^(mapŃftHgl^)
     *
     * std::string  std::map ftHglɂ Properties 쐬B
     * l͑SăftHglƂĐݒ肳B
     * 
     * @param defaults ftHglƂĎw肳map
     * 
     * @else
     *
     * @brief Constructor(Give the default value with map)
     *
     * Create Properties with default value of std::string map.
     * All values are set as default value.
     * 
     * @param defaults map that is spcified as the default value
     * 
     * @endif
     */
    Properties(std::map<std::string, std::string>& defaults);
    
    /*!
     * @if jp
     *
     * @brief RXgN^(char*[] ŃftHgl^)
     *
     * w肳ꂽftHgl̃vpeBXg쐬B
     * l͑SăftHglƂĐݒ肳B
     * ftHgl char* ̔zɂ^Akey  value ̑΂ɂȂ
     * AXg̏I[͔z̐\ num A󕶎 key ŗ^
     * Ȃ΂ȂȂB
     * ȉɗB
     *
     * <pre>
     * const char* defaults = {
     *     "key1", "value1",
     *     "key2", "value2",
     *     "key3", "value3",
     *     "key4", "value4",
     *     "key5", "value5",
     *     "" };
     * Properties p(defaults);
     * // 
     * Properties p(defaults, 10);
     * </pre>
     * 
     * @param defaults ftHglw肷z
     * @param num ftHglݒ肷vf(ftHgl:LONG_MAX)
     * 
     * @else
     *
     * @brief Constructor(Give the default value with char*[])
     *
     * Creates an empty property list with the specified defaults.
     * All values are set as the default values.
     * The default values are given by array of char*, which should be pairs
     * of "key" and "value". The end of list is specified by argument "num"
     * which specifies number of array, or null character of key.
     * The following is an example.
     *
     * <pre>
     * const char* defaults = {
     *     "key1", "value1",
     *     "key2", "value2",
     *     "key3", "value3",
     *     "key4", "value4",
     *     "key5", "value5",
     *     "" };
     * Properties p(defaults);
     * // or
     * Properties p(defaults, 10);
     * </pre>
     * 
     * @param defaults Array that specifies the default values
     * @param num Number of elements that specifies the default value
     *            (The default value:LONG_MAX)
     * 
     * @endif
     */
    Properties(const char* defaults[], long num = LONG_MAX);
    
    /*!
     * @if jp
     * @brief Rs[RXgN^
     *
     * ɗ^ꂽ Properties ̃L[AlуftHgl
     * SĂ̂܂܃Rs[B
     *
     * @else
     *
     * @brief Copy Constructor
     *
     * All of given Properties's keys, values and default values 
     * are copied to new Properties.
     * 
     * @endif
     */
    Properties(const Properties& prop);
    
    /*!
     * @if jp
     * @brief Zq
     *
     * Ӓl Properties ̃L[AlуftHgl͑Sč폜A
     * EӒl Properties ̃L[AlуftHglSĂ̂܂
     * Rs[B
     *
     * @else
     * @brief Assignment operator
     *
     * All Properties's keys, values and default values of left side
     * are deleted, all Properties's keys, values and default values of
     * right side are copied to new Properties.
     *
     * @endif
     */
    Properties& operator=(const Properties& prop);
    
    /*!
     * @if jp
     *
     * @brief fXgN^
     *
     * @else
     *
     * @brief Destructor
     *
     * @endif
     */
    virtual ~Properties(void);
    
    //============================================================
    // public functions
    //============================================================
    
    /*!
     * @if jp
     * @brief Name ̎擾
     *
     * vpeB̖̂擾B
     *
     * @return vpeB
     *
     * @else
     * @brief Get Names
     *
     * Get Properties's names.
     *
     * @return Properties's names
     *
     * @endif
     */
    inline const char* getName(void) const          {return name.c_str();}
    
    /*!
     * @if jp
     * @brief l̎擾
     *
     * vpeB̒l擾B
     *
     * @return vpeBl
     *
     * @else
     * @brief Get values
     *
     * Get Properties's values.
     *
     * @return Properties's values
     *
     * @endif
     */
    inline const char* getValue(void) const         {return value.c_str();}
    
    /*!
     * @if jp
     * @brief ftHgl̎擾
     *
     * vpeB̃ftHgl擾B
     *
     * @return vpeBftHgl
     *
     * @else
     * @brief Get default values
     *
     * Get Properties's default values.
     *
     * @return Properties's default values
     *
     * @endif
     */
    inline const char* getDefaultValue(void) const {return default_value.c_str();}
    
    /*!
     * @if jp
     * @brief qvf̎擾
     *
     * vpeB̎qvf擾B
     *
     * @return qvf
     *
     * @else
     * @brief Get elements of leaf
     *
     * Get Properties's elements of leaf.
     *
     * @return Elements of leaf
     *
     * @endif
     */
    inline const std::vector<Properties*>& getLeaf(void) const {return leaf;}
    
    /*!
     * @if jp
     * @brief [gvf̎擾
     *
     * vpeB̃[gvf擾B
     *
     * @return [gvf
     *
     * @else
     * @brief Get root element
     *
     * Get Properties's root element.
     *
     * @return Root element
     *
     * @endif
     */
    inline const Properties* getRoot(void) const    {return root;}
    
    /*!
     * @if jp
     *
     * @brief w肳ꂽL[vpeBAvpeBXgT
     *
     * w肳ꂽL[vpeBAvpeBXgTB
     * ̃L[vpeBXgɂȂƁAftHg̃vpeBXgA
     * ɂ̃ftHglJԂׂB
     * ̃vpeBȂꍇ́Anull ԂB 
     *
     * @param key vpeBL[
     *
     * @return w肳ꂽL[l̃vpeBXg̒l
     *
     * @else
     *
     * @brief Search for the property with the specified key in this property
     *
     * Search for the property with the specified key in this property list.
     * If the key is not found in this property list, the default property list,
     * and its defaults, recursively, are then checked. The method returns null
     * if the property is not found. 
     *
     * @param key The property key.
     *
     * @return The value in this property list with the specified key value.
     *
     * @endif
     */
    const std::string& getProperty(const std::string& key) const;
    
    /*!
     * @if jp
     *
     * @brief w肳ꂽL[vpeBAvpeBXgT
     *
     * w肳ꂽL[vpeBAvpeBXgTB
     * ̃L[vpeBXgɂȂꍇ́AftHgl̈ԂB 
     *
     * @param key vpeBL[
     * @param def ftHgl
     *
     * @return w肳ꂽL[l̃vpeBXg̒l
     *
     * @else
     *
     * @brief Search for the property with the specified key in property list
     *
     * Search for the property with the specified key in this property list.
     * The method returns the default value argument if the property is not 
     * found.
     *
     * @param key The property key
     * @param def The  default value. 
     *
     * @return The value in this property list with the specified key value.
     *
     * @endif
     */
    const std::string& getProperty(const std::string& key,
				   const std::string& def) const;
    
    /*!
     * @if jp
     *
     * @brief w肳ꂽL[vpeBAvpeBXgT
     *
     * w肳ꂽL[vpeBԂB
     * ̃L[vpeBXgɂȂ΃ftHglԂB
     * ɌȂ΁A󕶎ԂB
     *
     * @param key vpeBL[
     *
     * @return w肳ꂽL[l̃vpeBXg̒l
     *
     * @else
     *
     * @brief Search for the property with the specified key in property list
     *
     * Search for the property with the specified key in this property list.
     * If the key is not found in this property list, the default property list,
     * and its defaults, recursively, are then checked. The method returns 
     * empty string if the property is not found. 
     *
     * @param key The property key
     *
     * @return The value in this property list with the specified key value.
     *
     * @endif
     */
    const std::string& operator[](const std::string& key) const;
    
    /*!
     * @if jp
     *
     * @brief w肳ꂽL[vpeBAvpeBXgT
     *
     * w肳ꂽL[vpeBԂB
     * ̃L[̒lvpeBXgɂȂ΃ftHglԂB
     * ɌȂ΁A󕶎ԂB
     * ӒlɂȂꍇɁAlvfȂƂ͗^ꂽL[
     * ɑΉvpeBɉEӒl}B
     *
     * @param key vpeBL[
     *
     * @return w肳ꂽL[l̃vpeBXg̒l
     *
     * @else
     *
     * @brief Search for the property with the specified key in property list
     *
     * Search for the property with the specified key in this property list.
     * If the key is not found in this property list, the default property list,
     * and its defaults, recursively, are then checked. The method returns 
     * empty string if the property is not found.
     * If there is no element with the same value in the left value, insert the
     * right value in corresponding property.
     *
     * @param key The property key
     *
     * @return The value in this property list with the specified key value.
     *
     * @endif
     */
    std::string& operator[](const std::string& key);
    
    /*!
     * @if jp
     * @brief w肳ꂽL[ɑ΂ăftHgl擾
     *
     * w肳ꂽL[vpeB̃ftHglԂB
     * w肳ꂽL[vpeB݂Ȃꍇɂ͋󕶎ԂB
     *
     * @param key vpeBL[
     *
     * @return w肳ꂽL[lvpeB̃ftHgl
     *
     * @else
     * @brief Get the default values with specified key.
     *
     * Return the default values with specified key.
     * If the property with specified key does not exist, the method returns
     * empty string.
     *
     * @param key The property key
     *
     * @return The value in this property list with the specified key value.
     *
     * @endif
     */
    const std::string& getDefault(const std::string& key) const;
    
    /*!
     * @if jp
     *
     * @brief Properties  value  key ɂēo^
     *
     * Properties  value  key ɂēo^B
     * ł key ɑ΂lĂꍇA߂lɌÂlԂB
     *
     * @param key vpeBXgɔzuL[
     * @param value key ɑΉl 
     *
     * @return vpeBXg̎w肳ꂽL[̑O̒lBꂪȂꍇ null
     *
     * @else
     *
     * @brief Set a value associated with key in the property list
     *
     * This method sets the "value" associated with "key" in the property list.
     * If the property list has a value of "key", old value is returned.
     *
     * @param key The key to be placed into this property list.
     * @param value The value corresponding to key. 
     *
     * @return The previous value of the specified key in this property list,
     *         or null if it did not have one.
     *
     *@endif
     */
    std::string setProperty(const std::string& key, const std::string& value);
    
    /*!
     * @if jp
     * @brief ftHglo^
     *
     * key Ŏw肳vfɃftHglo^B
     *
     * @param key ftHglo^vpeB̃L[
     * @param value o^ftHgl
     *
     * @return w肳ꂽftHgl
     *
     * @else
     * @brief Set a default value associated with key in the property list
     *
     * Set the default value to element specified by "key".
     *
     * @param key Property's key to set the default value
     * @param value The default value that is set
     *
     * @return The specified default value
     *
     * @endif
     */
    std::string setDefault(const std::string& key, const std::string& value);
    
    /*!
     * @if jp
     * @brief Properties ɃftHgl܂Ƃ߂ēo^
     *
     * zŎw肳ꂽvfɃftHgl܂Ƃ߂ēo^B
     * ftHgl char* ̔zɂ^Akey  value ̑΂ɂȂ
     * AXg̏I[͔z̐\ num A󕶎 key ŗ^
     * Ȃ΂ȂȂB
     * 
     * @param defaults ftHglw肷z
     * @param num ftHglݒ肷vf(ftHgl:LONG_MAX)
     * 
     * @else
     * @brief Set a default value together in the property list
     *
     * Set the default value to the element specified by array together in the
     * property list.
     * The default values are given by array of char*, which should be pairs
     * of "key" and "value". The end of list is specified by argument "num",
     * which specifies number of array or null character of key.
     * 
     * @param defaults Array that specifies the default values
     * @param num Number of elements that specifies the default value
     *            (Default value:LONG_MAX)
     * 
     * @endif
     */
    void setDefaults(const char* defaults[], long num = LONG_MAX);
    
    //============================================================
    // load and save functions
    //============================================================
    /*!
     * @if jp
     *
     * @brief w肳ꂽo̓Xg[ɁAvpeBXgo͂
     *
     * w肳ꂽo̓Xg[ɁAvpeBXgo͂B
     * ̃\bh͎ɃfobOɗpB
     *
     * @param out o̓Xg[
     *
     * @else
     *
     * @brief Prints this property list out to the specified output stream
     *
     * Prints this property list out to the specified output stream.
     * This method is useful for debugging.
     *
     * @param out Output stream.
     *
     * @endif
     */
    void list(std::ostream& out);
    
    /*!
     * @if jp
     *
     * @brief ̓Xg[L[Ɨvf΂ɂȂvpeBXgǂݍ
     *
     * ̓Xg[L[Ɨvf΂ɂȂvpeBXgǂݍށB
     * Xg[́AISO 8859-1 GR[fBOgpĂƂ݂ȂB
     * evpeB́A̓Xg[ɍsPʂœo^Ă̂Ƃ݂ȂA
     * es͍s؂蕶 (\\nA\\rA܂ \\r\\n) ŏIB
     * ̓Xg[ǂݍ񂾍śA̓Xg[Ńt@C̏I
     * B܂ŏB
     *
     * 󔒕̍sA܂͍ŏ̔󔒕 ASCII  # ܂ ! ł
     * s͖B܂A# ܂ ! ̓RgsB
     *
     * 󔒍s܂̓RgsȊÔׂĂ̍śAe[uɒǉvpeB
     * LqBAs̏I肪 \ ̏ꍇ́A̍sΌpsƂ
     *  (LQ)B L[́Aŏ̔󔒕Aŏ ASCII 
     * =A:A܂͋󔒕̒O܂ł́AŝׂĂ̕\B
     *
     * L[̏ÍAO \ t邱ƂɂL[Ɋ܂߂邱Ƃ
     * łBL[̌̋󔒂ׂ͂ăXLbvB
     * L[̌̍ŏ̔󔒕 = ܂ : łꍇ́ÃL[
     * ÂƂ̋󔒕ׂăXLbvB
     * ŝȊOׂ͂̕āA֘Avf̈ꕔƂȂB
     * vfł́AASCII GXP[vV[PX \\tA\\nA\\rA\\\\A\\"A
     * \\'A\\ (~LƃXy[X)A \\uxxxx ͔FAPƂ̕ɕϊ
     * B
     * ܂As̍Ō̕ \ łꍇ́A̍s݂͌̍špƂ
     * B̏ꍇA\ ƍs؂蕶jAps̐擪ɋ󔒂
     * ΂ׂĔjAvf̈ꕔɂ͂ȂȂB 
     *
     * Ƃ΁A 3 s͂ꂼL[ Truth Ɗ֘Avfl Beauty \B
     * 
     * Truth = Beauty <BR>
     * Truth:Beauty <BR>
     * Truth\\t\\t\\t:Beauty <BR>
     *
     * ܂A 3 s 1 ̃vpeB\B 
     *
     * fruits\\t\\t\\t\\tapple, banana, pear, \ <BR>
     *                                  cantaloupe, watermelon, \ <BR>
     *                                  kiwi, mango <BR>
     * L[ fruits ŁA̗vfɊ֘AtB 
     * "apple, banana, pear, cantaloupe, watermelon, kiwi, mango"
     * ŏIIȌʂŃR}̂ƂɕKXy[X\悤ɁA
     * e \ ̑OɃXy[XBs̏I \ ƁAps̐擪ɂ
     * 󔒂͔jA̕ɒuȂB 
     * ܂A 3 Ԗڂ̗ł́AL[ cheeses ŁA֘Avf̕
     * ł邱Ƃ\B 
     *
     * cheeses <BR>
     * L[́Acheeses ŁA֘Avf͋̕ł邱Ƃw肵ĂB 
     *
     * @param inStream ̓Xg[ 
     *
     * @else
     *
     * @brief Loads property list that consists of key:value from input stream
     *
     * Reads a property list (key and element pairs) from the input stream.
     * The stream is assumed to be using the ISO 8859-1 character encoding.
     * Each property is assumed to be registered in the input stream by each
     * line, and each line terminator is should be a line break characters
     * (\\n or \\r or \\r\\n).
     * Lines are read from the input stream until end of file is reached. 
     *
     * A line that contains only white space characters or a line that its
     * first non-white space character is an ASCII '#' or '!' is ignored.
     * In a word, '#' or '!' represents comment lines.
     *
     * All lines except the blank line or comment line is described the property
     * that added to the table. However, if the line terminator is '\' and the 
     * next line continues, it is treated as a continuation line (See below).
     * The key is composed of all characters.
     * All of these key termination characters in the line starting with the 
     * first non-white space character and up to, but not including, the first
     * unescaped '=', ':', or white space character other than a line 
     * terminator. 
     * 
     * Line terminator characters can be included using \ escape sequences.
     * Any white space after the key is skipped.
     * If the first non-white space character after the key is '=' or ':',
     * then it is ignored and any white space characters after it are also 
     * skipped.
     * All remaining characters on the line become part of the associated element
     * string.
     * In element string, ASCII escape sequence such as \\t and \\n and \\r
     * and \\\\ and \\" and \\' and \\ (backslash character and space) 
     * and \\uxxxx have affect and they will be converted into a single 
     * character.
     * Also, if termination character in the line is \, the next line will be  
     * treated as continuing. In that case, \ and break character will be 
     * destroyed, and also its first space character will be destroyed, 
     * so these characters on the line will not become part of the element 
     * string.
     *
     * As an example, each of the following three lines specifies the key 
     * "Truth" and the associated element value "Beauty": 
     * 
     * Truth = Beauty <BR>
     * Truth:Beauty <BR>
     * Truth\\t\\t\\t:Beauty <BR>
     *
     * As another example, the following three lines specify a single 
     * property: 
     *
     * fruits\\t\\t\\t\\tapple, banana, pear, \ <BR>
     *                                  cantaloupe, watermelon, \ <BR>
     *                                  kiwi, mango <BR>
     * The key is "fruits" and the associated element is: 
     * "apple, banana, pear, cantaloupe, watermelon, kiwi, mango".
     * Note that a space appears before each \ so that a space will
     * each comma in the final result; the \, line terminator, and leading white
     * space on the continuation line are merely discarded and are not replaced
     * by one or more other characters. 
     * As a third example, the line: 
     *
     * cheeses <BR>
     * specifies that the key is "cheeses" and the associated element is the
     * empty string "".
     *
     * @param inStream the input stream.
     *
     * @endif
     */
    void load(std::istream& inStream);
    
    /*!
     * @if jp
     *
     * @brief vpeBXgw肳ꂽXg[ɕۑ
     *
     * Ă܂BvpeBXg̕ۑ@ƂẮA
     * store(ostream out, string header) \bh̎gpB
     * ̃\bh Java Properties Ƃ̌݊̂߂ɒ`ĂB
     *
     * @param out o̓Xg[
     * @param header vpeBXg̋Lq 
     *
     * @else
     *
     * @brief Save the property list to the specified stream
     *
     * It is not recommended. To save the property list,
     * the store(ostream out, string header) method is recommended.
     * This method is defined for compatibility of Java Properties.
     *
     * @param out The output stream
     * @param header A description of the property list
     *
     * @endif
     */
    void save(std::ostream& out, const std::string& header);
    
    /*!
     * @if jp
     *
     * @brief vpeBXgo̓Xg[֕ۑ
     *
     * Properties e[ũvpeBXg (L[Ɨvf̃yA) Aload
     * \bhg Properties e[uɃ[ĥɓK؂ȃtH[}bg
     * o̓Xg[ɏށB 
     *
     * Properties e[ũvpeBXg (L[Ɨvf̃yA) Aload
     * \bhg Properties e[uɃ[ĥɓK؂ȃtH[}bg
     * o̓Xg[ɏށBXg[́AISO 8859-1 
     * GR[fBOgpď܂B 
     * Properties e[u (݂ꍇ) ̃ftHge[u
     * vpeB́Ã\bhɂĂ͏܂ȂB 
     *
     * header  null łȂꍇ́AASCII  #Aheader ̕A
     * эs؂蕶ŏɏo̓Xg[ɏ܂܂B̂߁A
     * header ͎ʃRgƂĎgƂłB 
     *
     * ɁAASCII  #A݂̓ (Date  toString \bhɂ
     * ݎ̂Ɠl)A Writer ɂĐs؂
     * ȂRgs܂B 
     *
     * āA Properties e[ûׂẴGg 1 soB
     * eGg̃L[AASCII =A֘Avf񂪏܂B
     * vf̊éAGXP[vV[PXƂĕ`悷Kv邩
     * ǂmFBASCII  \A^uAsAѕA͂ꂼ \\\\A
     * \\tA\\nA \\r Ƃď܂B\\u0020 菬
     * \\u007E 傫́AΉ 16 il xxxx g \\uxxxx Ƃ
     * ܂Bߍ݋󔒕ł㏑󔒕łȂs󔒕́A
     * O \ tď܂BL[ƒl̕ #A!A=A : ́A
     * K[h悤ɁAOɃXbVtď܂B 
     *
     * Gg܂ꂽƂŁAo̓Xg[tbVB
     * o̓Xg[͂̃\bh畜AƂJ܂܂ƂȂB 
     *
     * @param out o̓Xg[
     * @param header vpeBXg̋Lq 
     *
     * @else
     *
     * @brief Stores property list to the output stream
     *
     * Write this property list (key and element pairs) in this Properties 
     * table to the output stream in a format suitable for loading into a 
     * Properties table using the load method. The stream is written using the 
     * ISO 8859-1 character encoding. 
     * Properties from the defaults table of this Properties table (if any) are
     * not written out by this method. 
     *
     * If the header argument is not null, then an ASCII # character, the
     * comments string, and a line separator are first written to the output
     * stream. Thus, the header can serve as an identifying comment. 
     *
     * Next, a comment line is always written, consisting of an ASCII #
     * character, the current date and time (as if produced by the toString
     * method of Date for the current time), and a line separator as generated
     * by the Writer. 
     *
     * Then every entry in this Properties table is written out, one per line.
     * For each entry the key string is written, then an ASCII =, then the
     * associated element string. Each character of the key and element strings
     * is examined to see whether it should be rendered as an escape sequence.
     * The ASCII characters \, tab, form feed, newline, and carriage return are
     * written as \\\\, \\t, \\f \\n, and \\r, respectively. Characters less than
     * \\u0020 and characters greater than \\u007E are written as \\uxxxx for the
     * appropriate hexadecimal value xxxx. For the key, all space characters are
     * written with a preceding \ character. For the element, leading space
     * characters, but not embedded or trailing space characters, are written
     * with a preceding \ character. The key and element characters #, !, =, and
     * : are written with a preceding backslash to ensure that they are properly
     * loaded. 
     *
     * After the entries have been written, the output stream is flushed. The
     * output stream remains open after this method returns. 
     *
     * @param out An output stream.
     * @param header The description of the property list.
     *
     * @endif
     */
    void store(std::ostream& out, const std::string& header);
    
    //============================================================
    // other util functions
    //============================================================
    /*!
     * @if jp
     *
     * @brief vpeB̃L[̃Xg vector ŕԂ
     *
     * CvpeBXgɓÕL[Ȃꍇ́AftHg
     * vpeBXgɂʂ̃L[܂ށÃvpeBXgɂ邷ׂ
     * ̃L[̃XgԂB 
     *
     * @return vpeBXgɂ邷ׂẴL[̃XgB
     *         ftHg̃vpeBXgɂL[܂
     *
     * @else
     *
     * @brief Return an vector of all the keys in this property
     *
     * Returns an enumeration of all the keys in this property list, including
     * distinct keys in the default property list if a key of the same name has
     * not already been found from the main properties list.
     *
     * @return A vector of all the keys in this property list, including the
     *         keys in the default property list.
     *
     * @endif
     */
    std::vector<std::string> propertyNames(void) const;
    
    /*!
     * @if jp
     * @brief vpeB̐擾
     *
     * ݒς݂̃vpeB擾B
     *
     * @return vpeB
     *
     * @else
     * @brief Get the number of Properties
     *
     * Get the number of Properties that has already set.
     *
     * @return Number of Properties
     *
     * @endif
     */
    int size(void) const;
    
    /*!
     * @if jp
     * @brief m[h擾
     *
     * w肵L[m[h擾B
     * ݂ȂL[Aы󕶎̏ꍇ 0 ԂB
     *
     * @param key 擾Ώۃm[h̃L[
     *
     * @return Ώۃm[h
     *
     * @else
     * @brief Get node of properties
     *
     * Get node with specified key.
     *
     * @param key Target node key for getting
     *
     * @return Target node
     *
     * @endif
     */
    Properties* const findNode(const std::string& key) const;
    /*!
     * @if jp
     * @brief m[h擾
     *
     * w肵L[m[h擾B
     * ݂ȂL[Aы󕶎̏ꍇ 0 ԂB
     *
     * @param key 擾Ώۃm[h̃L[
     *
     * @return Ώۃm[h
     *
     * @else
     * @brief Get node of properties
     *
     * Get node with specified key.
     *
     * @param key Target node key for getting
     *
     * @return Target node
     *
     * @endif
     */
    Properties& getNode(const std::string& key);
    
    /*!
     * @if jp
     * @brief VKm[h𐶐
     *
     * w肵L[VKm[h𐶐B
     * ɓL[m[ho^ς݂̏ꍇɂ̓G[ԂB
     *
     * @param key VKm[h̃L[
     *
     * @return VKm[h
     *         w肵L[m[hɑ݂ꍇɂfalse
     *
     * @else
     * @brief Create newly node of Properties
     *
     * Create nowly node with specified key.
     * If the node with the same key has been registered, error will be returned.
     *
     * @param key Newly node key
     *
     * @return Newly node creation result.
     *         false will be returned if the node with specified key has already
     *         existed.
     *
     * @endif
     */
    bool createNode(const std::string& key);
    
    /*!
     * @if jp
     * @brief m[h폜
     *
     * w肵̂vpeB폜B
     * 폜vpeBԂB
     *
     * @param leaf_name 폜ΏۃvpeB
     *
     * @return 폜vpeB
     *
     * @else
     * @brief Remove node of Properties
     *
     * Remove properties with specified name.
     * Properties that were deleted will be returned.
     *
     * @param leaf_name Target property's name for the delete
     *
     * @return Deleted properties
     *
     * @endif
     */
    Properties* removeNode(const char* leaf_name);
    
    /*!
     * @if jp
     * @brief qm[hkey邩ǂ
     *
     * w肵L[qm[h݂邩ǂmFB
     * ݂ꍇAqm[hԂB
     *
     * @param key mFΏۂ̃L[
     *
     * @return qm[h
     *
     * @else
     * @brief Check whether key exists in the children
     *
     * Check whether the children with specified key exist.
     * If the children exist, they will be returned.
     *
     * @param key Check key
     *
     * @return Children node
     *
     * @endif
     */
    Properties* hasKey(const char* key) const;
    
    /*!
     * @if jp
     * @brief qm[hSč폜
     * @else
     * @brief Clear the children
     * @endif
     */
    void clear(void);
    
    /*!
     * @if jp
     * @brief Property}[W
     *
     * ݂̃vpeBɐݒ肵vpeB}[WB
     *
     * @param prop }[WvpeB
     *
     * @return vpeB}[W
     *
     * @else
     * @brief Merge properties
     *
     * Merge properties that have set to the current properties.
     *
     * @param prop Properties for the merge
     *
     * @return Property merge result
     *
     * @endif
     */
    Properties& operator<<(const Properties& prop);
    
  protected:
    /*!
     * @if jp
     * @brief L[ƒl̃yAɕ
     *
     * ^ꂽAݒ肳ꂽf~^ŃL[ƒl̃yAɕB
     * ܂ŏɗ^ꂽ':''='܂܂邩A
     * ǂ炩̕܂܂Ăꍇɂ͂f~^ƂĎgpB
     * Ƃ܂܂ĂȂꍇɂ́A' '(Xy[X)pĕ݂B
     * SẴf~^₪܂܂ĂȂꍇɂ́A^ꂽL[Ƃ
     * ݒ肵Alɋ̕ݒ肷B
     * ǂ̃f~^ɂĂGXP[vĂ(O'\'ݒ肳Ă)
     * ꍇɂ́Af~^ƂĎgpȂB
     *
     * @param str Ώە
     * @param key ʃL[
     * @param value ʒl
     *
     * @else
     * @brief Split the string into a pair of the key and the value.
     *
     * Split the given string into a pair of the key and the value with
     * the set delimiter.
     * First, search whether the fist given string includes ':' or '=', and 
     * if either character is included, it is used as delimiter.
     * If neither is included, try to divide it with ' '(space).
     * When all delimiter candidates are not included, set the given string
     * as key then set empty string to the its value.
     * If any delimiter candidate is escaped ('\' is set right before it),
     * this method does not use as delimiter.
     *
     * @param str Target string for the division
     * @param key Division result key
     * @param value Division result value
     *
     * @endif
     */
    static void splitKeyValue(const std::string& str, std::string& key,
			      std::string& value);
    
    /*!
     * @if jp
     * @brief 𕪊
     *
     * ^ꂽA^ꂽf~^ŕB
     * ^ꂽ񂪋̏ꍇ́AG[ԂB
     * ^ꂽf~^GXP[vĂ(O'\'ݒ肳Ă)ꍇ
     * ɂ́Af~^ƂĎgpȂB
     *
     * @param str Ώە
     * @param delim f~^
     * @param value ʒlXg
     *
     * @return 
     *
     * @else
     * @brief Split the string
     *
     * Divide given string with given delimiter.
     * If the given string is empty, error will be returned.
     * When the given delimiter is escaped ('\' is set right before it)
     * this method does not use as delimiter.
     *
     * @param str Target string for the division
     * @param delim Delimiter
     * @param value Division result list
     *
     * @return Division result
     *
     * @endif
     */
    static bool split(const std::string& str, const char delim,
		      std::vector<std::string>& value);
    
    /*!
     * @if jp
     * @brief vpeB擾
     *
     * L[XgŎw肳ꂽvpeB擾B
     * L[Xgł́Aw肷L[̃vpeBł̊Kw֌WXg`ŕ\
     * B
     * w肵L[XgɊYvpeB݂ȂꍇNULLԂB
     *
     * @param keys 擾ΏۃvpeB̃L[̃Xg\
     * @param index L[Xg̊Kw
     * @param curr ΏۃvpeB
     *
     * @return ΏۃvpeB
     *
     * @else
     * @brief Get properties
     *
     * Get properties specified by key list.
     * In the key list, hierarchical relation in the properties is represented
     * by a list format.
     * If properties corresponding to specified key list do not exist,
     * null will be returned.
     *
     * @param keys Target properties's key list representation for getting
     * @param index Number of hierarchy of key list
     * @param curr Target properties for the search
     *
     * @return Target properties for the search
     *
     * @endif
     */
    static Properties* _getNode(std::vector<std::string>& keys,
				std::vector<Properties*>::size_type index,
				const Properties* curr);
    
    /*!
     * @if jp
     * @brief vpeB̖̃Xg擾
     *
     * vpeB̖̂'.'؂ŕ\Xg擾B
     *
     * @param names vpeB̖̃Xg
     * @param curr_name ݂̃vpeB
     * @param curr ΏۃvpeB
     *
     * @else
     * @brief Get property name list
     *
     * Get a list expressed by separating each property name with '.'.
     *
     * @param names Name list of property
     * @param curr_name Current property's name
     * @param curr Target properties
     *
     * @endif
     */
    static void _propertiyNames(std::vector<std::string>& names,
				std::string curr_name,
				const Properties* curr);
    
    /*!
     * @if jp
     * @brief vpeB̖̃Xgۑ
     *
     * vpeB̖̂'.'؂ŕ\XgۑB
     *
     * @param out vpeB̖̃Xgۑ̏o̓Xg[
     * @param curr_name ݂̃vpeB
     * @param curr ΏۃvpeB
     *
     * @else
     * @brief Store the property name list
     *
     * Store a list expressed by separating each property name with '.'.
     *
     * @param out Output stream of property's name list of save destination.
     * @param curr_name Current property's name
     * @param curr Target properties
     *
     * @endif
     */
    static void _store(std::ostream& out, std::string curr_name,
		       Properties* curr);
    
    /*!
     * @if jp
     * @brief vpeB̓eۑ
     *
     * vpeBɐݒ肳ꂽeۑB
     * ۑɂ̓vpeBKw̐[\tB
     * lݒ肳ĂȂvpeBɂẮAftHglo͂B
     *
     * @param out vpeBeۑ̏o̓Xg[
     * @param curr ΏۃvpeB
     * @param index ݂̃vpeBKw
     *
     * @else
     * @brief Save property's contents
     *
     * Save the contents that were set to the property.
     * The figure represented the depth of the property hierarchy is
     * added when saving.
     * If property with the value that is not set, the default value will
     * be output.
     *
     * @param out Output stream of property's contents of save destination.
     * @param curr Target property
     * @param index Current property hierarchy
     *
     * @endif
     */
    static std::ostream& _dump(std::ostream& out, const Properties& curr,
			       int index);
    
    /*!
     * @if jp
     * @brief Cfg𐶐
     *
     * w肳ꂽɏ]ĐCfgԂB
     * ԂCfǵAw萔~2̋󔒁B
     *
     * @param index Cfg̎w
     *
     * @return ꂽCfg
     *
     * @else
     * @brief Create indents
     *
     * Return indents according to specified figure.
     * Returned indents are specified figure x two blanks.
     *
     * @param index The specification of Number of indent
     *
     * @return Created indent
     *
     * @endif
     */
    static std::string indent(int index);
    
  private:
    std::string name;
    std::string value;
    std::string default_value;
    Properties* root;
    std::vector<Properties*> leaf;
    const std::string m_empty;
    friend std::ostream& operator<<(std::ostream& lhs, const Properties& rhs);
  };   // class Properties
};     // namespace coil  
#endif // COIL_PROPERTIES_H

