#include "front_end.hpp"
#include "code_data_parser.hpp"

#include "display_module.hpp"
#include "instruction.hpp"

#include <ctype.h>
#include <fstream>


class CodeDataEditorUndo
{
};
class CodeDataEditorUndoManager
{
    
public:
    
};

class CodeDataEditorImpl
{
    ByteSequence& m_cdflags;
    const std::vector<PByteSequence>& m_banks;
    const std::vector<Word>& m_origins;
    
    Byte m_bank_num;
    Word m_begin;
    Word m_end;
    Word m_temp;

    bool m_debug;
    std::string m_command;
    std::string m_current_command;
    std::string m_default_file_name;
    ByteSequence m_working_cdflags;
public:
    CodeDataEditorImpl(
            PByteSequence& cdflags,
            const std::vector<PByteSequence>& banks,
            const std::vector<Word>& origins,
            FilePath file_name)
            :m_cdflags(*cdflags),
             m_banks(banks),
             m_origins(origins),
             m_default_file_name(file_name)
    {
        setBank(0);
        m_debug=false;
    }
    void run()
    {
        while(true){
            if (mainLoop()==false) break;
        }
    }
private:
    MemoryAddress romAddr(Byte bank_num, Word cursor)
    {
        Word result = cursor % nes::BANK_SIZE
            + bank_num * nes::BANK_SIZE;
        assert(result <= m_cdflags.size());
        return result;
    }
    void errorAddressIsOutOfBank(Byte bank_num)
    {
        errorMes("error: current bank area is ", false);
        errorMes(HexForm(m_origins[bank_num])
                 .to_s(),false);
        errorMes("-",false);
        errorMes(HexForm(m_origins[bank_num]
                         +nes::BANK_SIZE-1).to_s());
    }
    bool isAddressInBank(Word addr, Byte bank_num)
    {
        if (m_origins[bank_num] <= addr
            && addr<m_origins[bank_num]+nes::BANK_SIZE){
            return true;
        }
        return false;
    }
    void setBank(Byte bank_num)
    {
        if (bank_num < m_banks.size()){
            m_bank_num = bank_num;
            setBegin(nextBlank(m_origins[bank_num]));
        } else {
            errorMes("Bank Number Max is ", false);
            errorMes(HexForm(m_banks.size())
                     .width(2).to_s());
            
        }
    }
    void setBegin(Word addr)
    {
        if (isAddressInBank(addr, m_bank_num)){
            m_begin = addr;
            Word block = nextBlock(m_begin);
            Word nonblank = nextNonBlank(m_begin);
            Word boundary = nes::BANK_SIZE * (1+m_bank_num)
                + m_origins[m_bank_num];

            Word end = block < nonblank ? block : nonblank;
            end = end < boundary ? end : boundary;
            setEnd(end);
        } else {
            errorAddressIsOutOfBank(m_bank_num);
        }
    }
    void setEnd(Word addr)
    {
        if (isAddressInBank(addr-1, m_bank_num)){
            if (m_begin < addr){
                m_end = addr;
            } else {
                errorMes("error: end address must be greater.");            }
        } else {
            errorAddressIsOutOfBank(m_bank_num);
        }
    }
    Word nextBlock(Word addr)
    {
        const ByteSequence& bank = *(m_banks[m_bank_num]);
        Word origin = m_origins[m_bank_num];
        unsigned begin_index = addr & 0x1FFF;
        
        Word result = addr;
        for (unsigned i=begin_index; i < bank.size(); ) {
            result = i+origin;
			Instruction instr(result, bank, i);
            ReferenceType type = ReferenceHelper::map(
                    instr.type());
			i+=instr.bytelength();
            result = i+origin;
            unsigned cdf_index = i+m_bank_num*nes::BANK_SIZE;
            if (type == reference::RETURN
                || type == reference::JUMP
                || m_cdflags[cdf_index] != cdp::BLANK) {
                break;  
            }
        }
        if (result > origin + nes::BANK_SIZE) {
            result = origin + nes::BANK_SIZE;
        }
        return result;
    }
    Word inputHexNumber()
    {
        std::string address_str;
        int len=0;
        
        for (unsigned i=0;
             i<m_command.size(); ++i){
            char ch = m_command[i];
            if (!HexForm::isHexNumChar(ch)){
                break;
            }
            ++len;
        }
        address_str=m_command.substr(0,len);
        m_current_command += address_str;
        m_command = m_command.substr(
                len,std::string::npos);
        return HexForm(address_str).to_i();
    }
    void printAssembly()
    {
        displayRawAssemblyRegeon(
                std::cout, m_banks, m_origins,
                m_bank_num, m_begin, m_end);
    }
    void viewFlags()
    {
        displayCodeDataRegeon(
                std::cout,m_cdflags,m_origins,
                m_bank_num,m_begin,m_end);
    }
    void setOrRegeon(Byte flag)
    {
        unsigned b=nes::BANK_SIZE*m_bank_num+(m_begin&0x1FFF);
        unsigned e=nes::BANK_SIZE*m_bank_num+(m_end&0x1FFF);
        for (unsigned i=b; i<e; ++i){
            assert(i<m_cdflags.size());
            m_cdflags[i] |= flag;
        }
    }
    Word nextBlank(Word addr)
    {
        unsigned idx=nes::BANK_SIZE*m_bank_num+(addr&0x1FFF);
        idx+=1;
        for (; idx<m_cdflags.size(); ++idx){
            if (m_cdflags[idx] == cdp::BLANK) {
                break;
            }
        }
        Word result = m_origins[m_bank_num] + (idx&0x1FFF);
        return result;
    }
    Word nextNonBlank(Word addr)
    {
        unsigned idx=nes::BANK_SIZE*m_bank_num+(addr&0x1FFF);
        idx+=1;
        for (; idx<m_cdflags.size(); ++idx){
            if (m_cdflags[idx] != cdp::BLANK) {
                break;
            }
        }
        Word result = m_origins[m_bank_num] + (idx&0x1FFF);
        return result;
    }
    FilePath readCommandFilePath()
    {
        std::string fileName="";
        if (m_command.size()==0){
            return m_default_file_name;
        } else {
            m_current_command += m_command;
            
            while (!m_command.empty()){
                if (m_command[0] != ' ') {
                    break;
                }
                m_command
                    = m_command.substr(1,std::string::npos);
            }
            m_default_file_name = m_command;
            m_command = "";
            return m_default_file_name;
        }
    }
    bool openCDLFile()
    {
        FilePath file_name = readCommandFilePath();
        bool open_success=false;
        std::ifstream is;
        if (!file_name.empty()) {
            is.open(file_name.to_s(),
                    std::ios::in | std::ios::binary);
            if (is.good()) { open_success = true; }
        }
        m_working_cdflags.clear();
        if (open_success){
            for (char ch; is.get(ch); ) {
                m_working_cdflags.push_back(Byte(ch));
            }
            if (m_working_cdflags.size() == m_cdflags.size()){
                return true;
            } else {
                errorMes("error: mismatch file size- '",
                         false);
                errorMes(file_name.to_s(),false);
                errorMes("'");
                return false;
            }
        } else {
            errorMes("error: invalid file - '",false);
            errorMes(file_name.to_s(),false);
            errorMes("'");
            return false;
        }
    }
    void loadCDLFile()
    {
        if (openCDLFile()){
            for (unsigned i=0; i<m_cdflags.size(); ++i){
                m_cdflags[i] = m_working_cdflags[i];
            }
            std::cout << "loaded file - '";
            std::cout << m_default_file_name;
            std::cout << "'" << std::endl;
        }
    }
    void mergeCDLFile()
    {
        if (openCDLFile()){
            for (unsigned i=0; i<m_cdflags.size();++i){
                m_cdflags[i] |= m_working_cdflags[i];
            }
            std::cout << "merged file - '";
            std::cout << m_default_file_name;
            std::cout << "'" << std::endl;
        }
    }
    void writeCDLFile()
    {
        FilePath file_name = readCommandFilePath();
        bool open_success=false;
        std::ofstream os;
        if (!file_name.empty()) {
            os.open(file_name.to_s(),std::ios::out);
            if (os.good()) { open_success = true; }
        }
        if (open_success){
            for (unsigned i=0; i<m_cdflags.size(); ++i) {
                os << m_cdflags[i];
            }
            std::cout << "write file - '";
            std::cout << m_default_file_name;
            std::cout << "'" << std::endl;
        } else {
            errorMes("error: invalid file - '",false);
            errorMes(file_name.to_s(),false);
            errorMes("'");
        }
    }
    bool mainLoop()
    {
        if (m_cdflags[romAddr(m_bank_num, m_begin)]
            != cdp::BLANK){
                
        }
        displayBankAddress(std::cout,
                           m_bank_num, m_begin);
        std::cout << "," << HexForm(m_end);
        std::cout << ": ";
        std::getline(std::cin, m_command);
            
        char ch;
        while (!m_command.empty()){
            m_current_command = "";
                
            ch = m_command[0];
            m_current_command += ch;
            m_command
                = m_command.substr(1,std::string::npos);
                
            switch (ch){
            case 'q': debugMes("Quit"); return false;
            case 'm': mergeCDLFile(); break;
            case 'l': loadCDLFile(); break;
            case 'w': writeCDLFile(); break;
                
            case 'p': printAssembly(); break;
            case '.': case 'v': viewFlags();break;
                
            case 'n': setBegin(nextBlank(m_begin));break;
            case 'C': setOrRegeon(cdp::CODE); break;
            case 'D': setOrRegeon(cdp::DATA); break;
            case 'c':
                setOrRegeon(cdp::CODE);
                setBegin(nextBlank(m_begin));
                printAssembly();
                viewFlags();
                break;
            case 'd':
                setOrRegeon(cdp::DATA);
                setBegin(nextBlank(m_begin));
                printAssembly();
                viewFlags();
                break;
            case 'f':
                std::cout << "Default file name: '"
                          << m_default_file_name
                          << "'" << std::endl;
                break;
            case '!':
                m_debug = ! m_debug;
                debugMes("Debug mode: On");
                break;
            case ')':
                setBank(m_temp);
                if (HexForm::isHexNumChar(m_command[0])){
                    m_temp = inputHexNumber();
                }
                break;
            case ',': setBegin(m_temp);
                if (HexForm::isHexNumChar(m_command[0])){
                    m_temp = inputHexNumber();
                }
                break;
            case ':': setEnd(m_temp); break;
            case '@': m_temp = m_begin; break;
            case '=':  
                std::cout << HexForm(m_temp)
                          << std::endl;
                break;
            case '$':
            case '(':
                m_temp = inputHexNumber();
                debugMes("Evaluated as $", false);
                debugMes(HexForm(m_temp).to_s(),
                         false);
                debugMes(".");
                break;
            case '?':case 'h': //show help
                break;
            case ' ': break;
            default:
                errorMes("Invalid command - '",false);
                errorMes(m_current_command,false);
                errorMes("'");
                break;
            }
            debugMes("Current command - '",false);
            debugMes(m_current_command,false);
            debugMes("'");
        }
        return true;
    }
    void errorMes(std::string mes, bool endline=true)
    {
        std::cerr << mes;
        if (endline) {
            std::cerr << std::endl;
        }
    }
    void debugMes(std::string mes, bool endline=true)
    {
        if (m_debug) {
            errorMes(mes,endline);
        }
    }
};


void CodeDataModule::edit(
        PByteSequence& cdflags,
        const std::vector<PByteSequence>& banks,
        const std::vector<Word>& origins,
        FilePath file_name)
{
    CodeDataEditorImpl impl(cdflags, banks,
                            origins, file_name);
    impl.run();
}
