﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using SCFiler2.FileSystem;
using System.Threading;
using SCFiler2.CommonDialog;
using SCFiler2.ItemInterface;

namespace SCFiler2 {
	/// <summary>
	/// DialogResultがOKのとき、プロパティSelectedItemで検索結果のFilerItemが取得できる。
	/// Filerが見つかったが直接実行したときは、ResultはNoneを返す。
	/// </summary>
	public partial class FileSearchForm : Form {
		Thread thread;

		public FileSearchForm() {
		}

		/// <summary>
		/// 検索の起点となるフォルダ。このフォルダの中を検索する
		/// </summary>
		private string baseFolder;
		/// <summary>
		/// 検索結果の中からユーザが選択したファイル
		/// </summary>
		private FilerItem selectedItem = null;
		/// <summary>
		/// 検索の結果見つかったファイル/フォルダの数
		/// </summary>
		private int foundItemNum;

		private FileSearchThread searchThread;
		private List<string> errors = new List<string>();

		/// <summary>
		/// 検索結果の中からユーザが選択したファイル
		/// </summary>
		public FilerItem SelectedItem {
			get { return selectedItem; }
		}

		/// <summary>
		/// 検索フォーム
		/// </summary>
		/// <param name="baseFolderForSearch">検索を行うフォルダ</param>
		public FileSearchForm(string baseFolderForSearch) {
			InitializeComponent();
			this.baseFolder = baseFolderForSearch;
			this.searchPathLabel.Text = "検索対象： " + baseFolderForSearch;
			this.foundItemList.SmallImageList = new ImageList();
			this.foundItemList.SmallImageList.Images.Add(Images.Folder);
		}

		/// <summary>
		/// 検索フォームの条件で検索の実行。実際の検索はsearch()で行う
		/// </summary>
		public void Execute() {
			//１度実行済みの場合のため、状態の初期化
			foundItemList.Items.Clear();
			errors.Clear();
			foundItemNum = 0;

			//検索用のスレッドの生成
			searchThread = new FileSearchThread(this);
			searchThread.SearchPath = this.baseFolder;
			searchThread.SearchString = searchStringTextbox.Text.ToLower(); //小文字にあわせて検索（大文字の方がいいかも）
			thread = new Thread(new ThreadStart(searchThread.Start));
			thread.IsBackground = true;
			thread.Priority = ThreadPriority.BelowNormal;
			thread.Start();
		}

		private void cancelButton_Click(object sender, EventArgs e) {
			if (this.searchThread != null) {
				this.searchThread.End();
			}
			Close();
		}

		private void searchButton_Click(object sender, EventArgs e) {
			this.Execute();
		}

		/// <summary>
		/// 見つかったアイテムの中から、選択アイテムが替わったときの処理。
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void foundItemList_SelectedIndexChanged(object sender, EventArgs e) {
			//選択アイテムが替わったときは、そのアイテムの情報をステータスバーに表示する

			if (foundItemList.SelectedItems.Count == 0 || foundItemNum ==0) { //Countだけで判断すると、「見つかりませんでした」の文字があるため。前半の条件は見ないでいいかも
				fullnameLabel.Text = "";
				sizeLabel.Text = "";
				updateDateLabel.Text = "";
				return;
			}

			FilerItem item = (FilerItem)foundItemList.SelectedItems[0].Tag;
			if (item.Type == ItemType.File) {
				fullnameLabel.Text = item.FullName;
				sizeLabel.Text = "サイズ：" +item.SizeDisplayString;
				updateDateLabel.Text = item.UpdateDateString;
			} else {
				fullnameLabel.Text = item.FullName;
				sizeLabel.Text = "";
				updateDateLabel.Text = item.UpdateDateString;
			}
		}


		private void foundItemList_KeyDown(object sender, KeyEventArgs e) {
			if (e.KeyData == Keys.Enter) {
				//Enterキーが押されたら、選択中のアイテムをダイアログの選択結果に入れて、閉じる

				if (this.searchThread != null) {
					this.searchThread.End();
				}

				DecideSelectedItem();
				this.DialogResult = DialogResult.OK;
				Close();
			}
		}

		private void FileSearchForm_MouseDoubleClick(object sender, MouseEventArgs e) {

		}

		private void searchStringText_TextChanged(object sender, EventArgs e) {

		}

		private void searchStringText_KeyDown(object sender, KeyEventArgs e) {
			if (e.KeyData == Keys.Enter) {
				//検索文字列テキストボックスでEnterを押された場合、検索を実行

				if (this.searchThread != null) {
					//検索中だったら、いったん終了させる
					this.searchThread.End();
				}
				Execute();
				foundItemList.Focus();
			}
		}

		private void toolStripStatusLabel2_Click(object sender, EventArgs e) {

		}

		private void okButton_Click(object sender, EventArgs e) {
			if (this.searchThread != null) {
				this.searchThread.End();
			}
			DecideSelectedItem();
			this.DialogResult = DialogResult.OK;
			Close();
		}

		private void foundItemList_DoubleClick(object sender, EventArgs e) {
			if (this.searchThread != null) {
				this.searchThread.End();
			}
			DecideSelectedItem();
			this.DialogResult = DialogResult.OK;
			Close();
		}

		/// <summary>
		/// 選択アイテムを確定させる
		/// </summary>
		private void DecideSelectedItem() {
			if (foundItemNum > 0 && foundItemList.SelectedItems.Count > 0) {
				selectedItem = (FilerItem)foundItemList.SelectedItems[0].Tag;
			} else {
				selectedItem = null;
			}
		}

		/// <summary>
		/// [実行]ボタンが押されたとき
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void executeButton_Click(object sender, EventArgs e) {
			//実行ボタンが押されたときは、ファイラがそのアイテムの場所を開くのではなくて
			//関連付けられたアプリケーションで実行する

			if (this.searchThread != null) {
				this.searchThread.End();
			}
			DecideSelectedItem();
			if (selectedItem == null) {
				MessageBox.Show("アイテムが選択されていません");
				return;
			}
			FilerItem target = selectedItem;
			if (target.Type == ItemType.Folder) {
				MessageBox.Show("フォルダの直接実行には対応していません");
				return;
			}
			//実行
			((File)target).Execute();
			this.DialogResult = DialogResult.None;
			Close();
		}

		/// <summary>
		/// 検索結果をさらに絞り込み文字列（narrowDownString）で絞り込む
		/// </summary>
		/// <param name="narrowDownString"></param>
		private void narrowDownSearchResult(String narrowDownString) {
			if (this.searchThread != null) {
				this.searchThread.End();
			}

			String searchString = narrowDownString.ToLower();
			SuspendLayout();
			foreach (ListViewItem listViewItem in foundItemList.Items) {
				//見つかったアイテムのリストからnarrowDownStringを含んでいないのを削っていく
				FilerItem item = (FilerItem)listViewItem.Tag;
				String itemString = item.Name.ToLower();
				if (itemString.IndexOf(searchString) < 0) {
					listViewItem.Remove();
				}
			}
			ResumeLayout();
			//最後に絞り込み後アイテムが残っていたら、リストビューにフォーカスを移す。使い勝手のため
			if (foundItemList.Items.Count > 0) {
				foundItemList.Focus();
			}
		}

		/// <summary>
		/// 「絞り込み」テキストボックスでのキーダウンイベント
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void narrowDownTextBox_KeyDown(object sender, KeyEventArgs e) {
			if (e.KeyCode == Keys.Enter) {
				if (this.searchThread != null) {
					this.searchThread.End();
				}
				narrowDownSearchResult(narrowDownTextBox.Text);
			}
		}

		/// <summary>
		/// 「絞り込み」ボタン押下
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void narrowDownButton_Click(object sender, EventArgs e) {
			if (this.searchThread != null) {
				this.searchThread.End();
			}
			narrowDownSearchResult(narrowDownTextBox.Text);
		}

		/// <summary>
		/// フォーム表示時
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void FileSearchForm_Shown(object sender, EventArgs e) {
			this.searchStringTextbox.Focus();
		}

		#region マルチスレッド用
		/// <summary>
		/// FullName表示用テキスト領域への文字列表示デリゲート
		/// </summary>
		/// <param name="text"></param>
		public delegate void PrintFullNameLabelDelegate(string text);

		/// <summary>
		/// FullName表示用テキスト領域への文字列表示
		/// </summary>
		/// <param name="text"></param>
		public void PrintFullNameLabel(string text) {
			this.fullnameLabel.Text = text;
			this.statusbar.Update();
		}

		/// <summary>
		/// 検索結果欄へのアイテム追加デリゲート
		/// </summary>
		/// <param name="item"></param>
		public delegate void AddFoundItemDelegate(FilerItem item);

		/// <summary>
		/// 検索結果欄へのアイテム追加
		/// </summary>
		/// <param name="item"></param>
		public void AddFoundItem(FilerItem item) {
			ListViewItem listviewItem = new ListViewItem(item.Name);
			listviewItem.SubItems.Add(item.FullName);
			listviewItem.Tag = item;
			//フォルダならアイコン追加
			if (item.Type == ItemType.Folder) {
				listviewItem.ImageIndex = 0;
			}
			foundItemList.Items.Add(listviewItem);

			//１個目のアイテムの場合はフォーカスをセットする
			if (foundItemList.Items.Count == 1) {
				foundItemList.Items[0].Selected = true;
				foundItemList.Items[0].Focused = true;
			}
			foundItemNum++;
		}

		/// <summary>
		/// 検索中のエラーを記録するデリゲート
		/// </summary>
		/// <param name="text"></param>
		public delegate void RecordErrorDelegate(string text);

		/// <summary>
		/// 検索中のエラーを記録する
		/// </summary>
		/// <param name="text">記録するテキスト</param>
		public void RecordError(string text) {
			this.errors.Add(text);
		}

		/// <summary>
		/// 検索終了を通知するためのデリゲート
		/// </summary>
		public delegate void NotifySearchEndDelegate();

		/// <summary>
		/// 検索終了通知
		/// </summary>
		public void NotifySearchEnd() {
			this.fullnameLabel.Text = "検索終了";
			if (foundItemNum == 0) {
				foundItemList.Items.Add("該当するファイルはありませんでした");
			}

			//エラーがあった場合はダイアログで通知する
			if (errors.Count > 0) {
				ErrorDialog dialog = new ErrorDialog();
				foreach (string str in errors) {
					dialog.Add(str + "\r\n");
				}
				dialog.ShowDialog();
			}
		}
		#endregion マルチスレッド用

		/// <summary>
		/// 検索中氏ボタンの押下
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void stopSearchButton_Click(object sender, EventArgs e) {
			if (this.thread != null) {
				this.thread.Abort();
			}
			this.fullnameLabel.Text = "検索を中止しました";
		}


	}
}