﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Drawing;

namespace SCFiler2.Util {
	/// <summary>
	/// SCFilerで使うオプションを保存するときに使うユーティリティクラス
	/// 前提条件として、オプションを表すクラスはpublicフィールド値のみとすること
	/// 対応：int, bool, Color
	/// </summary>
	public class OptionUtil {
		/// <summary>
		/// オプションを表すクラスのフィールド値を全部保存するクラス
		/// オプションを表すクラスは全部publicフィールド値だとします。
		/// そのフィールド値をフィールド名=フィールド値のToString()　の形式で保存します。
		/// 対応：int, bool, Color
		/// </summary>
		/// <param name="sectionName">保存の時にセクションの名前としてつけたい文字列</param>
		/// <param name="instance">設定を保存したいインスタンス</param>
		static public void Save(string sectionName, Object instance) {
			SettingFile file = SettingFile.Instance;
			Type type = instance.GetType();
			FieldInfo[] fieldInfo = type.GetFields();
			foreach (FieldInfo info in fieldInfo) {
				if (info.FieldType == typeof(Color)) {
					int value = ((Color)info.GetValue(instance)).ToArgb();
					file.Save(sectionName, info.Name, value.ToString());
				} else if (info.FieldType == typeof(List<string>)) {
					List<string> list = (List<string>)info.GetValue(instance);
					string outputStr = Util.StringUtil.GetConnectedStringFromList(list, ';');
					file.Save(sectionName, info.Name, outputStr);
				} else {
					file.Save(sectionName, info.Name, info.GetValue(instance).ToString());
				}
			}
		}

		/// <summary>
		/// オプションを表すクラスのフィールド値を全部読み込む
		/// 対応：int, bool, Color
		/// </summary>
		/// <param name="sectionName">saveしたときにつけたセクション名</param>
		/// <param name="instance">読み込んだ値を保存するインスタンス</param>
		static public void Load(string sectionName, Object instance) {
			//TODO TryParseにした方がパフォーマンスがよさそう

			SettingFile file = SettingFile.Instance;
			Type type = instance.GetType();
			FieldInfo[] fieldInfo = type.GetFields();
			foreach (FieldInfo info in fieldInfo) {
				try {
					string value = file.Load(sectionName, info.Name);
					if (info.FieldType == typeof(int)) {
						info.SetValue(instance, int.Parse(value));
					} else if (info.FieldType == typeof(bool)) {
						info.SetValue(instance, bool.Parse(value));
					} else if (info.FieldType == typeof(Color)) {
						info.SetValue(instance, Color.FromArgb(int.Parse(value)));
					} else if (info.FieldType == typeof(List<string>)) {
						info.SetValue(instance, Util.StringUtil.GetListFromConnectedString(value, new char[] { ';' }));
					} else if (info.FieldType.IsEnum) {
						info.SetValue(instance, Enum.Parse(info.FieldType, value));
					}
				} catch (FormatException) {
					//parseに失敗した時は何もしない（デフォルト値を使う）
				} catch (ArgumentException) {
					//同じくArgumentExceptionの時も何もしない（デフォルト値を使う）
				}
			}
		}
	}
}
