/*
	decode_sci.c
	sciLogger high sample data decoder

	入力ファイル　Little Endian
	このプログラムが動作するマシンもLittle Endianである必要あり。

  Copyright Naoya Takamura@NT systemd design, 2011
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#define	VER	"1.0"
#define	AD_CH	6
#define	AD_SAMPLE	50
#define	AD_BYTES	3

struct _recdata_head_t {
    uint16_t	year;	// UTC
    uint8_t		month;
    uint8_t		day;
    uint8_t		hour;
    uint8_t		min;
    uint8_t		sec;
    uint32_t	tacc;	// ns
    uint8_t		valid;
    uint16_t	freq;	// 記録周波数 Hz
} __attribute__((__packed__));


typedef struct _recdata_head_t	recdata_head;

#define	REC_DATALEN_HEAD	sizeof(recdata_head)

/*
	Little Endian
	3byte -> signed long(4byte)
*/
static long b3_to_long32(unsigned char *ptr)
{
	char	buf[4];
	
	buf[0] = *ptr++;
	buf[1] = *ptr++;
	buf[2] = *ptr;
	if (*ptr & 0x80) {
		buf[3] = 0xFF;
	} else {
		buf[3] = 0;
	}
	return *((long*)buf);
}

/*
	data decode
*/
static void read_high(FILE *fp)
{
	recdata_head	d;
	int	ch, j;
	uint8_t	data[AD_CH * AD_SAMPLE * AD_BYTES];

	while(1) {
		// Header
		if (fread(&d, 1, REC_DATALEN_HEAD, fp) < 1) break;
//fprintf(stderr, "%02d:%02d:%02d.%03d %d %d", d.hour, d.min, d.sec, (1000*j/AD_SAMPLE), d.tacc, d.valid);
		if (d.freq > AD_SAMPLE || d.freq < 1) {
			fprintf(stderr, "read_high() Error freq=%d\n", d.freq);
			break;
		}
		int data_size = AD_CH * d.freq * AD_BYTES;
		// AD data
		if (fread(data, 1, data_size, fp) < 1) break;
		for(j = 0; j < d.freq; j++) {
			fprintf(stdout, "%02d:%02d:%02d.%03d", d.hour, d.min, d.sec, (1000*j/d.freq));
			for(ch = 0; ch < AD_CH; ch++) {
				fprintf(stdout, ",%+07ld", b3_to_long32((char*)(data + ch*d.freq*AD_BYTES + j*AD_BYTES)));
			}
			fprintf(stdout, "\n");
		}
	}
}

int main(int argc, char *argv[])
{
	FILE	*fpin;
	FILE	*fpout;
	char	fname_out[128];
	char	sz[32];
	unsigned char	buf[256];

//printf("len=%d\n", REC_DATALEN_HEAD);
// exit(0);

	if (argc == 1) {
		printf("sciLogger high sample data converter Ver"VER"\n");
		printf("Please set filename.\n");
		exit(0);
	}
	
	fpin = fopen(argv[1], "rb");

	if (fpin == NULL) {
		fprintf(stderr, "Input file open error.\n");
		goto errout2;
	}
	// 出力ファイル名
	strcpy(fname_out, argv[1]);
	strcat(fname_out, ".txt");

	fpout = freopen(fname_out, "w", stdout);
	if (fpout == NULL) {
		fprintf(stderr, "Output file open ERROR. %s\n", fname_out);
		goto errout1;
	}
	read_high(fpin);

	fclose(fpout);
	fclose(fpin);
	return 0;
errout1:
	fclose(fpin);
errout2:
	return -1;
}
