package jp.co.secioss.SeciossOTP;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

/**
 * Simple SeciossOTP database access helper class. Defines the basic CRUD operations
 * for the SeciossOTP, and gives the ability to list all secrets as well as
 * retrieve or modify a specific secret.
 */
public class SeciossOTPDbAdapter {

    public static final String KEY_ROWID = "_id";
    public static final String KEY_ALIAS = "alias";
    public static final String KEY_SECRET = "secret";

    private static final String TAG = "SeciossOTPDbAdapter";
    private DatabaseHelper mDbHelper;
    private SQLiteDatabase mDb;

    /**
     * Database creation SQL statement
     */
    private static final String DATABASE_CREATE =
    		"create table secret (_id integer primary key autoincrement, alias text not null, secret text not null);";

    private static final String DATABASE_NAME = "seciossotp";
    private static final String DATABASE_TABLE = "secret";
    private static final int DATABASE_VERSION = 2;

    private final Context mCtx;

    private static class DatabaseHelper extends SQLiteOpenHelper {

        DatabaseHelper(Context context) {
            super(context, DATABASE_NAME, null, DATABASE_VERSION);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            db.execSQL(DATABASE_CREATE);
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            Log.w(TAG, "Upgrading database from version " + oldVersion + " to "
                    + newVersion + ", which will destroy all old data");
            db.execSQL("DROP TABLE IF EXISTS secrets");
            onCreate(db);
        }
    }

    /**
     * Constructor - takes the context to allow the database to be
     * opened/created
     * 
     * @param ctx the Context within which to work
     */
    public SeciossOTPDbAdapter(Context ctx) {
        this.mCtx = ctx;
    }

    /**
     * Open the SeciossOTP database. If it cannot be opened, try to create a new
     * instance of the database. If it cannot be created, throw an exception to
     * signal the failure
     * 
     * @return this (self reference, allowing this to be chained in an
     *         initialization call)
     * @throws SQLException if the database could be neither opened or created
     */
    public SeciossOTPDbAdapter open() throws SQLException {
        mDbHelper = new DatabaseHelper(mCtx);
        mDb = mDbHelper.getWritableDatabase();
        return this;
    }

    public void close() {
        mDbHelper.close();
    }


    /**
     * Create a new secret using the alias and secret provided. If the secret is
     * successfully created return the new rowId for that secret, otherwise return
     * a -1 to indicate failure.
     * 
     * @param alias the alias of the secret
     * @param secret the secret of the secret
     * @return rowId or -1 if failed
     */
    public long createSecret(String alias, String secret) {
        ContentValues initialValues = new ContentValues();
        initialValues.put(KEY_ALIAS, alias);
        initialValues.put(KEY_SECRET, secret);

        return mDb.insert(DATABASE_TABLE, null, initialValues);
    }

    /**
     * Delete the secret with the given rowId
     * 
     * @param rowId id of secret to delete
     * @return true if deleted, false otherwise
     */
    public boolean deleteSecret(long rowId) {

        return mDb.delete(DATABASE_TABLE, KEY_ROWID + "=" + rowId, null) > 0;
    }

    /**
     * Return a Cursor over the list of all secrets in the database
     * 
     * @return Cursor over all secrets
     */
    public Cursor fetchAllSecrets() {

        return mDb.query(DATABASE_TABLE, new String[] {KEY_ROWID, KEY_ALIAS,
                KEY_SECRET}, null, null, null, null, null);
    }

    /**
     * Return a Cursor positioned at the secret that matches the given rowId
     * 
     * @param rowId id of secret to retrieve
     * @return Cursor positioned to matching secret, if found
     * @throws SQLException if secret could not be found/retrieved
     */
    public Cursor fetchSecret(long rowId) throws SQLException {

        Cursor mCursor =

            mDb.query(true, DATABASE_TABLE, new String[] {KEY_ROWID,
                    KEY_ALIAS, KEY_SECRET}, KEY_ROWID + "=" + rowId, null,
                    null, null, null, null);
        if (mCursor != null) {
            mCursor.moveToFirst();
        }
        return mCursor;

    }

    /**
     * Update the secret using the details provided. The secret to be updated is
     * specified using the rowId, and it is altered to use the alias and secret
     * values passed in
     * 
     * @param rowId id of secret to update
     * @param alias value to set secret alias to
     * @param secret value to set secret secret to
     * @return true if the secret was successfully updated, false otherwise
     */
    public boolean updateSecret(long rowId, String alias, String secret) {
        ContentValues args = new ContentValues();
        args.put(KEY_ALIAS, alias);
        args.put(KEY_SECRET, secret);

        return mDb.update(DATABASE_TABLE, args, KEY_ROWID + "=" + rowId, null) > 0;
    }

}
