﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "Console.h"


namespace bearmini
{
    ///
    ///  コンストラクタ。
    ///  コンソールウインドウにアタッチします。
    ///
    ///  @param[in] processId  アタッチしたいコンソールウインドウを持っているプロセスのプロセス ID
    ///
    Console::Console(::DWORD processId) :
        m_hConin(0),
        m_hConout(0)
    {
        // 指定されたプロセスのコンソールにアタッチする
        attachTo(processId);

        // コンソールを一番手前に持ってくる
        bringToTop();

        // コンソール入出力のハンドルを得る
        ::SECURITY_ATTRIBUTES sa;
        sa.nLength = sizeof(sa);
        sa.lpSecurityDescriptor = 0;
        sa.bInheritHandle = TRUE;
        m_hConin  = ::CreateFileW(L"CONIN$",  GENERIC_READ | GENERIC_WRITE, 0, &sa, OPEN_EXISTING, 0, 0);
        m_hConout = ::CreateFileW(L"CONOUT$", GENERIC_READ | GENERIC_WRITE, 0, &sa, OPEN_EXISTING, 0, 0);
    }


    ///
    ///  デストラクタ。
    ///  コンソールウインドウを閉じます。
    ///
    Console::~Console()
    {
        ::CloseHandle(m_hConin);
        ::CloseHandle(m_hConout);
        ::FreeConsole();
    }


    ///
    ///  コンソールウインドウにタイトルを設定します。
    ///
    ///  @param[in] title タイトルバーに設定するタイトル
    ///
    void Console::SetTitle(const std::wstring& title)
    {
        // タイトルを設定
        ::SetConsoleTitleW(title.c_str());
    }


    ///
    ///  コンソールに文字列を入力します。
    ///
    ///  @param[in] str 入力したい文字列
    ///
    void Console::Input(const ConsoleInputArray& inputs)
    {
        ::DWORD nWritten;
        ::BOOL ok = ::WriteConsoleInputW(m_hConin, inputs.InputArray(), inputs.Length(), &nWritten);
        if (!ok)
        {
            throw std::exception("WriteConsoleInputW() failed.");
        }
    }


    ///
    ///  processId で指定された ID のプロセスが持っているコンソールにアタッチします。
    ///
    void Console::attachTo(::DWORD processId)
    {
        // 現在のコンソールを解放してから目的のプロセスのコンソールにアタッチする。
        ::FreeConsole();

        if (::AttachConsole(processId) == 0)
        {
            ::DWORD errorCode = ::GetLastError();
            std::stringstream ss;
            ss << "Could not attach to a console window." << std::endl << "Error Code: " << errorCode;
            throw std::exception(ss.str().c_str());
        }
    }


    ///
    ///  コンソールウインドウを一番手前に持ってくる
    ///
    void Console::bringToTop()
    {
        // コンソールのウインドウハンドルを取得
        ::HWND hWnd = ::GetConsoleWindow();

        // ウインドウを一番手前に持ってくる
        ::SetForegroundWindow(hWnd);
    }

} 
