﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "Path.h"

// Windows headers
#include <shlwapi.h>  // for PathXxxx()
#pragma  comment(lib,"shlwapi.lib")


namespace bearmini
{

    ///
    ///  path で指定されたパス文字列のうち、ディレクトリ部分を取得します。（ドライブ名を含む）
    ///
    ///  @param path ディレクトリ部分を取得したいパス文字列
    ///  
    ///  @return    path のディレクトリ部分（ドライブ名を含む）
    ///
    std::wstring Path::GetDirectory(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        std::wstringstream s;
        s << drive << dir;
        return s.str();
    }


    ///
    ///  path で指定されたパス文字列のうち、ディレクトリ部分を取得します。（ドライブ名を除く）
    ///
    ///  @param path ディレクトリ部分を取得したいパス文字列
    ///  
    ///  @return    path のディレクトリ部分（ドライブ名を除く）
    ///
    std::wstring Path::GetDirectoryWithoutDrive(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        return dir;
    }


    ///
    ///  path で指定されたパス文字列のうち、ファイル名部分を取得します。（拡張子を含む）
    ///
    ///  @param path ファイル名部分を取得したいパス文字列
    ///  
    ///  @return    path のファイル名部分（拡張子を含む）
    ///
    std::wstring Path::GetFileName(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        std::wstringstream s;
        s << fname << ext;
        return s.str();
    }


    ///
    ///  path で指定されたパス文字列のうち、ファイル名部分を取得します。（拡張子を除く）
    ///
    ///  @param path ファイル名部分を取得したいパス文字列
    ///  
    ///  @return    path のファイル名部分（拡張子を除く）
    ///
    std::wstring Path::GetFileNameWithoutExtension(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        return fname;
    }


    ///
    ///  path で指定されたパス文字列のうち、拡張子部分を取得します。（ドットを含む）
    ///
    ///  @param path 拡張子部分を取得したいパス文字列
    ///  
    ///  @return    path の拡張子部分（ドットを含む）
    ///
    std::wstring Path::GetExtension(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        return ext;
    }


    ///
    ///  path で指定されたパス文字列のうち、拡張子部分を取得します。（ドットを除く）
    ///
    ///  @param path 拡張子部分を取得したいパス文字列
    ///  
    ///  @return    path の拡張子部分（ドットを除く）
    ///
    std::wstring Path::GetExtensionWithoutDot(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        std::wstring result = ext;
        if (result.at(0) == '.')
        {
            result = result.substr(1);
        }
        return result;
    }


    ///
    ///  path で指定されたパス文字列のうち、ドライブ名部分を取得します。（コロン含む）
    ///
    ///  @param path ドライブ名部分を取得したいパス文字列
    ///  
    ///  @return    path のドライブ名部分（コロンを含む）
    ///
    std::wstring Path::GetDrive(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        return drive;
    }


    ///
    ///  path で指定されたパス文字列のうち、ドライブ名部分を取得します。（コロンを除く）
    ///
    ///  @param path ドライブ名部分を取得したいパス文字列
    ///  
    ///  @return    path のドライブ名部分（コロンを除く）
    ///
    std::wstring Path::GetDriveWithoutColon(const std::wstring& path)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        if (drive[1] == ':')
        {
            drive[1] = '\0';
        }
        return drive;
    }


    ///
    ///  path で指定されたパス文字列のうち、拡張子を newExt に変更します。
    ///  
    ///  @param[in] path    拡張子部分を変更したいパス文字列
    ///  @param[in] newExt  変更後の拡張子
    ///
    ///  @return    拡張子部分が変更されたパス文字列
    ///
    std::wstring Path::ChangeExtension(const std::wstring& path, const std::wstring& newExt)
    {
        wchar_t drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
        ::_wsplitpath_s(path.c_str(), drive, _MAX_DRIVE, dir, _MAX_DIR, fname, _MAX_FNAME, ext, _MAX_EXT);

        std::wstringstream s;
        s << drive << dir << fname << newExt;
        return s.str();
    }


    ///
    ///  path で指定されたパス文字列が確実にバックスラッシュ(\) で終端されるようにします。
    ///  すでにバックスラッシュで終端されている場合はなにもせず、
    ///  まだバックスラッシュがない場合は末尾にバックスラッシュを追加します。
    ///
    ///  @param[in] path   バックスラッシュを付加したいパス
    ///
    ///  @return  バックスラッシュで終端されているパス
    ///
    std::wstring Path::AddBackslash(const std::wstring& path)
    {
        wchar_t buf[_MAX_PATH + 2] = { 0 };  // + 2 は、バックスラッシュの分と NULL 文字の分
        path.copy(buf, min(_MAX_PATH, path.length()));
        LPWSTR pTail = ::PathAddBackslashW(buf);
        if (pTail == 0)
        {
            throw std::exception("PathAddBackslash() failed.");
        }

        return buf;
    }


	///
	///  path で指定されたパス文字列がバックスラッシュ(\) で終端されていれば削除します。
	///  バックスラッシュで終端されていない場合は何もしません。
	///  ただしルートディレクトリを指定された場合はバックスラッシュを削除しません。
	///
	///  @param[in] path   バックスラッシュを削除したいパス
	///
	///  @return   バックスラッシュが削除されたパス
	///
	std::wstring Path::RemoveTrailingBackslash(const std::wstring& path)
	{
		wchar_t buf[_MAX_PATH + 1] = { 0 };
		path.copy(buf, min(_MAX_PATH, path.length()));

		::PathRemoveBackslashW(buf);
		
		return buf;
	}

    ///
    ///  s で指定された文字列がダブルクォーテーションで囲まれていたら取り除きます。
    ///  s で指定された文字列がダブルクォーテーションで囲まれていなければ、s で指定された文字列を
    ///  そのままコピーしたものを返します。
    ///
    ///  @param[in] s ダブルクォーテーションを取り除きたい文字列。
    ///
    ///  @return    ダブルクォーテーションが取り除かれた文字列。
    ///
    std::wstring Path::Unquote(const std::wstring& s)
    {
        if ((s.at(0) == '\"') && (s.at(s.length()-1) == '\"'))
        {
            return s.substr(1,s.length()-2);
        }
        else
        {
            return s;
        }
    }


    ///
    ///  dir で指定されたディレクトリ名と file で指定されたファイル名を結合して、1つのパス文字列を作ります。
    ///
    ///  @param[in] dir     ディレクトリ名
    ///  @param[in] file    ファイル名
    ///
    ///  @return   結合されたパス名
    ///
    std::wstring Path::Combine(const std::wstring& dir, const std::wstring& file)
    {
        wchar_t buf[_MAX_PATH + 1] = { 0 };
        ::PathCombineW(buf, dir.c_str(), file.c_str());
        return buf;
    }


    ///
    ///  path で指定されたパス文字列がさしているものがディレクトリかファイルかを調べます。
    ///  path で指定されたパスが、ファイルシステム上に実在するものでなければこの関数は例外を送出します。
    ///  
    ///  path で指定されたパス文字列がディレクトリをさしていても、
    ///  文字列の末尾が \ で終わっていてはいけないことに注意してください。
    ///  （内部で使用している FindFirstFile() Win32 API の仕様より）
    ///  （末尾が \ だったらこの関数の中では無視するようにしてもいいのかも）
    ///
    ///  @param[in] path ディレクトリかどうかを調べたいパス文字列
    ///
    ///  @return path で指定されたパス文字列がディレクトリであれば true、そうでなければ false
    ///
    bool Path::IsDirectory(const std::wstring& path)
    {
        /*  なぜ PathIsDirectory() を使っていなかったんだろう？
        ::WIN32_FIND_DATAW finddata;
        ::HANDLE hFind = ::FindFirstFileW(path.c_str(), &finddata);
        if (hFind == INVALID_HANDLE_VALUE)
        {
            ::FindClose(hFind);
            throw std::exception("FindFirstFileW() failed.");
        }
        ::FindClose(hFind);

        return (finddata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
        */

        return BOOL2bool(::PathIsDirectory(path.c_str()));
    }


    ///
    ///  path で指定されたパス文字列が UNC かどうかを調べます。
    ///
    bool Path::IsUNC(const std::wstring& path)
    {
        return BOOL2bool(::PathIsUNCW(path.c_str()));
    }


}