﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "Process.h"

#include <shellapi.h>

#include "Foundation.h"
#include "Path.h"


namespace bearmini
{

    ///
    ///  コンストラクタ
    ///
    ///  @param[in] path    カレントディレクトリで起動したいプログラムの実行ファイル
    ///
    Process::Process(const std::wstring& path) :
        m_path(path),
        m_workDir(L""),
        m_hProcess(0),
        m_hThread(0)
    {
    }


    ///
    ///  コンストラクタ
    ///
    ///  @param[in] path    起動したいプログラムの実行ファイル
    ///  @param[in] workDir プログラムを実行するときの作業ディレクトリ（プログラムの初期カレントディレクトリ）
    ///
    Process::Process(const std::wstring& path, const std::wstring& workDir) :
        m_path(path),
        m_workDir(workDir),
        m_hProcess(0),
        m_hThread(0)
    {
    }


    ///
    ///  デストラクタ
    ///
    Process::~Process()
    {
        ::CloseHandle(m_hThread);
        ::CloseHandle(m_hProcess);
    }


    ///
    ///  あらかじめメンバで指定されたプログラムを起動し、プロセスを作成します。
    ///  （コンソールプロセス専用）
    ///
    ///  @param[in] asAdmin   Administrator 権限で起動するかどうかを指定します。
    ///
    void Process::Run(bool asAdmin)
    {
        if (!asAdmin)
        {
            run();
        }
        else
        {
            runAsAdmin();
        }
    }


    ///
    ///  あらかじめメンバで指定されたプログラムを起動し、プロセスを作成します。
    ///  （private なヘルパ関数）
    ///
    void Process::run()
    {
        // StartupInfo を初期化
        ::STARTUPINFOW si = { 0 };
        si.cb = sizeof(si);
        
        LPCWSTR workdir = 0;
        if (m_workDir != L"")
        {
            workdir = m_workDir.c_str();
        }

        // プロセスを作成
        ::PROCESS_INFORMATION pi;
        ::BOOL ok = ::CreateProcessW(0, (::LPWSTR)m_path.c_str(), 0, 0, FALSE, CREATE_NEW_CONSOLE, 0, workdir, &si, &pi);
        if (!ok)
        {
            throw std::exception("Failed to start the process.");
        }

        m_hProcess = pi.hProcess;
        m_hThread = pi.hThread;
    }

    
    ///
    ///  あらかじめメンバで指定されたプログラムを Administrator 権限で起動し、プロセスを作成します。
    ///  （private なヘルパ関数）
    ///
    void Process::runAsAdmin()
    {
        LPCWSTR workdir = 0;
        if (m_workDir != L"")
        {
            workdir = m_workDir.c_str();
        }

        ::SHELLEXECUTEINFOW sei = { 0 };
        sei.cbSize = sizeof(sei);
        sei.fMask = SEE_MASK_NOCLOSEPROCESS;
        sei.hwnd = 0;
        sei.lpVerb = L"runas";
        sei.lpFile = (::LPCWSTR)m_path.c_str();
        sei.lpParameters = 0;
        sei.lpDirectory = workdir;
        sei.nShow = SW_NORMAL;
        sei.hInstApp = 0;

        ::BOOL ok = ::ShellExecuteExW(&sei);
        if (!ok)
        {
            throw std::exception("Failed to start the process.");
        }

        m_hProcess = sei.hProcess;
        m_hThread = 0;
    }


    ///
    ///  コールバック関数に引数を渡して結果を受け取るための構造体です。
    ///
    struct TARGET_WINDOW_SEARCH_PARAM_AND_RESULT
    {
        // parameter
        ::DWORD targetProcessId;

        // result
        bool found;
    };


    ///
    ///  ウインドウを探すために EnumWindows() から呼び出されるコールバック関数です。
    ///
    ::BOOL CALLBACK isTargetWindow(HWND hWnd, LPARAM lParam)
    {
        TARGET_WINDOW_SEARCH_PARAM_AND_RESULT* ps = (TARGET_WINDOW_SEARCH_PARAM_AND_RESULT*) lParam;

        ::DWORD currProcessId = 0;
        ::GetWindowThreadProcessId(hWnd, &currProcessId);

        if (currProcessId == ps->targetProcessId)
        {
            ps->found = true;
            return FALSE;
        }

        return TRUE;
    }
    

    ///
    ///  現在のプロセスにコンソールウインドウが割りつけられるまで待ちます。
    ///
    void Process::WaitForConsoleAllocated()
    {
        // コンソールアプリのように、メッセージループを持たないプロセスに対しては WaitForInputIdle は使用できない。
        //::WaitForInputIdle(process.GetHandle(), 10*1000);

        TARGET_WINDOW_SEARCH_PARAM_AND_RESULT s;
        s.targetProcessId = GetId();
        s.found = false;

        int loop = 0;
        while (!s.found)
        {
            ::BOOL ok = ::EnumWindows(isTargetWindow, (LPARAM) &s);
            if (s.found)
            {
                return;
            }

            if (!ok)
            {
                throw std::exception("EnumWindows() failed.");
            }

            if (++loop > 100)
            {
                throw std::exception("WaitForConsoleAllocated() timed out.");
            }

            ::Sleep(100);
        }
    }
}