﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "SendToCygwin.h"

// private headers
#include "Console.h"
#include "ConsoleInputArray.h"
#include "EffectiveBehavior.h"
#include "MessageBox.h"
#include "Path.h"
#include "Process.h"
#include "SettingRepository.h"
#include "ShellApi.h"
#include "Shortcut.h"
#include "SystemInfo.h"


namespace bearmini
{
    ///
    ///  「Cygwin に送る」のプログラムを実行します。
    ///  
    ///  @param args コマンドラインで指定された引数
    ///
    void SendToCygwin::Run(const wstring_vector& args)
    {
        validateArguments(args);

        // Cygwin.bat を実行するためのコマンドライン文字列を作成
        std::wstring commandLine = generateCommandLine();

        FOREACH(wstring_vector::const_iterator, path, args)
        {
            std::wstring unquotedPath = Path::Unquote(*path);
            std::wstring workDir = getWorkDirectoryFor(unquotedPath);

			Process process(commandLine);
            process.Run();
            process.WaitForConsoleAllocated();

            Console console(process.GetId());
            console.SetTitle(getConsoleTitle());

            // ファイル名があるときは、コンソールに送る
            if (!Path::IsDirectory(unquotedPath))
            {
                ConsoleInputArray inputs;
                arrangeConsoleInputArray(unquotedPath, workDir, &inputs);
                console.Input(inputs);
            }
            else
            {
                ConsoleInputArray inputs;
                arrangeConsoleInputArrayChangeDirOnly(workDir, &inputs);
                console.Input(inputs);
            }
        }

    }


    ///
    ///  プログラムのセットアップ（インストールもしくはアンインストール）を行います。
    ///
    void SendToCygwin::Setup()
    {
        int result = MessageBox::Show(this, L"インストールしますか？", MessageBoxType_YesNo);
        if (result == DialogResult_Yes)
        {
            install();
            return;
        }

        result = MessageBox::Show(this, L"それではアンインストールしますか？", MessageBoxType_YesNo);
        if (result == DialogResult_Yes)
        {
            uninstall();
            return;
        }
    }


    ///
    ///  コマンドラインで渡されてきた引数が CMD.EXE の実行にふさわしいものかどうか、
    ///  検証します。
    ///  実行にふさわしいかどうかの検証とは、ファイル数が多すぎないか、とか
    ///  Cygwin.bat で取り扱うことのできないネットワーク上のファイルでないか、
    ///  といったような点を調べます。
    ///  実行にふさわしくないと判断された場合、例外を送出します。
    ///
    ///  @param[in] args コマンドラインで渡されてきた引数
    ///
    void SendToCygwin::validateArguments(const wstring_vector& args)
    {
        static const unsigned int MAX_SIMULTANEOUS_EXEC_COUNT = SettingRepository::GetMaxSimultaneous();

        if (args.size() > MAX_SIMULTANEOUS_EXEC_COUNT)
        {
            throw std::exception("Too many files. \r\n\r\nIn order to change the number of PowerShell can be started, please modify the value of MAX_SIMULTANEOUS in SendToPS.ini file.");
        }
    }


    ///
    ///  Cygwin.bat を実行するためのコマンドラインを生成します。
    ///
    ///  @return    Cygwin.bat を実行するためのコマンドライン
    ///
    std::wstring SendToCygwin::generateCommandLine()
    {
        std::wstringstream s;
        s << getCygwinBatPath() << L" " << getAdditionalParamForCygwin();
        return s.str();
    }


    ///
    ///  cygwin.bat のパスを取得します。
    ///
    ///  @return    cygwin.bat のフルパス
    ///
    std::wstring SendToCygwin::getCygwinBatPath()
    {
        // 設定ファイルで特に指定されていない場合はデフォルトのパス
        std::wstring settingCygwinLauncher = SettingRepository::GetCygwinLauncher();
        
        if (settingCygwinLauncher.compare(L"") == 0)
        {
            std::wstringstream path;
            path << Path::AddBackslash(getCygwinRootDirectoryFromRegistry()) << L"cygwin.bat";
            return path.str();
        }

        // 設定ファイルで指定されている場合はそれを使用
        else
        {
            return settingCygwinLauncher;
        }
    }


    ///
    ///  レジストリから Cygwin のルートディレクトリを取得します。
    ///
    ///  FIXME: HKEY をラップするクラスを作って、例外発生時のリソースリークを防ぐ
    ///
    std::wstring SendToCygwin::getCygwinRootDirectoryFromRegistry()
    {
        const std::wstring regkeyCygwinRoot = L"SOFTWARE\\Cygnus Solutions\\Cygwin\\mounts v2\\/";
        
        ::HKEY hkey;
        ::LONG result = ::RegOpenKeyEx(HKEY_LOCAL_MACHINE, regkeyCygwinRoot.c_str(), 0, KEY_QUERY_VALUE, &hkey);
        if (result != ERROR_SUCCESS)
        {
            ::DWORD errorCode = ::GetLastError();
            std::stringstream ss;
            ss << "RegOpenKeyEx() failed." << std::endl << "Error Code: " << errorCode;
            throw std::exception(ss.str().c_str());
        }

        ::DWORD type;
        wchar_t buf[_MAX_PATH + 1] = { 0 };
        ::DWORD size = sizeof(buf) - sizeof(wchar_t);

        result = ::RegQueryValueExW(hkey, L"native", 0, &type, (LPBYTE) buf, &size);
        if (result != ERROR_SUCCESS)
        {
            throw std::exception("RegQueryValueEx() failed.");
        }

        if (type != REG_SZ)
        {
            throw std::exception("Invalid registry subkey type.");
        }

        ::RegCloseKey(hkey);

        return buf;
    }


    ///
    ///  Cygwin.bat に渡す追加のパラメータを設定ファイルから取得します。
    ///
    ///  @return    Cygwin.bat に渡す追加のパラメータ
    ///
    std::wstring SendToCygwin::getAdditionalParamForCygwin()
    {
        return L"";
    }


    ///
    ///  path で指定されたパスのファイルまたはディレクトリをコマンドプロンプトに送る際の
    ///  作業ディレクトリとなるディレクトリを取得します。
    ///
    std::wstring SendToCygwin::getWorkDirectoryFor(const std::wstring& path)
    {
        if (Path::IsDirectory(path))
        {
            return path;
        }
        else
        {
            return Path::GetDirectory(path);
        }
    }


    ///
    ///  path で指定されたファイルのパスに応じて、プロンプトに送り込むべきキー入力の配列を取得します。
    ///
    ///  @param[in]     path                 ユーザから指定されたファイルのパス
    ///  @param[in/out] pConsoleInputArray   キー入力の配列を受け取るための ConsoleInputArray クラスのインスタンスへのポインタ
    ///
    void SendToCygwin::arrangeConsoleInputArray(const std::wstring& path, const std::wstring& workDir, ConsoleInputArray* pConsoleInputArray)
    {
        BehaviorSettingsCollection behaviorSettings = SettingRepository::GetBehaviorSettings();
        BehaviorSetting bs = behaviorSettings.GetBehaviorSettingFor(path);
        EffectiveBehavior eb = EffectiveBehavior::CreateForCygwinFrom(bs, path);

        // まずは cd コマンドの分を取得
        arrangeConsoleInputArrayChangeDirOnly(workDir, pConsoleInputArray);

        // 目的のプロンプトを入力
        pConsoleInputArray->Append(eb.CommandLine());

        // いったんコマンドラインの先頭にカーソルを移動してから...
        pConsoleInputArray->AppendVirtualKeyCode(VK_LEFT, static_cast<unsigned int>(eb.CommandLine().length()));

        // カーソルを指定の位置に移動
        pConsoleInputArray->AppendVirtualKeyCode(VK_RIGHT, eb.CursorPosition());

        // 自動実行の場合は ENTER を入力する
        if (bs.auto_exec)
        {
            //pConsoleInputArray->AppendVirtualKeyCode(VK_RETURN);  <- なぜかうまくいかない
            pConsoleInputArray->Append(L"\r");  // cygwin の場合は \r\n だと改行されすぎてしまう（2回 ENTER が押される）ので \r のみ。
        }
    }


    void SendToCygwin::arrangeConsoleInputArrayChangeDirOnly(const std::wstring& workDir, ConsoleInputArray* pConsoleInputArray)
    {
        // Cygwin 用にディレクトリ名を変換
        std::wstring workDirecotryNameForCygwin = convertDirectoryNameForCygwin(workDir);

        // change directory コマンドを得る
        // ふつうは cd だが、alias したりほかのコマンドを使っていたり変なシェルを使っている人もいるかもしれないので
        std::wstring changeDirCommand = SettingRepository::GetCygwinChangeDirCommand();

        // cd コマンドで移動
        pConsoleInputArray->Append(changeDirCommand);
        pConsoleInputArray->Append(L" \"");
        pConsoleInputArray->Append(workDirecotryNameForCygwin);
        pConsoleInputArray->Append(L"\"");
        pConsoleInputArray->Append(L"\r");  // cygwin の場合は \r\n だと改行されすぎてしまう（2回 ENTER が押される）ので \r のみ。
    }


    ///
    ///  このプログラム（SendToCygwin.exe）のフルパスを取得します。
    ///
    ///  @return    このプログラムのフルパス
    ///
    std::wstring SendToCygwin::getSendToCygwinExeFullPath()
    {
        wchar_t lpszModuleName[_MAX_PATH];
        ::GetModuleFileNameW(0, lpszModuleName, _MAX_PATH);
        return lpszModuleName;
    }


    ///
    ///  このプログラム（SendToCygwin.exe）がインストールされているディレクトリを取得します。
    ///
    ///  @return    このプログラムがインストールされているディレクトリ
    ///
    std::wstring SendToCygwin::getSendToCygwinExeDirectory()
    {
        return Path::GetDirectory(getSendToCygwinExeFullPath());
    }


    ///
    ///  コンソールのタイトルとして設定する文字列を取得します。
    ///
    std::wstring SendToCygwin::getConsoleTitle()
    {
        return L"Cygwin を起動しています...";
    }


    ///
    ///  Cygwin 用にディレクトリ名を変換します。
    ///
    ///  @param[in] folderName 変換したいフォルダ名
    ///  
    ///  @return    Cygwin で使用できる形式に変換されたフォルダ名
    ///
    std::wstring SendToCygwin::convertDirectoryNameForCygwin(const std::wstring& folderName)
    {
        std::wstring result = folderName;

        // \ 記号を / に変換
        std::wstring::size_type n, nb = 0;
        while ((n = result.find(L"\\",nb)) != std::wstring::npos)
        {
            result.replace(n,1,L"/");
            nb = n + 1;
        }

        return result;
    }


    ///
    ///  インストールを行います。
    ///
    void SendToCygwin::install()
    {
        // ショートカットを作成（通常実行）
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"Cygwin に送る.lnk");
            const std::wstring sendToCygwinExePath = getSendToCygwinExeFullPath();
            const std::wstring sendToCygwinExeDir = getSendToCygwinExeDirectory();
            const std::wstring description = L"ファイルやディレクトリを指定して Cygwin を起動します.";
            Shortcut::Create(linkFilePath, sendToCygwinExePath, description, L"", sendToCygwinExeDir, L"");
        }
        
        // ショートカットを作成（管理者として実行）
        if (SystemInfo::GetOsMajorVersion() >= SystemInfo::OsMajorVersion_Vista)
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"Cygwin に送る（管理者として実行...）.lnk");
            const std::wstring sendToCygwinExePath = getSendToCygwinExeFullPath();
            const std::wstring sendToCygwinExeDir = getSendToCygwinExeDirectory();
            const std::wstring description = L"ファイルやディレクトリを指定して Cygwin を起動します.";
            Shortcut::Create(linkFilePath, sendToCygwinExePath, description, L"", sendToCygwinExeDir, L"", 0, 1, true);
        }

        MessageBox::Show(this, L"インストールが完了しました.");
    }


    ///
    ///  アンインストールを行います。
    ///
    void SendToCygwin::uninstall()
    {
        // ショートカットファイルを削除
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"Cygwin に送る.lnk");
            ::DeleteFileW(linkFilePath.c_str());
        }

        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"Cygwin に送る（管理者として実行...）.lnk");
            ::DeleteFileW(linkFilePath.c_str());
        }

        MessageBox::Show(this, L"アンインストールが完了しました.\r\nご利用ありがとうございました.");
    }

}