﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "EffectiveBehavior.h"

// private headers
#include "Foundation.h"
#include "Path.h"
#include "Util.h"


namespace bearmini
{
    ///
    ///  コンストラクタ
    ///
    EffectiveBehavior::EffectiveBehavior(const std::wstring& commandLine, bool autoExec, unsigned int cursorPosition) :
        m_commandLine(commandLine),
        m_autoExec(autoExec),
        m_cursorPosition(cursorPosition)
    {
    }


    ///
    ///  デストラクタ
    ///
    EffectiveBehavior::~EffectiveBehavior()
    {
    }


    ///
    ///  behaviorSetting で指定されたビヘイビア設定から、実効ビヘイビアを生成します。
    ///
    EffectiveBehavior EffectiveBehavior::CreateFrom(const BehaviorSetting& behaviorSetting, const std::wstring& path)
    {
        std::wstring commandLine = behaviorSetting.command_line;

        // %F 等の特殊文字をファイル名に置き換える
        commandLine = replaceSpecialCharacters(commandLine, path, false);

        // || の位置を探しておく（あとでその位置にカーソルを移動するため）
        std::wstring::size_type cursorPos = commandLine.find(L"||");
        if (cursorPos == std::wstring::npos)
        {
            cursorPos = 0;
        }

        // || を消去
        commandLine = ReplaceString(commandLine, L"||", L"");

        return EffectiveBehavior(commandLine, behaviorSetting.auto_exec, static_cast<unsigned int>(cursorPos));
    }


    ///
    ///  behaviorSetting で指定されたビヘイビア設定から、実効ビヘイビアを生成します。（Cygwin 用）
    ///
    EffectiveBehavior EffectiveBehavior::CreateForCygwinFrom(const BehaviorSetting& behaviorSetting, const std::wstring& path)
    {
        std::wstring commandLine = behaviorSetting.command_line;

        // %F 等の特殊文字をファイル名に置き換える
        commandLine = replaceSpecialCharacters(commandLine, path, true); // 通常版との違いはこの true だけ

        // || の位置を探しておく（あとでその位置にカーソルを移動するため）
        std::wstring::size_type cursorPos = commandLine.find(L"||");
        if (cursorPos == std::wstring::npos)
        {
            cursorPos = 0;
        }

        // || を消去
        commandLine = ReplaceString(commandLine, L"||", L"");

        return EffectiveBehavior(commandLine, behaviorSetting.auto_exec, static_cast<unsigned int>(cursorPos));
    }


    ///
    ///  文字列 line の中に含まれる特殊文字（%F や %P など）を置き換えます。
    ///
    ///  @param[in] line  置き換え処理対象の文字列
    ///  @param[in] path  置き換えるときに使用する path
    ///  @param[in] bEscapeBackslash \ を \\ に置き換えるかどうか（デフォルト false）
    ///
    ///  @return    文字列 line の中に含まれる特殊文字を置き換えた文字列
    ///
    std::wstring EffectiveBehavior::replaceSpecialCharacters(const std::wstring& line, const std::wstring& path, bool bEscapeBackslash)
    {
        // 結果をここに作っていく
        std::wstring result = line;

        // % の後ろの一文字を置き換える文字列
        std::wstring rep;

        // % を探す
        std::wstring::size_type n, nb = 0;
        while ((n = result.find_first_of(L'%',nb)) != std::wstring::npos)
        {
            // 一番最後に % があったら何もしないで終了
            if (n == result.length()-1)
            {
                break;
            }

            // % の次の 1 文字か 2 文字によって、置き換え後の文字列を決める
			int consumedChars = 2; // % 自体と、そのあとの 1 文字分は消費することが確定しているのでデフォルトで 2 を入れておく。2 文字以上使うときだけこれを更新する。

            switch (result.at(n+1))
            {
            case 'F': rep = Path::GetFileName(path);                    break;
            case 'f': rep = Path::GetFileNameWithoutExtension(path);    break;
            case 'D':
			{
				// D の次に来ている文字を調べる
				wchar_t followingChar = ' ';
				if (result.length() > (n + 2))
				{
					followingChar = result.at(n+2);
				}

				if (followingChar == '-')
				{
					rep = Path::RemoveTrailingBackslash(Path::GetDirectory(path));
					consumedChars++;
				}
				else
				{
					rep = Path::GetDirectory(path);
				}
                break;
			}
            case 'd':
			{
				// d の次に来ている文字を調べる
				wchar_t followingChar = ' ';
				if (result.length() > (n + 2))
				{
					followingChar = result.at(n+2);
				}

				if (followingChar == '-')
				{
					rep = Path::RemoveTrailingBackslash(Path::GetDirectoryWithoutDrive(path));
					consumedChars++;
				}
				else
				{
					rep = Path::GetDirectoryWithoutDrive(path);
				}
				break;
			}
            case 'E': rep = Path::GetExtension(path);                   break;
            case 'e': rep = Path::GetExtensionWithoutDot(path);         break;
            case 'R': rep = Path::GetDrive(path);                       break;
            case 'r': rep = Path::GetDriveWithoutColon(path);           break;
            case 'P': rep = path;                                       break;
            case '%': rep = L"%";                                       break;
            default:  rep = result.at(n+1);                             break;
            }

            // 置き換え用の文字列に \ が入っていたら、\\ にする（Cygwin 用）
            if (bEscapeBackslash)
            {
                rep = ReplaceString(rep, L"\\", L"\\\\");
            }

            // 置き換え用の文字列に空白が入っていたら "" で囲む
            if (rep.find(' ') != std::wstring::npos)
            {
                rep = Quote(rep);
            }

            result.replace(n, consumedChars, rep);
            nb = n + rep.length();
        }

        return result;
    }

}