﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "PopupMenu.h"

namespace bearmini
{

    ///
    ///  コンストラクタ
    ///
    ///  @param hWnd このポップアップメニューの親になるウインドウのハンドル
    ///
    PopupMenu::PopupMenu(::HWND hwndParent) :
        m_hwndParent(hwndParent),
        m_hMenu(0)
    {
        m_hMenu = ::CreatePopupMenu();

        ::HMENU hMainMenu = ::GetMenu(hwndParent);
        ::AppendMenuW(hMainMenu, MF_STRING | MF_POPUP, (::UINT_PTR)m_hMenu, L"Dummy"); 
    }


    ///
    ///  デストラクタ
    ///
    PopupMenu::~PopupMenu() throw()
    {
        ::DestroyMenu(m_hMenu);
    }


    ///
    ///  ポップアップメニューに登録されている項目の数を取得します。
    ///
    ///  @return   ポップアップメニューに登録されている項目の数
    ///
    int PopupMenu::GetItemCount()
    {
        return ::GetMenuItemCount(m_hMenu);
    }


    ///
    ///  caption で指定された項目をポップアップメニューに追加します。
    ///
    ///  @param[in] id      追加する項目の ID。この項目が選択されたとき、この ID が返ります。
    ///  @param[in] caption 追加する項目のキャプション文字列。ポップアップメニューに表示されます。
    ///
    void PopupMenu::AppendItem(int id, const std::wstring& caption)
    {
        ::BOOL ok = ::AppendMenuW(m_hMenu, MF_ENABLED | MF_STRING, id, caption.c_str());
        if (!ok)
        {
            throw std::exception("AppendMenu() failed.");
        }
    }


    ///
    ///  caption で指定された項目をポップアップメニューに追加します。（チェック印付き）
    ///
    ///  @param[in] id      追加する項目の ID。この項目が選択されたとき、この ID が返ります。
    ///  @param[in] caption 追加する項目のキャプション文字列。ポップアップメニューに表示されます。
    ///
    void PopupMenu::AppendCheckedItem(int id, const std::wstring& caption)
    {
        ::BOOL oK = ::AppendMenuW(m_hMenu, MF_ENABLED | MF_STRING | MF_CHECKED, id, caption.c_str());
        if (!oK)
        {
            throw std::exception("AppendMenu() failed.");
        }
    }

    ///
    ///  caption で指定された項目をポップアップメニューに追加します。（ビットマップ付き）
    ///
    ///  @param[in] id      追加する項目の ID。この項目が選択されたとき、この ID が返ります。
    ///  @param[in] caption 追加する項目のキャプション文字列。ポップアップメニューに表示されます。
    ///  @param[in] hbmp    追加する項目のビットマップのハンドル。
    ///
    void PopupMenu::AppendItem(int id, const std::wstring& caption, ::HBITMAP hbmp)
    {
        ::MENUITEMINFOW mii;
        mii.cbSize = sizeof(mii);
        mii.fMask = MIIM_BITMAP | MIIM_STRING | MIIM_ID;
        mii.fState = MFS_ENABLED;
        mii.wID = id;
        mii.dwTypeData = (::LPWSTR)(caption.c_str());
        mii.cch = (::UINT)caption.length();
        mii.hbmpItem = hbmp;

        int pos = GetItemCount(); // 最後尾に追加されるように
        bool fByPosition = true;
        ::BOOL oK = ::InsertMenuItemW(m_hMenu, pos, fByPosition, &mii);
        if (!oK)
        {
            throw std::exception("InsertMenuItem() failed.");
        }
    }

    ///
    ///  マウスカーソルの現在位置にポップアップメニューを表示し、選ばれたメニュー項目の ID を返します。
    ///
    ///  @return    選択されたメニュー項目の ID。何も選択されなかった場合やエラーの場合は 0 を返します。
    ///  
    int PopupMenu::Show()
    {
        // 親ウインドウを手前に持ってくる（そうしないと、PopupMenu がキーボードで操作できない）
        ::SetForegroundWindow(m_hwndParent);

        ::POINT p;
        ::GetCursorPos(&p);
        return ::TrackPopupMenuEx(m_hMenu, TPM_RETURNCMD, p.x, p.y, m_hwndParent, 0);
    }

}