package noguchi.SentenceLearner;

import java.util.ArrayList;
import java.util.List;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.database.sqlite.SQLiteDatabase;
import android.media.MediaPlayer;
import android.os.Bundle;
import android.os.Handler;
import android.view.ContextMenu;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.ContextMenu.ContextMenuInfo;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.ListView;
import android.widget.SeekBar;
import android.widget.TextView;
import android.widget.Toast;

public class QuestionCreator extends Activity 
	implements Button.OnClickListener,
	Button.OnTouchListener,
	MediaPlayer.OnCompletionListener,
	SeekBar.OnSeekBarChangeListener,
	DialogInterface.OnClickListener,
	QuestionListListener {
	
	// Constants
	private final int REPEAT_INTERVAL = 1000; // Used in showing progress
	private final int REPEAT_INTERVAL2 = 10; // Used in forward/rewind
	private final int DEFAULT_FORWARD_OR_REWIND_SPEED = 500;
	private final int FORWARD_OR_REWIND_ACCEL = 10;
	
	// DB object
	private DatabaseHelper helper;
	private SQLiteDatabase db;
	
	// GUI components
	ListView listView; // Question List
	
	private Button rewindButton = null;
	private Button playButton = null;
	private Button forwardButton = null;
	private Button addButton = null;
	private Button undoButton = null;
	
	private SeekBar seekBar = null;
	private boolean isSeekBarTouched = false;
	
	private TextView currentTimeTextView = null;
	
	// Sound Player
	private MediaPlayer mplayer;
	private boolean isPlay;
	private boolean isPause;
	private SoundItem sound;
	private int length;
	private int forwardOrRewindSpeed = 0;
	
	// Threads which are used for forward/rewind/progress
    private Handler handler = new Handler();
    private Runnable runnable1; // Used in showing progress
    private Runnable runnable2; // Used in forward/rewind
    
    // Manage question items
    private List<QuestionItem> list;
    private QuestionListAdapter adapter;
    private int addStatus = 0;
    private int questionIndex = 0;
    private int initialQuestionCount = 0;
    private QuestionItem lastItem;
    
    // constants for Intent
    private static final int QUESTIONEDITOR = 0;
    
	// Context menu
	private final static int CONTEXT_ITEM_ID1 = (Menu.FIRST + 1);
	private final static int CONTEXT_ITEM_ID2 = (Menu.FIRST + 2);
	private final static int CONTEXT_ITEM_ID3 = (Menu.FIRST + 3);
	private int selectedPos;
	
	// For menu
	private static final int MENU_ID_MENU1 = (Menu.FIRST + 4);
	
    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.questioncreator);
        
		// Create the database (if not exist yet)
		helper = new DatabaseHelper(this);
		db = helper.getReadableDatabase();
		
		// Create GUI component objects
		
		listView = (ListView)findViewById(R.id.QuestionListView);

		rewindButton = (Button)findViewById(R.id.btnRewind);
		playButton = (Button)findViewById(R.id.btnPlay);
		forwardButton = (Button)findViewById(R.id.btnForward);
		addButton = (Button)findViewById(R.id.btnAdd);
		undoButton = (Button)findViewById(R.id.btnUndo);
		
		seekBar = (SeekBar)findViewById(R.id.SeekBar);
		currentTimeTextView = (TextView)findViewById(R.id.CurrentTime);
		
		// Get parameters
		sound = new SoundItem();
		Bundle extras = getIntent().getExtras();		
		if (extras != null)
		{
			sound.setName(extras.getString("NAME"));
			sound.setFileName(extras.getString("FILENAME"));			
		}
		
		// Initialize sound object
		mplayer = new MediaPlayer();
		isPlay = false;
    	playButton.setText("▶");
    	seekBar.setProgress(0);
    	currentTimeTextView.setText("0:00 / 0:00");
		
    	// Initialize ItemList
    	list = new ArrayList<QuestionItem>(); // In case getQuestions failed, assign the list at first
    	try 
    	{
    		list = helper.getQuestions(db, sound.getFileName(), 0);
    	}
    	catch (Exception ex)
    	{
            Toast.makeText(this, R.string.DBFailed, Toast.LENGTH_LONG).show();
    	}    	
    	adapter = new QuestionListAdapter(getApplicationContext(),list, this);
    	listView.setAdapter(adapter);

    	initialQuestionCount = list.size();
		addStatus = 0;
    	questionIndex = initialQuestionCount;    	
		
		// Register Listeners
		addButton.setOnClickListener(this);
		undoButton.setOnClickListener(this);
		seekBar.setOnSeekBarChangeListener(this);
		
		playButton.setOnTouchListener(this);
        forwardButton.setOnTouchListener(this);
        rewindButton.setOnTouchListener(this);
        
		// Register context menu
		registerForContextMenu(listView);
		
        // Set the thread which is used to show the progress
        runnable1 = new Runnable() {
            @Override
            public void run() {            	
                countProgress();           
                handler.postDelayed(this, REPEAT_INTERVAL);
            }
        };
        
        // Set the thread which is used in forward/rewind
        runnable2 = new Runnable() {
            @Override
            public void run() {            	
                forwardOrRewindPosition();    
                countProgress();
                handler.postDelayed(this, REPEAT_INTERVAL2);
            }
        };
    }
    
    private int InitializeSound(){
    	
		try
		{
			mplayer.setDataSource(sound.getFileName());
			mplayer.prepare();
			mplayer.start();
			
			mplayer.setOnCompletionListener(this);
			int res = mplayer.getDuration();

			playButton.setText("■");
	    	seekBar.setProgress(0);
	    	currentTimeTextView.setText("0:00 / " + QuestionItem.GetTimeString(res));
	    	seekBar.setMax(res);
			isPlay = true;
			isSeekBarTouched = false;
			return res;
		}
		catch (Exception ee)
		{
            Toast.makeText(this, R.string.palyerInitializefailed, Toast.LENGTH_LONG).show();
			return -1;
		}

    }
    
    private void StopSound(){
    	playButton.setText("▶");
    	seekBar.setProgress(0);
    	currentTimeTextView.setText("0:00 / 0:00");

    	mplayer.stop();
    	mplayer.reset();
    	isPlay = false;
    	isPause = false;
    }
    
    private void pauseSound()
    {
    	if (!isPlay) return;
    	
    	if (isPause)
    	{
    		mplayer.start();
    		isPause = false;
    		playButton.setText("||");
    	}
    	else
    	{
    		mplayer.pause();
    		isPause = true;
    		playButton.setText("▶");
    	}
    }

    private void countProgress()
    {
    	if (isPlay)
    	{
    		int pos = mplayer.getCurrentPosition(); 
			if (!isSeekBarTouched) seekBar.setProgress(pos);
			currentTimeTextView.setText(QuestionItem.GetTimeString(pos) + " / " + QuestionItem.GetTimeString(length));    		
    	}
    }
    
	private void forwardOrRewindPosition()
	{
		if (isPlay)
		{
			int pos = mplayer.getCurrentPosition();
			pos += forwardOrRewindSpeed;
			forwardOrRewindSpeed += (forwardOrRewindSpeed > 0 ? FORWARD_OR_REWIND_ACCEL:-FORWARD_OR_REWIND_ACCEL);
			if (pos > 0 || pos < length)
			{
				mplayer.seekTo(pos);
			}
		}
	}
    
	@Override
	public void OnQuestionListClick(QuestionItem item) {
		// TODO Auto-generated method stub
		Intent intent = new Intent(QuestionCreator.this, QuestionEditor.class);
		intent.putExtra("NAME", item.getName());
		intent.putExtra("ANSWER", item.getAnswer());
		intent.putExtra("SOUNDNAME", item.getSoundName());
		intent.putExtra("QSTARTTIME",item.getQuestionStartTime());
		intent.putExtra("ASTARTTIME",item.getAnswerStartTime());
		intent.putExtra("ENDTIME",item.getEndTime());
		
		startActivityForResult(intent,QUESTIONEDITOR);
	}

	@Override
	public void onProgressChanged(SeekBar arg0, int progress, boolean fromUser) {
		if (fromUser && isPlay)
		{
			int pos = seekBar.getProgress();
			mplayer.seekTo(pos);			
		}
	}

	@Override
	public void onStartTrackingTouch(SeekBar arg0) {
		isSeekBarTouched = true;
	}

	@Override
	public void onStopTrackingTouch(SeekBar seekBar) {
		isSeekBarTouched = false;
	}

	@Override
	public void onCompletion(MediaPlayer mp) {
		StopSound();
	}

	@Override
	public boolean onTouch(View v, MotionEvent event) {
		if(event.getAction() == MotionEvent.ACTION_DOWN) {
			if (v == forwardButton)
			{
				forwardOrRewindSpeed = DEFAULT_FORWARD_OR_REWIND_SPEED;
				handler.postDelayed(runnable2, REPEAT_INTERVAL2);
			}
			else if(v == rewindButton)
			{
				forwardOrRewindSpeed = -DEFAULT_FORWARD_OR_REWIND_SPEED;
				handler.postDelayed(runnable2, REPEAT_INTERVAL2);
			}
			else if (v == playButton)
			{
				if (!isPlay)
				{
					length = InitializeSound();
				}
				else
				{
					pauseSound();
				}
				handler.postDelayed(runnable1, REPEAT_INTERVAL);
			}
		}
		else if(event.getAction() == MotionEvent.ACTION_UP)
		{
			if (v == forwardButton || v == rewindButton)
				handler.removeCallbacks(runnable2);
			else
			{
				pauseSound();
				handler.removeCallbacks(runnable1);
			}
		}
		return false;
	}

	@Override
	public void onClick(View arg0) {
		if (arg0 == addButton)
		{
			int pos;
			if (isPlay)
				pos =  mplayer.getCurrentPosition();
			else
				pos = 0;
			
			switch (addStatus){
			case 0:
				questionIndex ++;
				lastItem = new QuestionItem();
				lastItem.setName("Question " + questionIndex);
				lastItem.setSoundName(sound.getName());
				lastItem.setSoundFileName(sound.getFileName());
				lastItem.setQuestionStartTime(pos);
				list.add(lastItem);
				break;
			case 1:
				lastItem.setAnswerStartTime(pos);
				break;
			case 2:
				lastItem.setEndTime(pos);				
			}
			adapter.notifyDataSetChanged();
			listView.setSelection(listView.getCount());
			
			addStatus += 1;
			if (addStatus > 2) addStatus = 0;
		}
		else if (arg0 == undoButton && list.size()>0 && lastItem != null)
		{
			addStatus -= 1;
			if (addStatus < 0) addStatus = 2;
			switch (addStatus){
			case 0:
				questionIndex --;
				int last = list.size()-1;
				list.remove(last);
				if (last > 0)
					lastItem = list.get(last - 1);
				else
					lastItem = null;
				break;
			case 1:
				lastItem.setAnswerStartTime(-1);
				break;
			case 2:
				lastItem.setEndTime(-1);				
			}
			adapter.notifyDataSetChanged();
			listView.setSelection(listView.getCount());			
		}
	}
	
	@Override
	public boolean dispatchKeyEvent(KeyEvent event) {
		if (event.getAction() == KeyEvent.ACTION_UP) {
			switch (event.getKeyCode()) {
			case KeyEvent.KEYCODE_BACK: // Back key
				saveAndExit();
				return true;
			default:
			}
		}
		return super.dispatchKeyEvent(event);
	}
	
	
	private void saveAndExit(){
		if (true) //TODO: Check if something is modified
		{
			AlertDialog.Builder alertDialogBuilder = new AlertDialog.Builder(this);
			alertDialogBuilder.setTitle(getResources().getString(R.string.SaveChanges));
			alertDialogBuilder.setMessage(getResources().getString(R.string.DoYouWantToSaveTheChanges));
			alertDialogBuilder.setPositiveButton(getResources().getString(R.string.Yes),this);
			alertDialogBuilder.setNegativeButton(getResources().getString(R.string.No),this);
			alertDialogBuilder.setCancelable(true);
			AlertDialog alertDialog = alertDialogBuilder.create();
			alertDialog.show();
			return;
		}
	}
	
	@Override
	public void onClick(DialogInterface dialog, int which) {
		if (which == -1) // When Yes is clicked
		{
			Intent intent = new Intent();
			try {
				helper.updateQuestionsList(sound.getFileName(), list, db);
				setResult(RESULT_OK, intent);
				Toast.makeText(this, getResources().getString(R.string.ItemAdded), Toast.LENGTH_SHORT).show();					
			}
			catch (Exception ex)
			{
				Toast.makeText(this, getResources().getString(R.string.DBFailed), Toast.LENGTH_SHORT).show();
			}
			finally
			{
				finish();
			}
		}
		else
		{
			finish();
		}
	}
	
	// Menu
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Add menus
        menu.add(Menu.NONE, MENU_ID_MENU1, Menu.NONE, getResources().getString(R.string.DeleteAll));
        return super.onCreateOptionsMenu(menu);
    }
    
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
    	
        AlertDialog.Builder alertDialogBuilder = new AlertDialog.Builder(this);
        
    	switch (item.getItemId()) {
		case MENU_ID_MENU1:
			// Delete all
        	alertDialogBuilder.setTitle(getResources().getString(R.string.Delete));
        	alertDialogBuilder.setMessage(getResources().getString(R.string.DoYouReallyWantToDeleteAll));
        	alertDialogBuilder.setPositiveButton(getResources().getString(R.string.Yes),
        			new DialogInterface.OnClickListener() {
        		@Override
        		public void onClick(DialogInterface dialog, int which) {
            		list.clear();
            		questionIndex = 0;
            		adapter.notifyDataSetChanged();
        		}
        	});
        	alertDialogBuilder.setNegativeButton(getResources().getString(R.string.No),
        			new DialogInterface.OnClickListener() {
        		@Override
        		public void onClick(DialogInterface dialog, int which) {
        			// Do nothing
        		}
        	});
        	alertDialogBuilder.setCancelable(true);
        	AlertDialog alertDialog2 = alertDialogBuilder.create();
        	alertDialog2.show();	
			break;
		default:
			break;
		}
    	return true;
    }
    

    // Context menu
    @Override
    public void onCreateContextMenu(ContextMenu menu, View v,
    		ContextMenuInfo menuInfo) {
    	super.onCreateContextMenu(menu, v, menuInfo);
        AdapterView.AdapterContextMenuInfo info = (AdapterView.AdapterContextMenuInfo)menuInfo;
        String headerText;
        if (list.get(info.position).getName().length() < 10 ) {
        	headerText = list.get(info.position).getName();
        } else {
        	headerText = (String) list.get(info.position).getName().subSequence(0, 10);
        }
	    menu.setHeaderTitle(headerText);
	    menu.add(0, CONTEXT_ITEM_ID1, 0, getResources().getString(R.string.Edit));
	    menu.add(0, CONTEXT_ITEM_ID2, 0, getResources().getString(R.string.Delete));
	    menu.add(0, CONTEXT_ITEM_ID3, 0, getResources().getString(R.string.DeleteAll));
	    selectedPos = info.position;
    }
    
    @Override
    public boolean onContextItemSelected(MenuItem item) {

        AlertDialog.Builder alertDialogBuilder = new AlertDialog.Builder(this);
        Intent intent;
        
        QuestionItem qItem = list.get(selectedPos);
        
    	switch (item.getItemId()) {
		case CONTEXT_ITEM_ID1:
			intent = new Intent(QuestionCreator.this, QuestionEditor.class);
			intent.putExtra("NAME", qItem.getName());
			intent.putExtra("ANSWER", qItem.getAnswer());
			intent.putExtra("SOUNDNAME", qItem.getSoundName());
			intent.putExtra("QSTARTTIME",qItem.getQuestionStartTime());
			intent.putExtra("ASTARTTIME",qItem.getAnswerStartTime());
			intent.putExtra("ENDTIME",qItem.getEndTime());
			startActivityForResult(intent,QUESTIONEDITOR);
			break;
		case CONTEXT_ITEM_ID2:
			// Delete item
        	alertDialogBuilder.setTitle(getResources().getString(R.string.Delete));
        	alertDialogBuilder.setMessage(getResources().getString(R.string.DoYouReallyWantToDeleteThisItem));
        	alertDialogBuilder.setPositiveButton(getResources().getString(R.string.Yes),
        			new DialogInterface.OnClickListener() {
        		@Override
        		public void onClick(DialogInterface dialog, int which) {
            		list.remove(selectedPos);
            		adapter.notifyDataSetChanged();
        		}
        	});
        	alertDialogBuilder.setNegativeButton(getResources().getString(R.string.No),
        			new DialogInterface.OnClickListener() {
        		@Override
        		public void onClick(DialogInterface dialog, int which) {
        			// Do nothing
        		}
        	});
        	alertDialogBuilder.setCancelable(true);
        	AlertDialog alertDialog = alertDialogBuilder.create();
        	alertDialog.show();
			break;
		case CONTEXT_ITEM_ID3:
			// Delete all
        	alertDialogBuilder.setTitle(getResources().getString(R.string.Delete));
        	alertDialogBuilder.setMessage(getResources().getString(R.string.DoYouReallyWantToDeleteAll));
        	alertDialogBuilder.setPositiveButton(getResources().getString(R.string.Yes),
        			new DialogInterface.OnClickListener() {
        		@Override
        		public void onClick(DialogInterface dialog, int which) {
            		list.clear();
            		adapter.notifyDataSetChanged();
        		}
        	});
        	alertDialogBuilder.setNegativeButton(getResources().getString(R.string.No),
        			new DialogInterface.OnClickListener() {
        		@Override
        		public void onClick(DialogInterface dialog, int which) {
        			// Do nothing
        		}
        	});
        	alertDialogBuilder.setCancelable(true);
        	AlertDialog alertDialog2 = alertDialogBuilder.create();
        	alertDialog2.show();			
		default:
			break;
		}
    	return true;
    }
}
