const { app, ipcMain, protocol, session, BrowserWindow, BrowserView, Menu, nativeImage, clipboard, dialog, Notification } = require('electron');
const path = require('path');
const fs = require('fs');
const url = require('url');
const os = require('os');

const localShortcut = require("electron-localshortcut");

const Config = require('electron-store');
const config = new Config({
    defaults: {
        design: {
            homeButton: false,
            darkTheme: false,
            theme: 'default'
        },
        homePage: {
            defaultPage: 'my://newtab',
            defaultEngine: 'Google',
            searchEngines: [
                {
                    name: 'Google',
                    url: 'https://www.google.com/search?q=%s'
                },
                {
                    name: 'Bing',
                    url: 'https://www.bing.com/search?q=%s'
                },
                {
                    name: 'Yahoo! Japan',
                    url: 'https://search.yahoo.co.jp/search?p=%s'
                },
                {
                    name: 'goo',
                    url: 'https://search.goo.ne.jp/web.jsp?MT=%s'
                },
                {
                    name: 'Google Translate',
                    url: 'https://translate.google.com/?text=%s'
                },
                {
                    name: 'Youtube',
                    url: 'https://www.youtube.com/results?search_query=%s'
                },
                {
                    name: 'Twitter',
                    url: 'https://www.twitter.com/search?q=%s'
                },
                {
                    name: 'GitHub',
                    url: 'https://github.com/search?q=%s'
                }
            ]
        },
        adBlocker: true,
        window: {
            isCustomTitlebar: true,
            isMaximized: false,
            bounds: {
                width: 1100,
                height: 680
            }
        }
    },
});

const Datastore = require('nedb');
let db = {};

db.history = new Datastore({
    filename: path.join(app.getPath('userData'), 'Files', 'History.db'),
    autoload: true,
    timestampData: true
});

const { loadFilters, updateFilters, removeAds } = require('./AdBlocker');
const WindowManager = require('./WindowManager');
const windowManager = new WindowManager();

let loginCallback;
let mainWindow;
let subWindow;

getBaseWindow = (width = 1100, height = 680, minWidth = 320, minHeight = 600, x, y, frame = false) => {
    return new BrowserWindow({
        width, height, minWidth, minHeight, x, y, 'titleBarStyle': 'hidden', frame, fullscreenable: true,
        webPreferences: {
            nodeIntegration: true,
            webviewTag: true,
            plugins: true,
            experimentalFeatures: true,
            contextIsolation: false,
        }
    });
}

/*
showFilterDialog = () => {
    if (!hasFile()) {
        const dialogResult = dialog.showMessageBox({
            type: 'info',
            buttons: ['Ok'],
            title: '本当にデータをリセットしてよろしいですか？',
            message: '"続行"を押した場合データのリセット後アプリが再起動します。',
            defaultId: 0,
            cancelId: 1
        });

        if (dialogResult === 0) {
            showFilterDialog();
        }
    }
}
*/

registerProtocols = () => {
    protocol.registerFileProtocol('my', (request, callback) => {
        const parsed = url.parse(request.url);

        if (parsed.hostname.endsWith('.css') || parsed.hostname.endsWith('.js')) {
            return callback({
                path: path.join(app.getAppPath(), 'pages', parsed.hostname),
            });
        } else {
            return callback({
                path: path.join(app.getAppPath(), 'pages', `${parsed.hostname}.html`),
            });
        }
    }, (error) => {
        if (error) console.error('Failed to register protocol: ' + error);
    });
}

module.exports = class Application {
    loadWindow = () => {
        protocol.registerSchemesAsPrivileged([
            { scheme: 'my', privileges: { standard: true, bypassCSP: true, secure: true } }
        ]);

        app.on('ready', () => {
            loadFilters();
            registerProtocols();
            
            Menu.setApplicationMenu(null);

            windowManager.addWindow();
        });

        app.on('window-all-closed', () => {
            if (process.platform !== 'darwin') {
                app.quit();
            }
        });

        app.on('activate', () => {
        });

        app.on('login', (e, webContents, request, authInfo, callback) => {
            e.preventDefault();

            subWindow = getBaseWindow(320, 230, 320, 230);
            // subWindow.setParentWindow(mainWindow);
            subWindow.setMovable(false);
            subWindow.setResizable(false);
            subWindow.setMinimizable(false);
            subWindow.setMaximizable(false);
            const startUrl = process.env.ELECTRON_START_URL || url.format({
                pathname: path.join(__dirname, '/../build/index.html'), // 警告：このファイルを移動する場合ここの相対パスの指定に注意してください
                protocol: 'file:',
                slashes: true,
                hash: '/auth',
            });

            subWindow.loadURL(startUrl);
            loginCallback = callback;
        });

        ipcMain.on('authorization', (event, arg) => {
            loginCallback(arg.username, arg.password);
            subWindow.close();
        });

        ipcMain.on('data-history-get', (e, args) => {
            db.history.find({}).sort({ createdAt: -1 }).exec((err, docs) => {
                e.sender.send('data-history-get', { historys: docs });
            });
        });

        ipcMain.on('data-history-clear', (e, args) => {
            db.history.remove({}, { multi: true });
        });

        ipcMain.on('clear-browsing-data', () => {
            const ses = session.defaultSession;
            ses.clearCache((err) => {
                if (err) log.error(err);
            });

            ses.clearStorageData({
                storages: [
                    'appcache',
                    'cookies',
                    'filesystem',
                    'indexdb',
                    'localstorage',
                    'shadercache',
                    'websql',
                    'serviceworkers',
                    'cachestorage',
                ],
            });

            config.clear();
            db.history.remove({}, { multi: true });
        });
    }

    loadExtension = (id) => {
        const extensionDir = path.resolve(os.homedir(), 'AppData\\Local\\Google\\Chrome\\User Data\\Default\\Extensions');

        const versions = fs.readdirSync(`${extensionDir}/${id}`).sort();
        const version = versions.pop();

        BrowserWindow.addExtension(`${extensionDir}/${id}/${version}`);
    }
};