/*
 * Copyright 2006-2007 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.impl;

import org.seasar.codegen.LanguageDataTypeSelectUtil;
import org.seasar.codegen.element.LangDataType;
import org.seasar.codegen.element.lang.JavaDataType;
import org.seasar.framework.util.StringUtil;

public class JavaDataTypeSelectUtil implements LanguageDataTypeSelectUtil {

    public LangDataType getDataType(String typeName, boolean isNotNull,
            String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        if (typeName.equals("String")) {
            javaDataType.setClassName("String");
            javaDataType.setJavaObj(true);
            if (isNotNull || (!StringUtil.isEmpty(defaultVal))) {
                javaDataType.setDefVal("\"" + defaultVal + "\"");
            }
        } else if (typeName.equals("long")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("long", "Long",
                    isNotNull, defaultVal);
        } else if (typeName.equals("byte[]")) {
            javaDataType = (JavaDataType) getByteArrayDataType(typeName,
                    isNotNull, defaultVal);
        } else if (typeName.equals("Date")) {
            javaDataType = (JavaDataType) getDateDataType(typeName, isNotNull,
                    defaultVal);
        } else if (typeName.equals("BigDecimal")) {
            javaDataType = (JavaDataType) getBigDecimalDataType(typeName,
                    isNotNull, defaultVal);
        } else if (typeName.equals("BigInteger")) {
            javaDataType = (JavaDataType) getBigIntegerDataType(typeName,
                    isNotNull, defaultVal);
        } else if (typeName.equals("double")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("double",
                    "Double", isNotNull, defaultVal);
        } else if (typeName.equals("float")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("float", "Float",
                    isNotNull, defaultVal);
        } else if (typeName.equals("int")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("int", "Integer",
                    isNotNull, defaultVal);
        } else if (typeName.equals("short")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("short", "Short",
                    isNotNull, defaultVal);
        } else if (typeName.equals("Time")) {
            javaDataType = (JavaDataType) getTimeDataType(typeName, isNotNull,
                    defaultVal);
        } else if (typeName.equals("Timestamp")) {
            javaDataType = (JavaDataType) getTimestampDataType(typeName,
                    isNotNull, defaultVal);
        } else if (typeName.equals("Double")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("double",
                    "Double", isNotNull, defaultVal);
        } else if (typeName.equals("Long")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("long", "Long",
                    isNotNull, defaultVal);
        } else if (typeName.equals("boolean")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("boolean",
                    "Boolean", isNotNull, defaultVal.toLowerCase());
        } else if (typeName.equals("Boolean")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("boolean",
                    "Boolean", isNotNull, defaultVal);
        } else if (typeName.equals("byte")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("byte", "Byte",
                    isNotNull, defaultVal);
        } else if (typeName.equals("Objet")) {
            javaDataType = (JavaDataType) getJavaAnyDataType("Object",
                    "Object", isNotNull, defaultVal);
        }

        return javaDataType;
    }

    protected LangDataType getBigIntegerDataType(String typeName,
            boolean isNotNull, String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        javaDataType.addImportName("java.math.BigInteger");
        javaDataType.setClassName("BigInteger");
        if (!StringUtil.isEmpty(defaultVal)) {
            javaDataType.setDefVal("new BigInteger(" + defaultVal + ")");
        }
        javaDataType.setJavaObj(true);
        return javaDataType;
    }

    protected LangDataType getBigDecimalDataType(String typeName,
            boolean isNotNull, String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        javaDataType.addImportName("java.math.BigDecimal");
        javaDataType.setClassName("BigDecimal");
        if (!StringUtil.isEmpty(defaultVal)) {
            javaDataType.setDefVal("new BigDecimal(" + defaultVal + ")");
        }
        javaDataType.setJavaObj(true);
        return javaDataType;
    }

    protected LangDataType getByteArrayDataType(String typeName,
            boolean isNotNull, String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        javaDataType.setClassName("byte[]");
        javaDataType.setJavaObj(false);
        return javaDataType;
    }

    protected LangDataType getJavaAnyDataType(String primiName, String ObjName,
            boolean isNotNull, String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        if (isNotNull) {
            javaDataType.setClassName(primiName);
            if (!StringUtil.isEmpty(defaultVal)) {
                javaDataType.setDefVal(defaultVal);
            }
            javaDataType.setJavaObj(false);
        } else {
            javaDataType.setClassName(ObjName);
            if (!StringUtil.isEmpty(defaultVal)) {
                javaDataType.setDefVal("new " + ObjName + "(" + defaultVal
                        + ")");
            }
            javaDataType.setJavaObj(true);
        }
        return javaDataType;
    }

    protected LangDataType getDateDataType(String typeName, boolean isNotNull,
            String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        javaDataType.setClassName("Date");
        javaDataType.addImportName("java.util.Date");
        if (!StringUtil.isEmpty(defaultVal)) {
            if (defaultVal.startsWith("MAX")) {
                javaDataType.setDefVal("new Date(9999,11,31,23,59,59)");
            } else if (defaultVal.startsWith("MIN")) {
                javaDataType.setDefVal("new Date(1000,0,1,1,1,1)");
            } else if (defaultVal.startsWith("CUR")
                    || defaultVal.startsWith("NOW")) {
                javaDataType.setDefVal("new Date()");
            } else {
                javaDataType.setDefVal(defaultVal);
            }
        }
        javaDataType.setJavaObj(true);
        return javaDataType;
    }

    protected LangDataType getTimeDataType(String typeName, boolean isNotNull,
            String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        javaDataType.setClassName("Time");
        javaDataType.addImportName("java.sql.Time");
        if (!StringUtil.isEmpty(defaultVal)) {
            if (defaultVal.startsWith("MAX")) {
                javaDataType.setDefVal("new Time(23,59,59)");
            } else if (defaultVal.startsWith("MIN")) {
                javaDataType.setDefVal("new Time(0,0,0)");
            } else if (defaultVal.startsWith("CUR")
                    || defaultVal.startsWith("NOW")) {
                javaDataType.addImportName("java.util.GregorianCalendar");
                javaDataType
                        .setDefVal("new Time(GregorianCalendar.getInstance().getTimeInMillis())");
            } else {
                javaDataType.setDefVal(defaultVal);
            }
        }
        javaDataType.setJavaObj(true);
        return javaDataType;
    }

    protected LangDataType getTimestampDataType(String typeName,
            boolean isNotNull, String defaultVal) {
        JavaDataType javaDataType = new JavaDataType();
        javaDataType.setClassName("Timestamp");
        javaDataType.addImportName("java.sql.Timestamp");
        if (!StringUtil.isEmpty(defaultVal)) {
            if (defaultVal.startsWith("MAX")) {
                javaDataType.setDefVal("new Timestamp(9999,11,31,23,59,59,99)");
            } else if (defaultVal.startsWith("MIN")) {
                javaDataType.setDefVal("new Timestamp(1000,0,1,1,0,0,0)");
            } else if (defaultVal.startsWith("CUR")
                    || defaultVal.startsWith("NOW")) {
                javaDataType.addImportName("java.util.GregorianCalendar");
                javaDataType
                        .setDefVal("new Timestamp(GregorianCalendar.getInstance().getTimeInMillis())");
            } else {
                javaDataType.setDefVal(defaultVal);
            }
        }
        javaDataType.setJavaObj(true);
        return javaDataType;
    }

}
