require 'test_helper'

class AchievementsControllerTest < ActionController::TestCase
  include Devise::TestHelpers

  setup do
    @user = FactoryGirl.create :user
    @achievement = FactoryGirl.create :achievement, :user_id => @user.id
    alt_user = FactoryGirl.create :user
    @alt_achievement = FactoryGirl.create :achievement, :user_id => alt_user.id
    sign_in @user
  end

  test "should get index" do
    get :index
    assert_response :success
    assert_not_nil assigns(:achievements)
  end

  test "should get new" do
    get :new
    assert_response :success
  end

  test "should create achievement" do
    assert_difference('Achievement.count') do
      post :create, achievement: { result: @achievement.result }
    end

    assert_redirected_to achievement_path(assigns(:achievement))
  end

  test "should normalize result" do
    assert_difference('Achievement.count') do
      post :create, :achievement => { :task_id => @alt_achievement.task_id, :result => '1,    2' }
    end
    a = assigns :achievement
    assert_not_nil a
    assert_equal '1,2', a.result
  end

  test "should show achievement" do
    get :show, id: @achievement
    assert_response :success
  end

  test "should get edit" do
    get :edit, id: @achievement
    assert_response :success
  end

  test "should update achievement" do
    put :update, id: @achievement, achievement: { result: @achievement.result }
    assert_redirected_to achievement_path(assigns(:achievement))
  end

  test "should destroy achievement" do
    assert_difference('Achievement.count', -1) do
      delete :destroy, id: @achievement
    end

    assert_redirected_to achievements_path
  end

  test "should not show another user's achievement" do
    get :show, :id => @alt_achievement.id
    assert_redirected_to achievements_path
    assert_not_nil flash.alert
  end

  test "should not edit another user's achievement" do
    get :edit, :id => @alt_achievement.id
    assert_redirected_to achievements_path
    assert_not_nil flash.alert
  end

  test "should not update another user's achievement" do
    put :update, :id => @alt_achievement.id, :achievement => { :result => @achievement.result }
    assert_redirected_to achievements_path
    assert_not_nil flash.alert
  end

  test "should not create achievement with the same task" do
    assert_difference('Achievement.count', 0) do
      post :create, :achievement => { :task_id => @achievement.task_id, :result => @achievement.result }
    end
  end

  test "should merge results" do
    xs = @achievement.results
    ys = Array.new(xs.size) { 0 }
    ys[0] = 10000
    put :update, :id => @achievement, :achievement => { :result => ys.join(',') }
    updated_achievement = assigns :achievement
    assert_not_nil updated_achievement
    assert_equal Achievement.merge(xs, ys), updated_achievement.results
  end

  test "reject invalid merge" do
    xs = @achievement.results
    ys = xs + [0]
    put :update, :format => :json, :id => @achievement, :achievement => { :result => ys.join(',') }
    errors = JSON.parse @response.body
    assert errors.has_key?('result')
    assert_match_any /merge/, errors['result']
  end

  test "reject invalid format" do
    assert_difference('Achievement.count', 0) do
      post :create, :format => :json, :achievement => { :task_id => @alt_achievement.task_id, :result => 'a' }
    end
    errors = JSON.parse @response.body
    assert errors.has_key?('result')
    assert_match_any /element/, errors['result']
  end

  test "reject empty result" do
    assert_difference('Achievement.count', 0) do
      post :create, :format => :json, :achievement => { :task_id => @alt_achievement.task_id, :result => '' }
    end
    errors = JSON.parse @response.body
    assert errors.has_key?('result')
    assert_match_any /empty/, errors['result']
  end

  test "show achievement by task_id" do
    get :show, :task_id => @achievement.task
    assert_response :success
    achievement = assigns :achievement
    assert_not_nil achievement
    assert_equal @achievement.id, achievement.id
  end

  test "get new by task_id" do
    get :new, :task_id => @achievement.task
    assert_response :success
    assert_select '#achievement_task_id option[selected][value=?]', @achievement.task.id
  end

  test "get edit by task_id" do
    get :edit, :task_id => @achievement.task
    assert_response :success
    achievement = assigns :achievement
    assert_not_nil achievement
    assert_equal @achievement.id, achievement.id
  end

  test "destroy achievement by task_id" do
    assert_difference('Achievement.count', -1) do
      delete :destroy, :task_id => @achievement.task
    end
  end

  test "raise AR::RecordNotFound when given task_id's achievement does not exist" do
    task = FactoryGirl.create :task
    assert_raises(ActiveRecord::RecordNotFound) { get :show, :task_id => task.id }
  end
end
