class AchievementsController < ApplicationController
  class Unauthorized < StandardError; end

  rescue_from Unauthorized, :with => :unauthorized

  # GET /achievements
  # GET /achievements.json
  def index
    if teacher_signed_in?
      @achievements = current_teacher.achievements
    elsif user_signed_in?
      @achievements = current_user.achievements
    else
      authenticate_user!
    end

    respond_to do |format|
      format.html # index.html.erb
      format.json { render json: @achievements }
    end
  end

  # GET /achievements/1
  # GET /achievements/1.json
  def show
    @achievement = find_achievement params
    if teacher_signed_in?
      check_teacher!
    elsif user_signed_in?
      check_user!
    else
      authenticate_user!
    end

    respond_to do |format|
      format.html # show.html.erb
      format.json { render json: @achievement }
    end
  end

  # GET /achievements/new
  # GET /achievements/new.json
  def new
    authenticate_user!
    @achievement = Achievement.new
    if params[:task_id]
      @achievement.task_id = params[:task_id]
    end

    respond_to do |format|
      format.html # new.html.erb
      format.json { render json: @achievement }
    end
  end

  # GET /achievements/1/edit
  def edit
    authenticate_user!
    @achievement = find_achievement params
    check_user!
  end

  # POST /achievements
  # POST /achievements.json
  def create
    authenticate_user!
    @achievement = Achievement.new(params[:achievement])
    @achievement.user = current_user

    respond_to do |format|
      if @achievement.save
        format.html { redirect_to @achievement, notice: 'Achievement was successfully created.' }
        format.json { render json: @achievement, status: :created, location: @achievement }
      else
        format.html { render action: "new" }
        format.json { render json: @achievement.errors, status: :unprocessable_entity }
      end
    end
  end

  # PUT /achievements/1
  # PUT /achievements/1.json
  def update
    authenticate_user!
    begin
      @achievement = find_achievement params
    rescue ActiveRecord::RecordNotFound => e
      if @achievement.nil? and params.has_key?(:task_id)
        # create new achievement
        @achievement = Achievement.new
        @achievement.user_id = current_user.id
        @achievement.task_id = params[:task_id]
      else
        raise e
      end
    end
    check_user!

    respond_to do |format|
      if @achievement.update_attributes(params[:achievement])
        format.html { redirect_to @achievement, notice: 'Achievement was successfully updated.' }
        format.json { render :json => @achievement }
      else
        format.html { render action: "edit" }
        format.json { render json: @achievement.errors, status: :unprocessable_entity }
      end
    end
  end

  # DELETE /achievements/1
  # DELETE /achievements/1.json
  def destroy
    authenticate_user!
    @achievement = find_achievement params
    check_user!
    @achievement.destroy

    respond_to do |format|
      format.html { redirect_to achievements_url }
      format.json { head :no_content }
    end
  end

  private

  def check_user!
    if @achievement.user_id != current_user.id
      raise Unauthorized
    end
  end

  def check_teacher!
    if @achievement.task.lecture.teacher.id != current_teacher.id
      raise Unauthorized
    end
  end

  def unauthorized(exception)
    error = I18n.t('achievement.unauthorized')
    respond_to do |format|
      format.html { redirect_to({ :action => 'index' }, :alert => error) }
      format.json { render :status => 401, :json => { :error => error } }
    end
  end

  def find_achievement(params)
    if params[:id]
      Achievement.find(params[:id])
    elsif params[:task_id]
      Task.find(params[:task_id]).achievement(current_user)
    end
  end
end
