# -*- coding: utf-8 -*-

import calendar
import os
import pkg_resources
import re
import time

from datetime import datetime

from trac.core import *
from trac.perm import IPermissionRequestor
from trac.resource import IResourceManager, Resource, ResourceNotFound
from trac.search import ISearchSource, search_to_sql, shorten_result
from trac.ticket import Ticket
from trac.timeline import ITimelineEventProvider
from trac.util.datefmt import to_timestamp, utc
from trac.util.presentation import Paginator
from trac.util.translation import _
from trac.web.api import IRequestHandler, IRequestFilter, ITemplateStreamFilter
from trac.web.chrome import add_link, add_stylesheet, add_script, add_ctxtnav, prevnext_nav, \
                        INavigationContributor, ITemplateProvider
from trac.wiki import wiki_to_oneliner

from genshi.filters.transform import Transformer
from genshi.template import MarkupTemplate

from attachment import MailArchiveAttachment
from model import Mail, MailFinder
from util import *

ml_permission_map = {}

class MailArchiveModule(Component):
    
    implements(ITemplateProvider, IRequestFilter, ITemplateStreamFilter,
        INavigationContributor, IPermissionRequestor, IResourceManager,
        IRequestHandler)

    FIELD_XPATH = 'div[@id="ticket"]/table[@class="properties"]/td[@headers="h_%s"]/text()'
    
    permission_actions = [('MAILARCHIVE_ADMIN', ['MAILARCHIVE_VIEW'])]
             
    def __init__(self):
        db = self.env.get_db_cnx()
        cursor = db.cursor()
        cursor.execute('SELECT DISTINCT mlid FROM mailarc_category ORDER BY mlid')
        ml_actions = []
        for row in cursor:
            ml_action = get_mlperm_action(row[0])
            ml_actions.append(ml_action)
            ml_permission_map.update({ml_action: row[0]})
        
        self.permission_actions.extend(ml_actions)
        if len(ml_actions) > 0:
            self.permission_actions.append(('MAILARCHIVE_VIEW', ml_actions))
        else:
            self.permission_actions.extend('MAILARCHIVE_VIEW')
             
    # ITemplateProvider methods
    def get_htdocs_dirs(self):
        return [('mailarchive',pkg_resources.resource_filename(__name__, 'htdocs'))]

    def get_templates_dirs(self):
        return [pkg_resources.resource_filename(__name__, 'templates')]
    
    # IRequestFilter methods
    def pre_process_request(self, req, handler):
        return handler
        
    def post_process_request(self, req, template, data, content_type):
        #チケット表示画面時はfilter_streamで加工できるように
        #dataに追加しておく
        if data and req.path_info.startswith('/ticket/'):
            ticket = data.get('ticket')
            if ticket is None:
                return template, data, content_type
                
            if not isinstance(ticket, Ticket):
                ticket = Ticket(self.env, ticket)
            mail_id_plain = ticket['mail_id']
            mail_ids = []
            if mail_id_plain is None:
                mail_ids = []
            else:
                for mail_id in mail_id_plain.split(','):
                    try:
                        id = int(mail_id.strip())
                        mail_ids.append(id)
                    except ValueError:
                        continue
                    
            if len(mail_ids) > 0:
                data['mailarchive'] = {
                    'mail_ids': mail_ids,
                    'link': linkify_ids(self.env, req, mail_ids),
                }
        return template, data, content_type

    # ITemplateStreamFilter methods
    def filter_stream(self, req, method, filename, stream, data):
        if 'mailarchive' in data:
            #cssを追加
            add_stylesheet(req, 'common/css/report.css')
            add_stylesheet(req, 'mailarchive/css/mailarchive.css')
            add_script(req, 'mailarchive/js/mailarchive.js')
            
            #mail_idをハイパーリンクに置き換える
            link =  data['mailarchive']['link']
            stream |= Transformer(self.FIELD_XPATH % 'mail_id').replace(link)
            
            #関連メールのスレッド表示を追加する
            thread_stream = show_thread(self.env, req, data['mailarchive']['mail_ids'])
            THREAD_PATH = '//div[@id="ticket"]'
            stream |= Transformer(THREAD_PATH).after(thread_stream)
            
        return stream

    # INavigationContributor methods
    def get_active_navigation_item(self, req):
        return 'mailarchive'

    def get_navigation_items(self, req):
        if has_mailarchive_view_permission(self.env, req):
            yield ('mainnav', 'mailarchive',
                   tag.a(_('MailArchive'), href=req.href.mailarchive()))

    # IResourceManager methods
    def get_resource_realms(self):
        yield 'mailarchive'

    def get_resource_url(self, resource, href, **kwargs):
        return href.mailarchive(resource.id)
        
    def get_resource_description(self, resource, format=None, context=None,
                                 **kwargs):
        if context:
            return tag.a('mail:'+resource.id, href=context.href.mailarchive(resource.id))
        else:
            return 'mail:'+resource.id

    # IPermissionRequestor method
    def get_permission_actions(self):
        return self.permission_actions

    # IRequestHandler methods
    def match_request(self, req):
        match = re.match(r'^/mailarchive(?:/(.*))?', req.path_info)
        if match:
            if match.group(1):
                req.args['messageid'] = match.group(1)
            return 1

    def process_request(self, req):
        db = self.env.get_db_cnx()

        messageid = req.args.get('messageid', '')
        action = req.args.get('action', 'list')
        flatmode = (req.args.get('flatmode', 'off') == 'on')
        
        if messageid != '':
            return self._render_view(req, db, messageid)
        else:
            return self._render_list(req, db, flatmode, False)

    def _render_view(self, req, db, id):
        data = {}
        data['action'] = 'view'

        mail = Mail(self.env, id, db=db)
            
        mail.assert_permission(req)

        target_threadroot = mail.thread_root

        if 'mailarc_mails' in req.session:
            self.log.debug(req.session['mailarc_mails'])
            mails = req.session['mailarc_mails'].split()
            if str(id) in mails:
                idx = mails.index(str(id))
                if idx > 0:
                    #add_ctxtnav(req, _('first'), req.href.mailarchive(mails[0]))
                    add_link(req, _('prev'), req.href.mailarchive(mails[idx - 1]))
                if idx < len(mails) - 1:
                    add_link(req, _('next'), req.href.mailarchive(mails[idx + 1]))
                    #add_ctxtnav(req, _('last'), req.href.mailarchive(mails[-1]))
                add_link(req, _('up'), req.session['mailarc_category_href'])
                prevnext_nav(req, u'メール', u'リストに戻る')

        #if target_threadroot == '':
        #    target_threadroot = messageid
        
        data['mail'] = mail
        data['related_tickets'] = get_related_tickets(self.env, req, db, mail.id)
        data['attachment'] = MailArchiveAttachment(self.env, mail.id) 
        
        return 'maildetail.html', data, None

   # Internal methods
    def _render_list(self, req, db, flatmode, month):
        target_category = req.args.get('category', '')

        data = {}

        # ログインユーザがアクセス可能なMLのカテゴリのみを取得する
        data['mls'], data['name'], data['year'], data['month'], target_category \
             = MailFinder.get_categories(self.env, req, db, target_category,
                                         ml_permission_map=ml_permission_map)
        
        results = MailFinder.find_by_category(self.env, target_category)
        
        #pagelize
        pagelized = self._pagelize_list(req, results, data)

        #Thanks http://d.hatena.ne.jp/ohgui/20090806/1249579406
        data['reversemode'] = reversemode = (req.args.get('reversemode', 'off') == 'on')
        data['flatmode'] = flatmode

        mails_per_page, cached_mails = self._get_mails_per_page(pagelized, reversemode, flatmode)

        idstext = self._collect_ids(pagelized.items, flatmode)
        
        self.log.debug("Idtext: %s" % idstext)
        
        req.session['mailarc_mails'] = idstext
        req.session['mailarc_category_href'] = get_category_href(req, target_category)

        data['mails'] = mails_per_page
        data['cached_mails'] = cached_mails

        return 'mailarchive.html', data, None
    
    def _collect_ids(self, mails, flatmode):
        if True:
            return ''.join(['%s ' % mail.id for mail in mails])
        #TODO: ツリー表示の場合に別処理を行うか？
    
    def _get_mails_per_page(self, pagelized, reversemode, flatmode):
        mails_per_page = []
        
        if flatmode:
            mails_per_page = [mail for mail in pagelized.items]
            if reversemode:
                mails_per_page.reverse()
            return mails_per_page, []
            
        else:
            from time import time
            start = time()
            
            #ツリー表示のため、rootだけを集める
            root_mails = []
            for mail in pagelized.items:
                root_mail = mail.get_thread_root(cached_mails=root_mails)
                
                if root_mail in mails_per_page:
                    #ルートの場合最後に登録し直すため一度削除する
                    mails_per_page.remove(root_mail)
                mails_per_page.append(root_mail)
                
                if root_mail not in root_mails:
                    root_mails.append(root_mail)
                    
            #ルートから関連メールを全てキャッシュする
            root_messageids = [x.messageid for x in root_mails]
            cached_mails = MailFinder.find_thread_mails(self.env, root_messageids)
    
            if reversemode:
                mails_per_page.reverse()
                
            end = time()
            self.env.log.debug('Making Threads time: %f' % (end - start))
                
            return mails_per_page, cached_mails
            
    def _pagelize_list(self, req, results, data):
        # get page from req(default page = max_page)
        page = int(req.args.get('page', '-1'))
        num_item_per_page = int(self.env.config.get('mailarchive', 'items_page','50'))
        num_shown_pages = int(self.env.config.get('mailarchive', 'shown_pages','30'))
        if page == -1:
            results_temp = Paginator(results, 0, num_item_per_page)
            page = results_temp.num_pages 

        results = Paginator(results, page - 1, num_item_per_page)
        
        pagedata = []    
        data['page_results'] = results
        shown_pages = results.get_shown_pages(num_shown_pages)
        for shown_page in shown_pages:
            page_href = req.href.mailarchive(category=req.args.get('category',None),
                                        page=shown_page, noquickjump=1)
            pagedata.append([page_href, None, str(shown_page),
                             'page ' + str(shown_page)])

        fields = ['href', 'class', 'string', 'title']
        results.shown_pages = [dict(zip(fields, p)) for p in pagedata]
        
        results.current_page = {'href': None, 'class': 'current',
                                'string': str(results.page + 1),
                                'title':None}

        if results.has_next_page:
            next_href = req.href.mailarchive(category=req.args.get('category',None),
                                        page=page + 1)
            add_link(req, 'next', next_href, _('Next Page'))

        if results.has_previous_page:
            prev_href = req.href.mailarchive(category=req.args.get('category',None),
                                        page=page - 1)
            add_link(req, 'prev', prev_href, _('Previous Page'))

        data['page_href'] = req.href.mailarchive(category=req.args.get('category',None))
        return results 
    
class MailArchiveAjaxModule(Component):
    
    implements(IRequestHandler)
    
    # IRequestHandler methods
    def match_request(self, req):
        match = re.match(r'^/ajaxmailarchive(?:/(.*))?', req.path_info)
        if match:
            if match.group(1):
                req.args['mail_id'] = match.group(1)
            return 1

    def process_request(self, req):
        db = self.env.get_db_cnx()

        mail_id = req.args.get('mail_id')
        mail = Mail(self.env, id=mail_id, db=db)
        
        mail.assert_permission(req)
        
        data = {'mail': mail}
        return 'mail_response.html', data, None
        
class Timeline(Component):
    
    implements(ITimelineEventProvider)

    # ITimelineEventProvider methods
    def get_timeline_filters(self, req):
        if has_mailarchive_view_permission(self.env, req):
            yield ('mailarchive', _(self.env.config.get('mailarchive', 'title', 'MailArchive')))

    def get_timeline_events(self, req, start, stop, filters):
        if 'mailarchive' in filters:
            add_stylesheet(req, 'mailarchive/css/mailarchive.css')

            mails = MailFinder.find_by_date(self.env, start, stop)

            mailarchive_realm = Resource('mailarchive')
                
            for mail in mails:
                resource = mailarchive_realm(id=mail.id, version=None)
                if not mail.has_permission(req):
                    continue
                yield ('mailarchive',
                       datetime.fromtimestamp(mail.utcdate, utc),
                       mail.get_fromtext(),
                       (resource, (mail.category, mail.get_fromtext(), mail.subject)))

    def render_timeline_event(self, context, field, event):
        mailarchive_page, (category,author,subject) = event[3]
        if field == 'url':
            return context.href.mailarchive(mailarchive_page.id, version=mailarchive_page.version)
        elif field == 'title':
            markup = tag(u'メールが ', category, u'に送信されました')
            return markup
        elif field == 'description':
            markup = tag(subject)
            return markup
    
class SearchProvider(Component):
    
    implements(ISearchSource)

    # ISearchProvider methods
    def get_search_filters(self, req):
        if has_mailarchive_view_permission(self.env, req):
            yield ('mailarchive', self.env.config.get('mailarchive', 'title', 'MailArchive'))

    def get_search_results(self, req, terms, filters):
        if 'mailarchive' in filters:
            if not has_mailarchive_view_permission(self.env, req):
                return
            
            db = self.env.get_db_cnx()
            sql_query, args = search_to_sql(db,
                                             ['m1.messageid', 'm1.subject', 'm1.fromname', 'm1.fromaddr', 'm1.text'],
                                             terms)
            sql = "SELECT m1.id, m1.category, m1.subject, m1.fromname, m1.fromaddr, m1.text, m1.utcdate as localdate " \
                  "FROM mailarc m1 " \
                  "WHERE "
            cursor = db.cursor()
            cursor.execute(sql + sql_query, args)
                
            mailarchive_realm = Resource('mailarchive')

            for id, category, subject, fromname, fromaddr, text, localdate in cursor:
                resource = mailarchive_realm(id=id, version=None)
                mlid = category[:-6]
                action = get_mlperm_action(mlid)
                if not action in req.perm(mailarchive_realm):
                    continue

                yield (req.href.mailarchive(id),
                       subject,
                       datetime.fromtimestamp(localdate, utc),
                       get_author(fromname, fromaddr),
                       shorten_result(text, terms))