﻿using NUnit.Framework;
using Meyn.TestLink;
using CookComputing.XmlRpc;

namespace BTS.Trac.Test
{
    using Resolution = TracNameAndValueField<IResolution>;

    [TestFixture]
    [TestLinkFixture(
        Url = TestLinkConnect.Url,
        ProjectName = TestLinkConnect.ProjectName,
        UserId = TestLinkConnect.UserId,
        TestPlan = TestLinkConnect.TestPlan,
        TestSuite = TestLinkConnect.TestSuite,
        DevKey = TestLinkConnect.DevKey )]
    public class ResolutionTest
    {
        [SetUp]
        public void SetUp()
        {
            Trac.Connect( TracConnect.Url, TracConnect.UserName, TracConnect.Password );

            // 最初にすべて削除しておく
            string[] resolutionNames = Resolution.GetAll();
            foreach ( string resolutionName in resolutionNames ) {
                Resolution.Delete( resolutionName );
            }
        }

        [TearDown]
        public void TearDown()
        {
            Trac.Close();
        }

        [Test]
        public void 解決方法の項目を取得する()
        {
            int count = 10;

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string resolutionName = "Test" + i.ToString();
                Resolution resolution = new Resolution();
                resolution.Value = i.ToString();
                resolution.Create( resolutionName );
            }

            string[] resolutionNames = Resolution.GetAll();
            Assert.AreEqual( resolutionNames.Length, count );
        }

        [Test]
        public void 解決方法の各項目を取得する()
        {
            int count = 10;
            string resolutionNameBase = "Test";

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string resolutionName = resolutionNameBase + i.ToString();
                Resolution resolution = new Resolution();
                resolution.Value = i.ToString();
                resolution.Create( resolutionName );
            }

            // 作成数を確認
            string[] resolutionNames = Resolution.GetAll();
            Assert.AreEqual( resolutionNames.Length, count );

            for ( int i = 0; i < count; ++i ) {
                string resolutionName = resolutionNameBase + i.ToString();
                Resolution resolution = new Resolution( resolutionName );

                // 作成したものと取得したものが一致する
                Assert.AreEqual( resolution.Name, resolutionName );
                Assert.AreEqual( resolution.Value, i.ToString() );
            }
        }

        [Test]
        public void 解決方法を作成する()
        {
            string[] before = Resolution.GetAll();

            // 作成
            string resolutionName = "Test";
            Resolution resolution = new Resolution();
            resolution.Value = "10";
            resolution.Create( resolutionName );

            // 解決方法の数が増えていること
            Assert.AreEqual( before.Length + 1, Resolution.GetAll().Length );

            // 取得した解決方法と一致すること
            Resolution getSeverity = new Resolution( resolutionName );
            Assert.AreEqual( resolution.Name, getSeverity.Name );
            Assert.AreEqual( resolution.Value, getSeverity.Value );
        }

        [Test]
        public void 解決方法を更新する()
        {
            string[] before = Resolution.GetAll();

            // 作成
            string resolutionName = "Test";
            Resolution resolution = new Resolution();
            resolution.Value = "10";
            resolution.Create( resolutionName );

            // 解決方法の数が増えていること
            Assert.AreEqual( before.Length + 1, Resolution.GetAll().Length );

            // 更新
            Resolution getSeverity = new Resolution( resolutionName );
            getSeverity.Value = "11";
            getSeverity.Update();

            // 更新した解決方法と一致すること
            Resolution updateSeverity = new Resolution( resolutionName );
            Assert.AreEqual( getSeverity.Name, updateSeverity.Name );
            Assert.AreEqual( getSeverity.Value, updateSeverity.Value );
        }

        [Test]
        public void 解決方法を削除する()
        {
            string[] before = Resolution.GetAll();

            // 作成
            string resolutionName = "Test";
            Resolution resolution = new Resolution();
            resolution.Value = "10";
            resolution.Create( resolutionName );

            // 解決方法の数が増えていること
            Assert.AreEqual( before.Length + 1, Resolution.GetAll().Length );

            resolution.Delete();

            // 解決方法の数が戻っていること
            Assert.AreEqual( before.Length, Resolution.GetAll().Length );
        }

        [Test]
        public void 解決方法を削除するstatic版()
        {
            string[] before = Resolution.GetAll();

            // 作成
            string resolutionName = "Test";
            Resolution resolution = new Resolution();
            resolution.Value = "10";
            resolution.Create( resolutionName );

            // 解決方法の数が増えていること
            Assert.AreEqual( before.Length + 1, Resolution.GetAll().Length );

            Resolution.Delete( resolutionName );

            // 解決方法の数が戻っていること
            Assert.AreEqual( before.Length, Resolution.GetAll().Length );
        }

        [Test]
        [Ignore]
        public void 同じ順序の解決方法を作成すると例外が発生する()
        {
            ///
            /// 今のところ例外が発生せず正常に登録されてしまう
            ///

            // 作成
            string resolutionName = "Test";
            Resolution resolution = new Resolution();
            resolution.Value = "10";
            resolution.Create( resolutionName );

            resolution.Create( resolutionName + "2" );

            // 同じ順序の解決方法を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 同じ名称の解決方法を作成すると例外が発生する()
        {
            // 作成
            string resolutionName = "Test";
            Resolution resolution = new Resolution();
            resolution.Value = "10";
            resolution.Create( resolutionName );

            resolution.Value = "11";
            resolution.Create( resolutionName );

            // 同じ名称の解決方法を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の解決方法を取得すると例外が発生する()
        {
            Resolution resolution = new Resolution( "存在しない解決方法" );

            // 登録されていない名称の解決方法を取得すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の解決方法を更新すると例外が発生する()
        {
            Resolution resolution = new Resolution();
            resolution.Update();

            // 登録されていない名称の解決方法を更新すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の解決方法を削除すると例外が発生する()
        {
            Resolution.Delete( "存在しない解決方法" );

            // 登録されていない名称の解決方法を削除すると例外が発生する
            Assert.Fail();
        }
    }
}
