﻿using System;
using CookComputing.XmlRpc;

namespace BTS.Trac
{
    /// <summary>
    /// コンポーネント
    /// </summary>
    public class Component
    {
        /// <summary>
        /// コンポーネント属性
        /// </summary>
        class ComponentAttributes
        {
            public const string NAME = "name";                  // コンポーネント名
            public const string OWNER = "owner";                // 担当者
            public const string DESCRIPTION = "description";    // 説明
        }

        /// <summary>
        /// 属性
        /// </summary>
        XmlRpcAttributes Attributes = new XmlRpcAttributes();

        /// <summary>
        /// コンポーネント名
        /// </summary>
        public string Name
        {
            get
            {
                return Attributes.Get<string>( ComponentAttributes.NAME );
            }

            private set
            {
                Attributes.Set( ComponentAttributes.NAME, value );
            }
        }

        /// <summary>
        /// 担当者
        /// </summary>
        public string Owner
        {
            get
            {
                return Attributes.Get<string>( ComponentAttributes.OWNER );
            }

            set
            {
                Attributes.Set( ComponentAttributes.OWNER, value );
            }
        }

        /// <summary>
        /// 詳細
        /// </summary>
        public string Description
        {
            get
            {
                return Attributes.Get<string>( ComponentAttributes.DESCRIPTION );
            }

            set
            {
                Attributes.Set( ComponentAttributes.DESCRIPTION, value );
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Component()
        {
            Name = "";
            Owner = "";
            Description = "";
        }

        /// <summary>
        /// コンポーネント
        /// </summary>
        /// <param name="name">詳細を取得するコンポーネント名</param>
        public Component( string name )
        {
            Get( name );
        }

        /// <summary>
        /// コンポーネントの詳細を取得する
        /// </summary>
        /// <param name="name">詳細を取得するコンポーネント名</param>
        /// <returns></returns>
        public void Get( string name )
        {
            Attributes.value = Trac.Rpc.ticketComponentGet( name );
        }

        /// <summary>
        /// コンポーネントを削除する
        /// </summary>
        public void Delete()
        {
            Delete( Name );
        }

        /// <summary>
        /// 指定されたコンポーネントを削除する
        /// </summary>
        /// <param name="name"></param>
        public static void Delete( string name )
        {
            int ret = Trac.Rpc.ticketComponentDelete( name );
            if ( ret != 0 ) {
                throw new Exception( "ticketComponentDelete" );
            }
        }

        /// <summary>
        /// 新しいコンポーネントを作成する
        /// </summary>
        /// <param name="name">作成するコンポーネントの名称</param>
        public void Create( string name )
        {
            Name = name;
            int ret = Trac.Rpc.ticketComponentCreate( name, Attributes.value );
            if ( ret != 0 ) {
                throw new Exception( "ticketComponentUpdate" );
            }
        }

        /// <summary>
        /// 属性の更新
        /// </summary>
        public void Update()
        {
            Update( Name );
        }

        /// <summary>
        /// 名称と属性の更新
        /// </summary>
        /// <param name="name">変更する名称</param>
        public void Update( string name )
        {
            string oldName = Name;
            Name = name;
            int ret = Trac.Rpc.ticketComponentUpdate( oldName, Attributes.value );
            if ( ret != 0 ) {
                throw new Exception( "ticketComponentUpdate" );
            }
        }

        /// <summary>
        /// コンポーネント一覧を取得
        /// </summary>
        /// <returns>コンポーネント一覧</returns>
        public static string[] GetAll()
        {
            return Trac.Rpc.ticketComponentGetAll();
        }
    }
}
