﻿using System;
using CookComputing.XmlRpc;

namespace BTS.Trac
{
    /// <summary>
    /// バージョン
    /// </summary>
    public class Version
    {
        /// <summary>
        /// バージョン属性
        /// </summary>
        class VersionAttributes
        {
            public const string NAME = "name";                  // バージョン名
            public const string DESCRIPTION = "description";    // 説明
            public const string TIME = "time";                  // リリース年月日
        }

        /// <summary>
        /// 属性
        /// </summary>
        XmlRpcAttributes Attributes = new XmlRpcAttributes();

        /// <summary>
        /// バージョン名
        /// </summary>
        public string Name
        {
            get
            {
                return Attributes.Get<string>( VersionAttributes.NAME );
            }

            private set
            {
                Attributes.Set( VersionAttributes.NAME, value );
            }
        }

        /// <summary>
        /// 担当者
        /// </summary>
        public DateTime ReleaseDate
        {
            get
            {
                return Attributes.Get<DateTime>( VersionAttributes.TIME );
            }

            set
            {
                Attributes.Set( VersionAttributes.TIME, value );
            }
        }

        /// <summary>
        /// 詳細
        /// </summary>
        public string Description
        {
            get
            {
                return Attributes.Get<string>( VersionAttributes.DESCRIPTION );
            }

            set
            {
                Attributes.Set( VersionAttributes.DESCRIPTION, value );
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Version()
        {
            Name = "";
            ReleaseDate = DateTime.Now;
            Description = "";
        }

        /// <summary>
        /// バージョン
        /// </summary>
        /// <param name="name">詳細を取得するバージョン名</param>
        public Version( string name )
        {
            Get( name );
        }

        /// <summary>
        /// バージョンの詳細を取得する
        /// </summary>
        /// <param name="name">詳細を取得するバージョン名</param>
        /// <returns></returns>
        public void Get( string name )
        {
            Attributes.value = Trac.Rpc.ticketVersionGet( name );
        }

        /// <summary>
        /// バージョンを削除する
        /// </summary>
        public void Delete()
        {
            Delete( Name );
        }

        /// <summary>
        /// 指定されたバージョンを削除する
        /// </summary>
        /// <param name="name"></param>
        public static void Delete( string name )
        {
            int ret = Trac.Rpc.ticketVersionDelete( name );
            if ( ret != 0 ) {
                throw new Exception( "ticketVersionDelete" );
            }
        }

        /// <summary>
        /// 新しいバージョンを作成する
        /// </summary>
        /// <param name="name">作成するバージョンの名称</param>
        /// <note>DateTime がうまくないらしくエラーになる</note>
        public void Create( string name )
        {
            Name = name;
            int ret = Trac.Rpc.ticketVersionCreate( name, Attributes.value );
            if ( ret != 0 ) {
                throw new Exception( "ticketVersionUpdate" );
            }
        }

        /// <summary>
        /// 属性の更新
        /// </summary>
        public void Update()
        {
            Update( Name );
        }

        /// <summary>
        /// 名称と属性の更新
        /// </summary>
        /// <param name="name">変更する名称</param>
        /// <note>DateTime がうまくないらしくエラーになる</note>
        public void Update( string name )
        {
            string oldName = Name;
            Name = name;
            int ret = Trac.Rpc.ticketVersionUpdate( oldName, Attributes.value );
            if ( ret != 0 ) {
                throw new Exception( "ticketVersionUpdate" );
            }
        }

        /// <summary>
        /// バージョン一覧を取得
        /// </summary>
        /// <returns>バージョン一覧</returns>
        public static string[] GetAll()
        {
            return Trac.Rpc.ticketVersionGetAll();
        }
    }
}
