# -*- encoding: utf-8 -*-
#   Copyright 2009 Agile42 GmbH, Berlin (Germany)
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#   
#   Author: 
#       - Felix Schwarz <felix.schwarz__at__agile42.com>

import unittest

from agilo.utils import Key
from agilo.utils.config import AgiloConfig
from agilo.test import TestEnvHelper


class AgiloConfigTest(unittest.TestCase):
    
    def setUp(self):
        super(AgiloConfigTest, self).setUp()
        self.teh = TestEnvHelper()
        self.env = self.teh.get_env()
        self.trac_config = self.env.config
        # Agilo looks directly in the config so we have to put it there, too
        self.trac_config.set('components', 'agilo.*', 'enabled')
        
        self.config = AgiloConfig(self.env)
    
    def test_days_are_fetched_correctly_from_config(self):
        """Regression test: Check that AgiloConfig uses the right configuration
        section and that use_days is really a bool, not a string."""
        self.trac_config.set('agilo-general', Key.USE_DAYS, False)
        self.assertEqual(False, self.config.use_days)
        
        self.trac_config.set('agilo-general', Key.USE_DAYS, True)
        self.config.reload()
        self.assertEqual(True, self.config.use_days)
    
    def test_can_enable_agilo_ui(self):
        self.config.enable_agilo_ui(save=True)
        self.assertTrue(self.config.is_agilo_ui_enabled)
        self.config.disable_agilo_ui(save=True)
        self.assertEqual(None, self.config.get('templates_dir', 'inherit'))
        self.assertFalse(self.config.is_agilo_ui_enabled)
    
    def test_can_enable_agilo(self):
        self.config.enable_agilo()
        self.assertTrue(self.config.is_agilo_enabled)
        self.config.disable_agilo()
        self.assertFalse(self.config.is_agilo_enabled)
    
    def test_can_disable_agilo_ui(self):
        self.assertEqual(True, self.config.is_agilo_ui_enabled)
        self.config.disable_agilo_ui(save=True)
        self.assertEqual(False, self.config.is_agilo_ui_enabled)
    
    def _set_template_dir(self, config, dirname):
        config.change_option('templates_dir', dirname, 'inherit', save=True)
    
    def test_configuration_detects_outdated_template_path(self):
        self.assertEqual(True, self.config.is_agilo_enabled)
        self.assertEqual(True, self.config.is_agilo_ui_enabled)
        
        current_dir = '/usr/share/agilo-0.42-r12345.egg/templates'
        self._set_template_dir(self.config, '')
        self.assertEqual(False, self.config.is_agilo_ui_enabled)
        self.assertEqual(False, self.config._is_template_dir_outdated(current_dir))
        
        self._set_template_dir(self.config, '/usr/share/agilo/templates')
        self.assertEqual(False, self.config._is_template_dir_outdated(current_dir))
        
        self._set_template_dir(self.config, current_dir.replace('12345', '54321'))
        self.assertEqual(True, self.config._is_template_dir_outdated(current_dir))
    
    def test_can_remove_whole_sections(self):
        section = self.config.get_section('fnord')
        section.change_option('foo', 'bar')
        self.assertTrue('fnord' in self.trac_config.sections())
        self.assertEquals('bar', section.get('foo'))
        section.remove()
        self.assertNotEquals('bar', section.get('foo'))
    
    def test_can_remove_sections_without_getting_it_first(self):
        section = self.config.get_section('fnord')
        section.change_option('foo', 'bar')
        self.assertTrue('fnord' in self.trac_config.sections())
        self.assertEquals('bar', section.get('foo'))
        self.config.remove(section='fnord')
        self.assertNotEquals('bar', section.get('foo'))
    
    def test_config_knows_if_an_option_is_set(self):
        self.assertFalse(self.config.has_option('foo', section='fnord'))
        self.config.change_option('foo', 'bar', section='fnord')
        self.assertTrue(self.config.has_option('foo', section='fnord'))


if __name__ == '__main__':
    suite = unittest.TestLoader().loadTestsFromTestCase(AgiloConfigTest)
    unittest.TextTestRunner(verbosity=0).run(suite)
