/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.connect;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import jdbcacsess2.main.Config;

final public class ConnectTableModel extends AbstractTableModel  {

	private static final long serialVersionUID = 6824005336951869877L;

	private static final String header[] = { "name", "user", "server", "DBname", "last connect" };
	private final List<ConfigConnect> connectList;

	private final Config config;

	public ConnectTableModel(Config config, ConfigConnect[] connectListArray) throws SQLException {
		this.config = config;
		this.connectList = new ArrayList<ConfigConnect>(Arrays.asList(connectListArray));
	}

	@Override
	public String getColumnName(int column) {
		return header[column];
	}

	@Override
	public int getColumnCount() {
		return header.length;
	}

	@Override
	public int getRowCount() {
		return connectList.size();
	}

	public ConfigConnect getRow(int rowIndex) {
		if (rowIndex < 0) {
			return null;
		}
		return connectList.get(rowIndex);
	}

	/**
	 * 接続先名称に一致する値を検索し、行インデックスを取得する。
	 * 
	 * @param connectName
	 *            検索したい接続先名称
	 * @return 検索結果の行インデックス。検索値以上の値が存在しない時は-1。
	 * @throws SQLException
	 */
	public int getRowIndex(ConfigConnect configConnect) throws SQLException {
		String connectName = configConnect.getConnectName();
		if (connectName == null) {
			connectName = "";
		}

		for (int i = 0; i < connectList.size(); i++) {
			int val = connectList.get(i).getConnectName().compareTo(connectName);
			if (val == 0) {
				return i;
			}
		}
		return -1;
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		Object rtn = null;

		switch (columnIndex) {
		case 0:
			rtn = connectList.get(rowIndex).getConnectName();
			break;
		case 1:
			rtn = connectList.get(rowIndex).getConnectUser();
			break;
		case 2:
			rtn = connectList.get(rowIndex).getHostName();
			break;
		case 3:
			rtn = connectList.get(rowIndex).getDatabaseName();
			break;
		case 4:
			rtn = connectList.get(rowIndex).getLastConnectdate();
			break;
		default:
			break;
		}
		return rtn;
	}

	public void update(ConfigConnect newConfigConnect) {
		int i = 0;
		while (i < connectList.size()) {
			if (connectList.get(i).getConnectName().equals(newConfigConnect.getConnectName())) {
				fireTableRowsUpdated(i, i);
				newConfigConnect.save();
				return;
			}
			i++;
		}
	}

	public class AlreadyExistsException extends Exception {
		private static final long serialVersionUID = 1411153554815205565L;
		public AlreadyExistsException(String string) {
			super(string);
		}
	}

	public void append(ConfigConnect newConfigConnect) throws AlreadyExistsException {
		int i = 0;
		while (i < connectList.size()) {
			if (connectList.get(i).getConnectName().equals(newConfigConnect.getConnectName())) {
				throw new AlreadyExistsException(newConfigConnect.getConnectName() + " is exists");
			}
			i++;
		}
		connectList.add(newConfigConnect);
		fireTableRowsInserted(i, i);
		newConfigConnect.save();
	}

	/**
	 * 接続名称が、存在しなければ最終データに追加し、存在すれば更新する。
	 * 
	 * @param configConnect
	 */
	public void save(ConfigConnect configConnect) {
		int i = 0;
		while (i < connectList.size()) {
			if (connectList.get(i).getConnectName().equals(configConnect.getConnectName())) {
				connectList.set(i, configConnect);
				fireTableRowsUpdated(i, i);
				configConnect.save();
				return;
			}
			i++;
		}
		connectList.add(configConnect);
		fireTableRowsInserted(i, i);
		configConnect.save();
	}

	public void remove(ConfigConnect oldConfigConnect) throws SQLException {
		for (int i = 0; i < connectList.size(); i++) {
			if (connectList.get(i).getConnectName().equals(oldConfigConnect.getConnectName())) {
				// delete
				connectList.remove(i);
				fireTableRowsDeleted(i, i);
				config.getEntityManager().delete(oldConfigConnect);
				return;
			}
		}
	}
}
