/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.main;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Properties;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.regex.Pattern;

import jdbcacsess2.connect.ConfigConnect;
import jdbcacsess2.connect.ConfigDbms;
import jdbcacsess2.connect.ConnectView;
import jdbcacsess2.sqlService.history.ExecHistory;
import net.java.ao.DBParam;
import net.java.ao.EntityManager;
import net.java.ao.Query;
import net.java.ao.RawEntity;

public class Config {

	private static final String CONFIG_DIR = System.getProperty("user.home") + System.getProperty("file.separator")
			+ ".jdbcacsess2";

	private static final String CONFIG_DB = "config.derby";
	private static final String CONFIG_JDBCURL = "jdbc:derby:" + CONFIG_DIR + System.getProperty("file.separator")
			+ CONFIG_DB + ";create=true";

	private static final String CONFIG_JDBCCLASS = "org.apache.derby.jdbc.EmbeddedDriver";
	private static final String USER = "";
	private static final String PASS = "";

	private final EntityManager entityManager;

	public OptionValues optionValues;

	/**
	 * コンストラクタ
	 * 
	 * @throws IOException
	 * @throws ClassNotFoundException
	 * @throws SQLException
	 */
	public Config() throws IOException, ClassNotFoundException, SQLException {

		// コンソールとファイルのハンドラーで使用するログフォーマッター
		Formatter oneLine = new OneLinenFormatter();

		// コンソールハンドラー
		Handler consoleHandler = new ConsoleHandler();
		consoleHandler.setFormatter(oneLine);
		consoleHandler.setLevel(Level.ALL);

		Jdbcacsess2.logger.addHandler(consoleHandler);
		Jdbcacsess2.logger.setUseParentHandlers(false);
		Jdbcacsess2.loggerActiveObjects.addHandler(consoleHandler);
		Jdbcacsess2.loggerActiveObjects.setUseParentHandlers(false);

		// DB作成
		File configDir = new File(CONFIG_DIR);
		if (!configDir.exists()) {
			if (!configDir.mkdir()) {
				throw new IOException("Couldnt create diretrory:" + configDir.getAbsolutePath());
			}
		}

		// ログファイルハンドラー
		Handler fileHandler = new FileHandler(CONFIG_DIR + File.separator + "message.log", 1024 * 1024, 5);
		fileHandler.setFormatter(oneLine);
		fileHandler.setLevel(Level.ALL);

		Jdbcacsess2.logger.addHandler(fileHandler);
		Jdbcacsess2.loggerActiveObjects.addHandler(fileHandler);

		Jdbcacsess2.logger.info("file logger start");
		Jdbcacsess2.logger.info("configure directory:" + CONFIG_DIR);

		// システムプロパティを表示する
		if (Jdbcacsess2.logger.isLoggable(Level.FINE)) {
			Properties prop = System.getProperties();
			ArrayList<String> aList = Collections.list(extracted(prop));
			Collections.sort(aList);

			for (String string : aList) {
				Jdbcacsess2.logger.fine(string + ":" + System.getProperty(string));
			}
		} else {
			Jdbcacsess2.logger.info(System.getProperty("java.vm.vendor") + " " + System.getProperty("java.vm.name")
			                        + " " + System.getProperty("java.vm.version"));
			Jdbcacsess2.logger.info(System.getProperty("java.home"));
			Jdbcacsess2.logger.info(System.getProperty("java.class.path"));
			Jdbcacsess2.logger.info(System.getProperty("java.library.path"));
			Jdbcacsess2.logger.info(System.getProperty("java.ext.dirs"));
		}

		// ActibObjectsの初期設定
		Class.forName(CONFIG_JDBCCLASS);

		this.entityManager = new EntityManager(CONFIG_JDBCURL, USER, PASS);
		Jdbcacsess2.loggerActiveObjects.setLevel(Level.WARNING);
		Jdbcacsess2.loggerActiveObjects.setLevel(Level.FINE);

		migrate();
		Jdbcacsess2.logger.info("Migrate end");
		setUpConfigDbms();

		// 接続情報をファイルにバックアップ
		try {
			File file = getBackupFile(CONFIG_DIR, "BackupConfigConnect", 5);
			Jdbcacsess2.logger.info("BackupConfigConnect:" + file.getAbsolutePath());

			ConfigConnect[] ccs = getEntityManager().find(ConfigConnect.class,
			                                              Query.select());
			if (ccs.length != 0) {
				try (BufferedWriter bw = new BufferedWriter(new FileWriter(file))) {
					bw.write(ccs[0].convertRowheader().toString());
					bw.newLine();
					for (ConfigConnect configConnect : ccs) {
						bw.write(configConnect.convertRow().toString());
						bw.newLine();
					}
				}
			}

		} catch (IOException e) {
			ShowDialog.errorMessage(e);
		}

		// プロパティの作成
		optionValues = new OptionValues(getEntityManager());

		Jdbcacsess2.logger.info("Config end");
	}

	@SuppressWarnings("unchecked")
	private Enumeration<String> extracted(Properties prop) {
		return (Enumeration<String>) prop.propertyNames();
	}

	/**
	 * 世代管理されたファイル名を生成する
	 * 
	 * @param dir
	 *            ディレクトリ名
	 * @param name
	 *            ファイル名
	 * @param cnt
	 *            世代数
	 * @return 次回作成すべきファイル
	 * @throws IOException
	 */
	private File getBackupFile(String dir, String name, int cnt) throws IOException {

		final Pattern backupFilePattern = Pattern.compile(name + "\\.[0-" + (cnt - 1) + "]");
		File[] f = new File(dir).listFiles(new FilenameFilter() {
			@Override
			public boolean accept(File dir, String name) {
				return backupFilePattern.matcher(name).find();
			}
		});

		Arrays.sort(f, new Comparator<File>() {
			@Override
			public int compare(File o1, File o2) {
				if (o1.lastModified() < o2.lastModified()) {
					return -1;
				}
				if (o1.lastModified() > o2.lastModified()) {
					return 1;
				}
				return 0;
			}
		});

		File backup;
		if (f.length == cnt) {
			backup = f[0];
			backup.delete();
		} else {
			backup = new File(dir + System.getProperty("file.separator") + name + "." + f.length);
		}
		backup.createNewFile();

		return backup;
	}

	/**
	 * ActiveObjects の migrate を呼ぶ。SuppressWarningsでワーニングを抑止する為にメソッド抽出。
	 * 
	 * @throws SQLException
	 */
	@SuppressWarnings("unchecked")
	private void migrate() throws SQLException {
		getEntityManager().migrate(ConfigDbms.class,
		                           ConfigConnect.class,
		                           ConnectView.ViewConnect.class,
		                           MainView.ViewMain.class,
		                           GuiFont.class,
		                           GuiWindow.class,
		                           GuiDivider.class,
		                           ExecHistory.class,
		                           ConfigProperty.class
				);
	}

	/**
	 * 接続名称で検索し、存在する・しないに関わらず、保存用の{@link ConfigConnect}を返却します。
	 * 
	 * @param connectName
	 *            接続名称
	 * @return データがあった場合は検索結果の{@link ConfigConnect}、なかった場合は挿入用の {@link ConfigConnect}
	 * @throws SQLException
	 */
	public ConfigConnect getConfigConnectRow(String connectName) throws SQLException {
		ConfigConnect[] configConnects = getEntityManager().find(ConfigConnect.class,
		                                                         Query.select().where(ConfigConnect.CONNECTNAME + "=?",
		                                                                              connectName));
		ConfigConnect rtn;
		if (configConnects.length == 0l) {
			rtn = getEntityManager().create(ConfigConnect.class, new DBParam(ConfigConnect.CONNECTNAME, connectName));
		} else {
			rtn = configConnects[0];
		}
		return rtn;
	}

	/**
	 * テーブル検索し、先頭１件だけに絞る。存在しない場合は初期値行を生成し返却します。
	 * 
	 * @param <T>
	 * @param <K>
	 * @param type
	 *            検索対象のテーブルクラス
	 * @return データが存在しない時は、 {@link EntityManager#create(Class, net.java.ao.DBParam...)} を実行し空行を取得します。
	 * @throws SQLException
	 */
	public <T extends RawEntity<K>, K> T getEntity(Class<T> type) throws SQLException {
		T[] e = getEntityManager().find(type);
		if (e.length != 0) {
			return e[0];
		}
		return getEntityManager().create(type);
	}

	/**
	 * @throws SQLException
	 */
	private void setUpConfigDbms() throws SQLException {
		String[][] rows =
			{
			 {"PostgreSQL","postgresql","//%HostName%%PortNumber%/%DataBaseName%","org.postgresql.Driver","","00.1"},
			 {"MySQL","mysql","//%HostName%%PortNumber%/%DataBaseName%","org.gjt.mm.mysql.Driver","","00.0"},
			 {"MySQL(com)","mysql","//%HostName%%PortNumber%/%DataBaseName%","com.mysql.jdbc.Driver","","00.0"},
			 {"HSQL","hsqldb:hsql","//%HostName%%PortNumber%/%DataBaseName%","org.hsqldb.jdbcDriver","","00.0"},
			 {"Oracle(TYPE4:Pure Java)","oracle:thin","@%HostName%%PortNumber%:%DataBaseName%","oracle.jdbc.driver.OracleDriver","","00.0"},
			 {"Oracle(TYPE2:NET)","oracle:oci","@%DataBaseName%","oracle.jdbc.driver.OracleDriver","","00.0"},
			 {"Oracle(TYPE2:NET8)","oracle:oci8","@%DataBaseName%","oracle.jdbc.driver.OracleDriver","","00.0"},
			 {"DB2(TYPE4:Universal)","db2","//%HostName%%PortNumber%/%DataBaseName%","com.ibm.db2.jcc.DB2Driver","","00.1"},
			 {"DB2(TYPE2:Universal)","db2","%DataBaseName%","com.ibm.db2.jcc.DB2Driver","","00.0"},
			 {"DB2(TYPE2:App)","db2","%DataBaseName%","COM.ibm.db2.jdbc.app.DB2Driver","","00.0"},
			 {"DB2(TYPE3:Net)","db2","//%HostName%%PortNumber%/%DataBaseName%","COM.ibm.db2.jdbc.net.DB2Driver","","00.0"},
			 {"SQL Server","sqlserver","//%HostName%%PortNumber%;databaseName=%DataBaseName%","com.microsoft.sqlserver.jdbc.SQLServerDriver","","00.0"},
			 {"SQL Server(2000)","microsoft:sqlserver","//%HostName%%PortNumber%;databaseName=%DataBaseName%","com.microsoft.jdbc.sqlserver.SQLServerDriver","","00.0"},
			 {"Derby(Embedded)","derby","%DataBaseName%","org.apache.derby.jdbc.EmbeddedDriver","","00.0"},
			 {"Derby(Client)","derby","%DataBaseName%","org.apache.derby.jdbc.ClientDriver","","00.0"},
			 {"SQLite","sqlite","/%DataBaseName%","org.sqlite.JDBC","","00.0"},
			 {"Firebird","firebirdsql","//%HostName%%PortNumber%/%DataBaseName%","org.firebirdsql.jdbc.FBDriver","","00.0"},
			 {"Symfoware","symford","//%HostName%%PortNumber%/%DataBaseName%","com.fujitsu.symfoware.jdbc.SYMDriver","","00.0"},
			 {"HiRDB","hitachi:hirdb","//DBID=%PortNumberRAW%,DBHOST=%HostName%","JP.co.Hitachi.soft.HiRDB.JDBC.HiRDBDriver","","00.0"},
			 {"ODBC(TYPE1)","odbc","%DataBaseName%","sun.jdbc.odbc.JdbcOdbcDriver","","00.0"},
			 {"free format","","","","","00.1"}
			};

		for (String[] row : rows) {
			ConfigDbms[] dbms = getEntityManager().find(ConfigDbms.class,
			                                            Query.select().where(ConfigDbms.DBMSNAME + "=?", row[0]));
			if (dbms.length == 0) {
				getEntityManager().create(ConfigDbms.class,
				                          new DBParam(ConfigDbms.DBMSNAME, row[0]),
				                          new DBParam(ConfigDbms.SUBPROTOCOL, row[1]),
				                          new DBParam(ConfigDbms.SUBNAME, row[2]),
				                          new DBParam(ConfigDbms.DRIVERCLASS, row[3]),
				                          new DBParam(ConfigDbms.URLOPTION, row[4]),
				                          new DBParam(ConfigDbms.VERSION, row[5])
						);
				Jdbcacsess2.logger.info("not exists:" + row[0]);
			} else {
				if (dbms[0].getVersion().compareTo(row[5]) < 0) {
					dbms[0].setSubprotocol(row[1]);
					dbms[0].setSubname(row[2]);
					dbms[0].setDriverclass(row[3]);
					dbms[0].setUrlOption(row[4]);
					dbms[0].setVersion(row[5]);
					dbms[0].save();
					Jdbcacsess2.logger.info("update:" + row[0]);
				}
			}
		}
	}

	/**
	 * 設定情報保管用のDBをクローズ
	 */
	public void close() {
		getEntityManager().flushAll();
		getEntityManager().getProvider().dispose();
	}

	/**
	 * @return entityManager
	 */
	public EntityManager getEntityManager() {
		return entityManager;
	}

}
