/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.main;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.lang.reflect.InvocationTargetException;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.DefaultTableCellRenderer;

import jdbcacsess2.main.icon.IconImage;
import jdbcacsess2.sqlService.ColumnAttributeResult;
import jdbcacsess2.sqlService.DataBaseConnection;
import jdbcacsess2.sqlService.DataBaseConnectionListener;
import jdbcacsess2.sqlService.DataBaseTransactionListener;
import jdbcacsess2.sqlService.PropertyResultSetTableModel;
import jdbcacsess2.sqlService.Result;
import jdbcacsess2.sqlService.ResultsTableModel;
import jdbcacsess2.sqlService.SqlExecuteTask;
import jdbcacsess2.sqlService.SqlExecutedListener;
import jdbcacsess2.sqlService.parse.SqlExecuteSentencies.SqlExecuteSentence;

import org.fife.ui.rsyntaxtextarea.RSyntaxTextArea;
import org.fife.ui.rsyntaxtextarea.SyntaxConstants;
import org.fife.ui.rtextarea.RTextScrollPane;

public class JPanelSql extends JPanel implements SqlExecutedListener, DataBaseConnectionListener,
DataBaseTransactionListener {

	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------

	final static private int TABBEDPANEL_INDEX_DATA = 0;
	final static private int TABBEDPANEL_INDEX_LOG = 1;
	final static private int TABBEDPANEL_INDEX_RS = 2;

	private DataBaseConnection dataBaseConnection; // @jve:decl-index=0:

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.DataBaseConnectionListener#dataBaseConnectionOpened
	 * (jdbcacsess2.sqlService.DataBaseConnection)
	 */
	@Override
	public void dataBaseConnectionOpened(DataBaseConnection dataBaseConnection) {
		this.dataBaseConnection = dataBaseConnection;

		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				writeGuiLog("connection opened");

				// dbコネクションの autocommit を、画面設定から取得して設定する。
				boolean flg = getJCheckBoxAutoCommit().isSelected();
				try {
					getDataBaseConnection().setAutoCommit(flg);

					getJToggleButtonExecution().setEnabled(true);
					getJButtonContinue().setEnabled(false);
					getJCheckBoxAutoCommit().setEnabled(true);
					getJToggleButtonConnection().setSelected(true);
				} catch (Exception e) {
					ShowDialog.errorMessage(e);
				}
			}
		});

	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.DataBaseConnectionListener#dataBaseConnectionClosing ()
	 */
	@Override
	public void dataBaseConnectionClosing(DataBaseConnection dataBaseConnection) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				getJToggleButtonExecution().setEnabled(false);
				getJButtonContinue().setEnabled(false);
				getJCheckBoxAutoCommit().setEnabled(false);
				getJButtonCommit().setEnabled(false);
				getJButtonRollback().setEnabled(false);
				getJToggleButtonConnection().setSelected(false);
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.DataBaseConnectionListener#dataBaseConnectionClosed ()
	 */
	@Override
	public void dataBaseConnectionClosed(DataBaseConnection dataBaseConnection) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				writeGuiLog("connection closed");
			}
		});
	}

	/**
	 * @return dataBaseConnection
	 */
	DataBaseConnection getDataBaseConnection() {
		return dataBaseConnection;
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.DataBaseTransactionListener#autoCommitChange(boolean )
	 */
	@Override
	public void autoCommitChange(DataBaseConnection dataBaseConnection, final boolean autoCommit) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				writeGuiLog("autocommit[" + autoCommit + "]");
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.DataBaseTransactionListener#commitEnd()
	 */
	@Override
	public void commitEnd(DataBaseConnection dataBaseConnection) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				getJTabbedPaneResult().setSelectedIndex(TABBEDPANEL_INDEX_LOG);
				writeGuiLog("commit end");
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.DataBaseTransactionListener#rollbackEnd()
	 */
	@Override
	public void rollbackEnd(DataBaseConnection dataBaseConnection) {
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				getJTabbedPaneResult().setSelectedIndex(TABBEDPANEL_INDEX_LOG);
				writeGuiLog("rollback end");
			}
		});
	}

	/**
	 * SQLパネルログへ出力
	 * 
	 * @param message
	 *            出力文字列
	 */
	private void writeGuiLog(String message) {
		int i;
		for (i = message.length(); 0 < i; i--) {
			String s = message.substring(i - 1, i);
			if (!s.equals("\n")) {
				break;
			}
		}

		getJTextAreaLog().append(message.substring(0, i));
		getJTextAreaLog().append("\n");
		getJTextAreaLog().setCaretPosition(getJTextAreaLog().getText().length());
	}

	//
	/**
	 * 文字列を挿入します。改行文字が含まれている場合、インデントされます。
	 * 
	 * @param str
	 *            挿入したい文字列
	 */
	public void insertStringPreviousCaret(String str) {
		if (str.equals("")) {
			return;
		}
		rSyntaxTextAreaSql.replaceRange(addIndent(str),
		                                rSyntaxTextAreaSql.getSelectionStart(),
		                                rSyntaxTextAreaSql.getSelectionEnd());

		rSyntaxTextAreaSql.requestFocus();
	}

	/**
	 * 改行文字があったら、現在行の左端からの位置をインデント文字数として、空白インデント処理する
	 * 
	 * @param editot
	 *            挿入文字列
	 * @return インデント挿入後の文字列
	 */
	private String addIndent(String editor) {

		// 先頭からのキャレット位置 - 現在行の先頭キャレット位置 で、現在行内でのキャレットオフセットを取得
		int x = rSyntaxTextAreaSql.getCaretPosition() - rSyntaxTextAreaSql.getLineStartOffsetOfCurrentLine();

		// indent用に左空白文字列を作成する
		StringBuilder indent = new StringBuilder();
		for (int i = 0; i < x; i++) {
			indent.append(' ');
		}

		// indent後の文字列を作成
		StringBuilder buf = new StringBuilder();
		for (int i = 0; i < editor.length(); i++) {
			char c = editor.charAt(i);
			buf.append(c);
			if (c == '\n') {
				// 改行の後にはインデントを挿入する
				buf.append(indent);
			}
		}
		return buf.toString();
	}

	/**
	 * 実行結果を受け取り、ボタンの制御を行う。
	 */
	private SqlExecuteTask sqlExecuteTask; // @jve:decl-index=0:

	public void taskWakuUp() {
		sqlExecuteTask.taskWakeUp();
	}

	public void taskCancel() {
		sqlExecuteTask.taskCancel();
	}

	/**
	 * 
	 */
	private TableColumnFit resultsTableColumnFit;
	/**
	 * 実行を開始したSQL情報
	 */
	private SqlExecuteSentence sqlExecuteSentenceLast; // @jve:decl-index=0:
	/**
	 * 例外発生時に処理を継続するか
	 */
	private boolean exceptionContinue;

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#taskAccept(jdbcacsess2.sqlService .SqlExecuteTask)
	 */
	@Override
	public void taskAccept(SqlExecuteTask sqlExecuteTask) {
		this.sqlExecuteTask = sqlExecuteTask;
		exceptionContinue = false;

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				getJToggleButtonExecution().setEnabled(false);
				getJToggleButtonExecution().setSelected(true);
				getJButtonContinue().setEnabled(false);
				getJToggleButtonConnection().setEnabled(false);
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executBegin(jdbcacsess2.sqlService
	 * .SqlExecuteSentencies.SqlExecuteSentence)
	 */
	@Override
	public void executBegin(final SqlExecuteSentence sqlExecuteSentence) {
		sqlExecuteSentenceLast = sqlExecuteSentence;

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {

				if (sqlExecuteSentence.getSqlCommand().toUpperCase().equals("SELECT")) {
					// カラム自動調整を有効にする
					resultsTableColumnFit.setEnableAutoFit(true);

					getJTabbedPaneResult().setSelectedIndex(TABBEDPANEL_INDEX_DATA);
				} else {
					getJTabbedPaneResult().setSelectedIndex(TABBEDPANEL_INDEX_LOG);
				}

				writeGuiLog("\nstart sql\n"
						+ sqlExecuteSentence.getSqlSentence());
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executNormalFinish(int)
	 */
	@Override
	public void executNormalFinish(final int rowCnt) {
		// sqlExecuteSentenceLast は、SwingUtilities.invokeLaterの外から
		// どんどん更新されるので、画面出力に必要な情報はEDTキューイング前に作成しておく
		final String counrMessage = countmsg();

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				getJToggleButtonExecution().setEnabled(true);
				writeGuiLog("normal end sql" + counrMessage + " "
						+ Integer.toString(rowCnt) + " row " + sqlExecuteTask.getExecutionTime()
						+ "ms");
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executeException(java.lang .Throwable)
	 */
	@Override
	public void executeException(final Throwable t) {
		// sqlExecuteSentenceLast は、SwingUtilities.invokeLaterの外から
		// どんどん更新されるので、画面出力に必要な情報はEDTキューイング前に作成しておく
		final String counrMessage = countmsg();

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				getJToggleButtonExecution().setEnabled(true);
				writeGuiLog("abnormal end " + counrMessage + "\n "
						+ t.getLocalizedMessage());
			}
		});
		ShowDialog.errorMessage(t);

		int zan = sqlExecuteTask.getCnt() - sqlExecuteSentenceLast.getSentenceCount();
		if (!exceptionContinue && zan > 0) {
			try {
				SwingUtilities.invokeAndWait(new Runnable() {
					@Override
					public void run() {
						Object[] options = {
						                    "はい、後続のSQL実行を中断", "いいえ、以降の全エラーを無視"
						};
						int n = JOptionPane.showOptionDialog(null,
						                                     "残りのSQL実行は中断しますか？",
						                                     "まだ実行予定のSQLが残ってます",
						                                     JOptionPane.YES_NO_OPTION,
						                                     JOptionPane.QUESTION_MESSAGE,
						                                     null,
						                                     options,
						                                     options[0]);
						if (n == JOptionPane.NO_OPTION) {
							exceptionContinue = true;
						} else {
							taskCancel();
						}
					}
				});
			} catch (InvocationTargetException e) {
				ShowDialog.errorMessage(e);
				e.printStackTrace();
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
			}
		}
	}

	private String countmsg() {
		return "[" + sqlExecuteSentenceLast.getSentenceCount() + "/" + sqlExecuteTask.getCnt() + "]";
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#resultHeader(java.util.List)
	 */
	@Override
	public void resultHeader(final List<ColumnAttributeResult> header) {
		final PropertyResultSetTableModel dataModel = new PropertyResultSetTableModel(header);
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				getJTableResultSet().setModel(dataModel);
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#resultDetail(int, java.util.List)
	 */
	@Override
	public void resultDetail(int seq, List<Result> detail) {
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#statusContinue(int)
	 */
	@Override
	public void statusContinue(final int seqNo) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				getJToggleButtonExecution().setEnabled(true);
				getJButtonContinue().setEnabled(true);
				getJToggleButtonConnection().setEnabled(false);

				writeGuiLog("data cotinue  now " + Integer.toString(seqNo) + " row");
			}
		});
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executeAllEnd()
	 */
	@Override
	public void executeAllEnd() {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				getJToggleButtonExecution().setEnabled(true);
				getJToggleButtonExecution().setSelected(false);
				getJButtonContinue().setEnabled(false);
				getJToggleButtonConnection().setEnabled(true);
			}
		});
	}

	class TableRenderer extends DefaultTableCellRenderer {

		@Override
		public Component getTableCellRendererComponent(JTable table, Object value,
		                                               boolean isSelected, boolean hasFocus,
		                                               int row, int column) {
			Result r = (Result) value;
			if (r.nulldata && !hasFocus) {
				setBackground(Color.ORANGE);
			} else {
				setBackground(null);
			}
			if (r.error && !hasFocus) {
				setBackground(Color.PINK);
			}

			if (r.num) {
				setHorizontalAlignment(JLabel.RIGHT);
			} else {
				setHorizontalAlignment(JLabel.LEFT);
			}

			super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
			return this;
		}
	}

	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------

	private static final long serialVersionUID = 1L;
	private JSplitPane jSplitPaneTable = null;
	private RTextScrollPane rTextScrollPaneSql = null;
	private RSyntaxTextArea rSyntaxTextAreaSql = null;
	private JTabbedPane jTabbedPaneResult = null;
	private JScrollPane jScrollPane = null;
	private JTable jTable = null;
	private JScrollPane jScrollPaneLog = null;
	private JTextArea jTextAreaLog = null;
	private JToolBar jJToolBarBar = null;
	private JButton jButtonContinue = null;
	private JCheckBox jCheckBoxAutoCommit = null;
	private JButton jButtonCommit = null;
	private JButton jButtonRollback = null;

	private JToggleButton jToggleButtonExecution = null;
	private JToggleButton jToggleButtonConnection = null;
	private JScrollPane jScrollPane1 = null;
	private JTable jTableResultSet = null;
	private JComboBox<SentenceSeparator> jComboBoxSentenceSeparator = null;

	/**
	 * This is the default constructor
	 */
	public JPanelSql() {
		super();
		initialize();
	}

	/**
	 * This method initializes this
	 * 
	 * @return void
	 */
	private void initialize() {
		this.setLayout(new BorderLayout());
		this.add(getJSplitPaneTable(), java.awt.BorderLayout.CENTER);
		this.add(getJJToolBarBar(), java.awt.BorderLayout.NORTH);
	}

	/**
	 * This method initializes jSplitPaneTable
	 * 
	 * @return javax.swing.JSplitPane
	 */
	JSplitPane getJSplitPaneTable() {
		if (jSplitPaneTable == null) {
			jSplitPaneTable = new JSplitPane();
			jSplitPaneTable.setOrientation(JSplitPane.VERTICAL_SPLIT);
			jSplitPaneTable.setDividerLocation(200);
			jSplitPaneTable.setDividerSize(10);
			jSplitPaneTable.setOneTouchExpandable(true);
			jSplitPaneTable.setTopComponent(getRTextScrollPaneSql());
			jSplitPaneTable.setBottomComponent(getJTabbedPaneResult());
		}
		return jSplitPaneTable;
	}

	/**
	 * This method initializes jScrollPane1
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private RTextScrollPane getRTextScrollPaneSql() {
		if (rTextScrollPaneSql == null) {
			rTextScrollPaneSql = new RTextScrollPane();
			rTextScrollPaneSql.setViewportView(getRSyntaxTextAreaSql());
		}
		return rTextScrollPaneSql;
	}

	/**
	 * This method initializes jEditorPaneSql
	 * 
	 * @return javax.swing.JEditorPane
	 */
	RSyntaxTextArea getRSyntaxTextAreaSql() {
		if (rSyntaxTextAreaSql == null) {
			rSyntaxTextAreaSql = new RSyntaxTextArea();
			rSyntaxTextAreaSql.setSyntaxEditingStyle(SyntaxConstants.SYNTAX_STYLE_SQL);
			rSyntaxTextAreaSql.setAntiAliasingEnabled(true);
			FontZoom.setMouseWheel(rSyntaxTextAreaSql);
		}
		return rSyntaxTextAreaSql;
	}

	/**
	 * This method initializes jTabbedPaneResult
	 * 
	 * @return javax.swing.JTabbedPane
	 */
	JTabbedPane getJTabbedPaneResult() {
		if (jTabbedPaneResult == null) {
			jTabbedPaneResult = new JTabbedPane();
			jTabbedPaneResult.setTabPlacement(SwingConstants.LEFT);
			jTabbedPaneResult.add(getJScrollPane(), "data", TABBEDPANEL_INDEX_DATA);
			jTabbedPaneResult.add(getJScrollPaneLog(), "log", TABBEDPANEL_INDEX_LOG);
			jTabbedPaneResult.add(getJScrollPane1(), "RS", TABBEDPANEL_INDEX_RS);
		}
		return jTabbedPaneResult;
	}

	/**
	 * This method initializes jScrollPane
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private JScrollPane getJScrollPane() {
		if (jScrollPane == null) {
			jScrollPane = new JScrollPane();
			jScrollPane.setViewportView(getJTable());
		}
		return jScrollPane;
	}

	/**
	 * This method initializes jTable
	 * 
	 * @return javax.swing.JTable
	 */
	JTable getJTable() {
		if (jTable == null) {
			jTable = new JTable();
			jTable.setIntercellSpacing(new Dimension(3, 2));
			jTable.setAutoCreateRowSorter(true);
			jTable.setCellSelectionEnabled(true);
			jTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
			Font font = jTable.getFont();
			jTable.setFont(new Font(Font.MONOSPACED, font.getStyle(), font.getSize()));

			FontZoom.setMouseWheel(jTable);
			resultsTableColumnFit = new TableColumnFit(jTable);
			jTable.setModel(new ResultsTableModel());

			jTable.setDefaultRenderer(Object.class, new TableRenderer());
		}
		return jTable;
	}

	/**
	 * This method initializes jJToolBarBar
	 * 
	 * @return javax.swing.JToolBar
	 */
	private JToolBar getJJToolBarBar() {
		if (jJToolBarBar == null) {
			jJToolBarBar = new JToolBar();
			jJToolBarBar.add(getJToggleButtonExecution());
			jJToolBarBar.add(getJButtonContinue());
			jJToolBarBar.add(getJCheckBoxAutoCommit());
			jJToolBarBar.add(getJButtonCommit());
			jJToolBarBar.add(getJButtonRollback());
			jJToolBarBar.add(getJToggleButtonConnection());
			jJToolBarBar.add(getJComboBoxSentenceSeparator());
		}
		return jJToolBarBar;
	}

	/**
	 * This method initializes jToggleButtonExecution
	 * 
	 * @return javax.swing.JButton
	 */
	JToggleButton getJToggleButtonExecution() {
		if (jToggleButtonExecution == null) {
			jToggleButtonExecution = new JToggleButton(IconImage.EXECUTE.getValue(), false);
			jToggleButtonExecution.setText("Execute");
			jToggleButtonExecution.setEnabled(false);
			jToggleButtonExecution.addChangeListener(new ChangeListener() {
				// pressedicon selectedicon では、うまく制御できないので、自前で制御する
				@Override
				public void stateChanged(ChangeEvent e) {
					JToggleButton b = (JToggleButton) e.getSource();
					if (b.isSelected()) {
						b.setIcon(IconImage.STOP.getValue());
					} else {
						b.setIcon(IconImage.EXECUTE.getValue());
					}
				}
			});
		}
		return jToggleButtonExecution;
	}

	/**
	 * This method initializes jButtonContinue
	 * 
	 * @return javax.swing.JButton
	 */
	JButton getJButtonContinue() {
		if (jButtonContinue == null) {
			jButtonContinue = new JButton(IconImage.GO_DOWN.getValue());
			jButtonContinue.setEnabled(false);
			jButtonContinue.setText("Continue");
		}
		return jButtonContinue;
	}

	/**
	 * This method initializes jCheckBoxAutoCommit
	 * 
	 * @return javax.swing.JCheckBox
	 */
	JCheckBox getJCheckBoxAutoCommit() {
		if (jCheckBoxAutoCommit == null) {
			jCheckBoxAutoCommit = new JCheckBox();
			jCheckBoxAutoCommit.setText("AutoCommit");
			jCheckBoxAutoCommit.setEnabled(false);
			jCheckBoxAutoCommit.addItemListener(new ItemListener() {
				@Override
				public void itemStateChanged(ItemEvent e) {
					boolean flg = jCheckBoxAutoCommit.isSelected();
					getJButtonCommit().setEnabled(!flg);
					getJButtonRollback().setEnabled(!flg);
					Jdbcacsess2.logger.fine("autocomit:" + flg);

					if (getDataBaseConnection() == null) {
						return;
					}
					try {
						getDataBaseConnection().setAutoCommit(flg);
					} catch (Exception ex) {
						ShowDialog.errorMessage(ex);
					}
				}
			});
			jCheckBoxAutoCommit.setSelected(true);
		}
		return jCheckBoxAutoCommit;
	}

	/**
	 * This method initializes jButtonCommit
	 * 
	 * @return javax.swing.JButton
	 */
	JButton getJButtonCommit() {
		if (jButtonCommit == null) {
			jButtonCommit = new JButton(IconImage.COMMIT.getValue());
			jButtonCommit.setText("Commit");
			jButtonCommit.setEnabled(false);
		}
		return jButtonCommit;
	}

	/**
	 * This method initializes jButtonRollback
	 * 
	 * @return javax.swing.JButton
	 */
	JButton getJButtonRollback() {
		if (jButtonRollback == null) {
			jButtonRollback = new JButton(IconImage.ROLLBACK.getValue());
			jButtonRollback.setText("Rollback");
			jButtonRollback.setEnabled(false);
		}
		return jButtonRollback;
	}

	/**
	 * This method initializes jScrollPaneLog
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private JScrollPane getJScrollPaneLog() {
		if (jScrollPaneLog == null) {
			jScrollPaneLog = new JScrollPane();
			jScrollPaneLog.setViewportView(getJTextAreaLog());
		}
		return jScrollPaneLog;
	}

	/**
	 * This method initializes jTextAreaLog
	 * 
	 * @return javax.swing.JTextArea
	 */
	JTextArea getJTextAreaLog() {
		if (jTextAreaLog == null) {
			jTextAreaLog = new JTextArea();
			jTextAreaLog.setEditable(false);
			jTextAreaLog.setLineWrap(false);
			jTextAreaLog.setOpaque(false);
		}
		return jTextAreaLog;
	}

	/**
	 * This method initializes jToggleButtonConnection
	 * 
	 * @return javax.swing.JToggleButton
	 */
	JToggleButton getJToggleButtonConnection() {
		if (jToggleButtonConnection == null) {
			jToggleButtonConnection = new JToggleButton(IconImage.DISCONNECT.getValue());
			jToggleButtonConnection.setText("session");
			jToggleButtonConnection.addChangeListener(new ChangeListener() {
				// pressedicon selectedicon では、うまく制御できないので、自前で制御する
				@Override
				public void stateChanged(ChangeEvent e) {
					JToggleButton b = (JToggleButton) e.getSource();
					if (b.isSelected()) {
						b.setIcon(IconImage.CONNECT.getValue());
					} else {
						b.setIcon(IconImage.DISCONNECT.getValue());
					}
				}
			});
		}
		return jToggleButtonConnection;
	}

	/**
	 * This method initializes jScrollPane1
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private JScrollPane getJScrollPane1() {
		if (jScrollPane1 == null) {
			jScrollPane1 = new JScrollPane();
			jScrollPane1.setViewportView(getJTableResultSet());
		}
		return jScrollPane1;
	}

	/**
	 * This method initializes jTableResultSet
	 * 
	 * @return javax.swing.JTable
	 */
	private JTable getJTableResultSet() {
		if (jTableResultSet == null) {
			jTableResultSet = new JmyTable();
		}
		return jTableResultSet;
	}

	/**
	 * This method initializes jComboBoxSentenceSeparator
	 * 
	 * @return javax.swing.JComboBox
	 */
	JComboBox<SentenceSeparator> getJComboBoxSentenceSeparator() {
		if (jComboBoxSentenceSeparator == null) {
			jComboBoxSentenceSeparator = new JComboBox<SentenceSeparator>();
			jComboBoxSentenceSeparator.setEditable(true);
			jComboBoxSentenceSeparator.addItem(new SentenceSeparator(";", ";\\s*\\n?"));
			jComboBoxSentenceSeparator.addItem(new SentenceSeparator("/", "\\n+\\s*/\\s*\\n?"));
			jComboBoxSentenceSeparator.addItem(new SentenceSeparator("//", "\\n+?\\s*//\\s*\\n?"));
			jComboBoxSentenceSeparator.addItem(new SentenceSeparator("GO", "\\n+\\s*[Gg][Oo]\\s*\\n?"));
		}
		return jComboBoxSentenceSeparator;
	}

	public static class SentenceSeparator {
		private final String guiVal;
		private final String regVal;

		SentenceSeparator(String guiVal, String regVal) {
			this.guiVal = guiVal;
			this.regVal = regVal;
		}

		@Override
		public String toString() {
			return guiVal;
		}

		/**
		 * @return regVal
		 */
		public String getRegVal() {
			return regVal;
		}
	}
}
