/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.parse;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * SQL文を字句要素に分解し、リスナーを通して要素種類毎に通知します。 字句要素は、下記６種類。
 * <ul>
 * <li>comment ハイフン開始コメント及び スラッシュアスタリスク範囲領域のコメント
 * <li>constant アスタリスク範囲領域の文字列定数
 * <li>symbol 記号
 * <li>delimiter 区切り文字
 * <li>input 入力パラメータ
 * <li>phrase その他を全てSQL句と判定
 * </ul>
 * 複数のSQL文を単一のSQLに分解するために、文区切りの正規表現にマッチした時に、リスナーを通じて分区切りを通知します。
 * その時のリスナーへの通知順は下記のようになります。
 * <ol>
 * <li>{@link SqlSentenceListener#started(String)}
 * <li>{@link SqlSentenceListener#finished(String, String)}
 * <li>
 * {@link SqlSentenceSparateListener#rangeSeparate(String, int, int, Pattern)}
 * <li>{@link SqlSentenceListener#started(String)}
 * <li>{@link SqlSentenceListener#finished(String, String)}
 * </ol>
 */
public class SqlSentenceParse {

	private String sqlSentence;

	private Pattern separatePattern;
	private SqlSentenceSparateListener sqlSentenceSparateListener;
	private final List<SqlSentenceListener> listenerList = new ArrayList<SqlSentenceListener>();

	/**
	 * 複数SQLを単一SQL文に分解するときの正規表現を登録する
	 * 
	 * @param pattern
	 */
	public void registSeparatePattern(Pattern pattern) {
		if (pattern == null) {
			throw new NullPointerException();
		}
		separatePattern = pattern;
	}

	/**
	 * SQL文の区切りを通知するリスナーを登録する
	 * 
	 * @param sqlSentenceSparateListener
	 */
	public void registSeparateListener(SqlSentenceSparateListener sqlSentenceSparateListener) {
		this.sqlSentenceSparateListener = sqlSentenceSparateListener;
	}
	/**
	 * 字句要素解析結果通知リスナーを登録する
	 * 
	 * @param listener
	 */
	public void addSqlSentenceListener(SqlSentenceListener listener) {
		if (listener == null) {
			throw new NullPointerException();
		}
		listenerList.add(listener);
	}

	/**
	 * 字句要素解析結果通知リスナーを削除する
	 */
	public void removeSqlSentenceListener(SqlSentenceListener listener) {
		if (listener == null) {
			throw new NullPointerException();
		}
		listenerList.remove(listener);
	}

	/**
	 * 字句要素解析結果通知リスナーを全削除する
	 */
	public void removeAllSqlSentenceListener() {
		listenerList.clear();
	}

	/**
	 * SQL文を解析し、字句要素に応じて、リスナーのメソッドを呼び出す。
	 * 
	 * @param sqlSentence
	 */
	public void parse(String sqlSentence) {
		this.sqlSentence = sqlSentence;

		int sentenceStartPosition = 0;

		State flg = State.PHRASE;

		// 解析開始の通知
		for (SqlSentenceListener l : listenerList)
			l.started(sqlSentence);

		// 字句要素範囲を一時保管するオブジェクトの初期化
		RangePhrase rangePhrase = new RangePhrase();
		RangeComment rangeComment = new RangeComment();
		RangeConstant rangeConstant = new RangeConstant();
		RangeDelimiter rangeDelimiter = new RangeDelimiter();
		RangeSymbol rangeSymbol = new RangeSymbol();

		// SQL文を１文字ずつ解析する
		for (int i = 0; i < sqlSentence.length(); i++) {

			// コメントは、２文字を同時に判断する必要がある
			char[] dst = new char[2];
			if (i == sqlSentence.length() - 1) {
				sqlSentence.getChars(i, i + 1, dst, 0);
				dst[1] = 0xff;
			} else {
				sqlSentence.getChars(i, i + 2, dst, 0);
			}

			// 読み込み文字を判断し、現状態からどの状態に遷移すればよいか決定する。
			flg = flg.next(dst);

			// デリミタとSQL句は、終了文字が規定できないので、コメントや定数が始まったら終了とする。
			// コメントや定数は、明確な終了記号が存在すので、開始・途中・終了のそれぞれを状態として管理する。

			if (flg == State.HIFUNCOMMENT_START || flg == State.SLASHCOMMENT_START) {
				rangePhrase.end(i - 1);
				rangeDelimiter.end(i - 1);
				rangeComment.begin(i);

			} else if (flg == State.COMMENT_END) {
				rangeComment.end(i);

			} else if (flg == State.APOST_START) {
				rangePhrase.end(i - 1);
				rangeDelimiter.end(i - 1);
				rangeConstant.begin(i);

			} else if (flg == State.APOST_END) {
				rangeConstant.end(i);

			} else if (flg == State.DELIMITER) {
				rangePhrase.end(i - 1);
				rangeDelimiter.begin(i);

			} else if (flg == State.PHRASE) {
				rangeDelimiter.end(i - 1);
				rangePhrase.begin(i);

			} else if (flg == State.SYMBOL_CHAR) {
				rangeDelimiter.end(i - 1);
				rangePhrase.end(i - 1);
				rangeSymbol.begin(i);
				rangeSymbol.end(i);

			}

			if (sqlSentenceSparateListener != null) {
				if (flg == State.PHRASE || flg == State.SYMBOL_CHAR || flg == State.DELIMITER) {
					int beginIndex = i;
					int endIndex = i + 10;
					if (endIndex > sqlSentence.length()) {
						endIndex = sqlSentence.length();
					}
					Matcher m = separatePattern.matcher(sqlSentence.substring(beginIndex, endIndex));
					if (m.lookingAt()) {
						// 解析終了通知
						for (SqlSentenceListener l : listenerList)
							l.finished(sqlSentence.substring(sentenceStartPosition, i), rangePhrase.getFirstSqlWord());

						String matchKeyword = m.group();
						sqlSentenceSparateListener.rangeSeparate(matchKeyword,
						                                         i + m.start(),
						                                         i + m.end(),
						                                         separatePattern);

						rangePhrase.setFirstSqlWord("");
						i = i + m.end() - 1;
						sentenceStartPosition = i;
						// 解析開始の通知
						for (SqlSentenceListener l : listenerList)
							l.started(sqlSentence.substring(i));
					}
				}
			}

		}

		// 中途半端な状態であったらその状態で終了させる。
		// 但し、どれか一つしかその対象にならない。
		// rangeSymbolは1文字だけであり、必ず閉じているので必要ない
		rangePhrase.finish();
		rangeComment.finish();
		rangeConstant.finish();
		rangeDelimiter.finish();

		// 解析終了通知
		for (SqlSentenceListener l : listenerList)
			l.finished(sqlSentence.substring(sentenceStartPosition), rangePhrase.getFirstSqlWord());
	}

	/**
	 * 記号文字の集合
	 */
	static final char[] SYMBOLS = { ',', '(', ')', '!', '+', '-', '*', '/', '=', '<', '>', ';' };

	/**
	 * 現時点状態に対し、読み込み文字が次状態へ遷移すべきかを決定する。
	 * 例えば、SQL句状態で、コメント開始文字(--や/*)を受け取ったら、コメント開始状態へ遷移する。
	 */
	enum State {
		PHRASE {
			@Override
			public State next(char[] dst) {
				if (dst[0] == '/' && dst[1] == '*')
					return SLASHCOMMENT_START;
				if (dst[0] == '-' && dst[1] == '-')
					return HIFUNCOMMENT_START;
				if (dst[0] == '\'')
					return APOST_START;
				if (dst[0] == ' ' || dst[0] == '\t' || dst[0] == '\n')
					return DELIMITER;
				for (int i = 0; i < SYMBOLS.length; i++) {
					if (dst[0] == SYMBOLS[i])
						return SYMBOL_CHAR;
				}
				return this;
			}
		},
		SYMBOL_CHAR {
			@Override
			public State next(char[] dst) {
				return PHRASE.next(dst);
			}
		},

		// APOST state
		APOST_START {
			@Override
			public State next(char[] dst) {
				if ((dst[0] == '\'' || dst[0] == '\\') && dst[1] == '\'')
					return APOST_ESCAPE;
				if ((dst[0] == '\'' || dst[0] == '\\') && dst[1] != '\'')
					return APOST_END;
				return APOST;
			}
		},
		APOST {
			@Override
			public State next(char[] dst) {
				if ((dst[0] == '\'' || dst[0] == '\\') && dst[1] == '\'')
					return APOST_ESCAPE;
				if ((dst[0] == '\'' || dst[0] == '\\') && dst[1] != '\'')
					return APOST_END;
				return this;
			}
		},
		APOST_ESCAPE {
			@Override
			public State next(char[] dst) {
				return APOST;
			}
		},
		APOST_END {
			@Override
			public State next(char[] dst) {
				return PHRASE.next(dst);
			}
		},

		// SLASH COMMENT state
		SLASHCOMMENT_START {
			@Override
			public State next(char[] dst) {
				return SLASHCOMMENT;
			}
		},
		SLASHCOMMENT {
			@Override
			public State next(char[] dst) {
				if (dst[0] == '*' && dst[1] == '/')
					return SLASHCOMMENT_PREEND;
				return this;
			}
		},
		SLASHCOMMENT_PREEND {
			@Override
			public State next(char[] dst) {
				return COMMENT_END;
			}
		},

		// HIFUN COMMENT state
		HIFUNCOMMENT_START {
			@Override
			public State next(char[] dst) {
				return HIFUNCOMMENT;
			}
		},
		HIFUNCOMMENT {
			@Override
			public State next(char[] dst) {
				if (dst[0] == '\n')
					return COMMENT_END;
				return this;
			}
		},

		// COMMENT state end (SLASH and HIFUN)
		COMMENT_END {
			@Override
			public State next(char[] dst) {
				return PHRASE.next(dst);
			}
		},

		// DELIMITER state
		DELIMITER {
			@Override
			public State next(char[] dst) {
				return PHRASE.next(dst);
			}
		},

		;

		public abstract State next(char[] dst);
	}

	/**
	 * 各字句要素の開始位置と終了位置を管理する
	 * 
	 * @author sima
	 * 
	 */
	abstract class Range {
		/**
		 * 範囲指定が有効状態であるかを示す
		 */
		boolean valid = false;

		int beginPosition = Integer.MIN_VALUE;
		int endPosition = Integer.MAX_VALUE;

		private String firstSqlWord = "";

		/**
		 * 開始位置を設定し有効状態にする。但し、既に有効の場合は、開始位置を設定せずに復帰する。
		 * 
		 * @param beginPosition
		 */
		void begin(int beginPosition) {
			if (valid) {
				return;
			}
			this.beginPosition = beginPosition;
			valid = true;
		}

		/**
		 * 開始位置が設定されており、終了が未設定の場合、残りの領域を全て強制終了する。
		 * 
		 * @param sqlSentence
		 * @param listeners
		 */
		void finish() {
			if (!valid) {
				return;
			}
			end(sqlSentence.length() - 1);
		}

		/**
		 * 終了位置を設定し無効状態にする。登録されているリスナーを呼び出す為に各実装クラスのメソッドを呼び出す。
		 * 但し、無効状態の場合は、何もせずに復帰する。
		 * 
		 * @param sqlSentence
		 * @param listeners
		 * @param endPosition
		 */
		void end(int endPosition) {
			if (!valid) {
				return;
			}

			this.endPosition = endPosition;
			String word = sqlSentence.substring(beginPosition, endPosition + 1);

			for (SqlSentenceListener l : listenerList) {
				fire(l, word);
			}
			valid = false;
			if (getFirstSqlWord().equals("")) {
				setFirstSqlWord(word);
			}
		}

		/**
		 * リスナーを呼びだす。継承先クラスで呼出処理は実装する。
		 * 
		 * @param l
		 *            リスナー
		 * @param s
		 *            通知文字列
		 */
		abstract void fire(SqlSentenceListener l, String sqlWord);

		@Override
		public String toString() {
			return beginPosition + ":" + endPosition;
		}

		/**
		 * @param firstSqlWord
		 *            セットする firstSqlWord
		 */
		void setFirstSqlWord(String firstSqlWord) {
			this.firstSqlWord = firstSqlWord;
		}

		/**
		 * @return firstSqlWord
		 */
		String getFirstSqlWord() {
			return firstSqlWord.toUpperCase();
		}
	}

	class RangeComment extends Range {
		@Override
		void fire(SqlSentenceListener l, String sqlWord) {
			l.rangeComment(sqlWord, beginPosition, endPosition);
		}
	}

	class RangePhrase extends Range {
		@Override
		void fire(SqlSentenceListener l, String sqlWord) {
			if (sqlWord.startsWith(":") || sqlWord.equals("?")) {
				l.rangeInput(sqlWord, beginPosition, endPosition);
			} else {
				l.rangePhrase(sqlWord, beginPosition, endPosition);
			}
		}
	}

	class RangeConstant extends Range {
		@Override
		void fire(SqlSentenceListener l, String sqlWord) {
			l.rangeConstant(sqlWord, beginPosition, endPosition);
		}
	}

	class RangeDelimiter extends Range {
		@Override
		void fire(SqlSentenceListener l, String sqlWord) {
			l.rangeDelimiter(sqlWord, beginPosition, endPosition);
		}
	}

	class RangeSymbol extends Range {
		@Override
		void fire(SqlSentenceListener l, String sqlWord) {
			l.rangeSymbol(sqlWord, beginPosition, endPosition);
		}
	}

}
